﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/demo.h>
#include <nw/snd.h>
#include "common.fsid"

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#include <cafe/axfx_presets.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#else
#include <winext/cafe.h>
#include <winext/cafe/axfx_presets.h>
#include <Windows.h>
#endif

namespace
{

#if defined(NW_USE_NINTENDO_SDK)
const char MOUNT_NAME[] = "content";
const char SOUND_ARC_PATH[] = "content:snd/common/common.bfsar";
#else
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
#endif
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

enum EffectType
{
    EFFECT_TYPE_NONE,
    EFFECT_TYPE_REVERB_STD,
    EFFECT_TYPE_REVERB_HI,
    EFFECT_TYPE_DELAY,
    EFFECT_TYPE_CHORUS,
    EFFECT_TYPE_REVERB_STD_DRC,
    EFFECT_TYPE_REVERB_HI_DRC,
    EFFECT_TYPE_DELAY_DRC,
    EFFECT_TYPE_CHORUS_DRC,
    EFFECT_TYPE_NUM,
    EFFECT_TYPE_MAX = EFFECT_TYPE_CHORUS_DRC
};

nw::snd::FxReverbStd                 s_FxReverbStd;
nw::snd::FxReverbHi                  s_FxReverbHi;
nw::snd::FxDelay                     s_FxDelay;
nw::snd::FxChorus                    s_FxChorus;
nw::snd::FxReverbStd::ReverbStdParam s_FxReverbStdParam;
nw::snd::FxReverbHi::ReverbHiParam   s_FxReverbHiParam;
nw::snd::FxDelay::DelayParam         s_FxDelayParam;
nw::snd::FxChorus::ChorusParam       s_FxChorusParam;

void* s_pMemoryForFxReverbStd;
void* s_pMemoryForFxReverbHi;
void* s_pMemoryForFxDelay;
void* s_pMemoryForFxChorus;

#if defined(NW_USE_NINTENDO_SDK)
void* Allocate(size_t size)
{
    nw::demo::DefaultAllocator allocator;
    return allocator.Alloc(size);
}

void Deallocate(void* memory, size_t size)
{
    NW_UNUSED_VARIABLE(size);

    nw::demo::DefaultAllocator allocator;
    allocator.Free(memory);
}
#endif

void InitializeSound()
{
    nw::demo::DefaultAllocator allocator;

    // サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );
    }

    // サウンドアーカイブの初期化
    {
    #if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = allocator.Alloc(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    #else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    #endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
    }
    if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
    }
}

void InitializeEffect( int effectType )
{
    nw::demo::DefaultAllocator allocator;

    switch ( effectType )
    {
        case EFFECT_TYPE_REVERB_STD:
            {
                s_FxReverbStdParam.fusedGain = 0.4f;
                s_FxReverbStd.SetParam( s_FxReverbStdParam );
                size_t reverbBufferSize = s_FxReverbStd.GetRequiredMemSize();
                s_pMemoryForFxReverbStd = allocator.Alloc( reverbBufferSize );
                s_FxReverbStd.AssignWorkBuffer( s_pMemoryForFxReverbStd, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxReverbStd );
                NW_LOG( "REVREB_STD %d\n", result );
            }
            break;
        case EFFECT_TYPE_REVERB_HI:
            {
                s_FxReverbHiParam.fusedGain = 0.4f;
                s_FxReverbHi.SetParam( s_FxReverbHiParam );
                size_t reverbBufferSize = s_FxReverbHi.GetRequiredMemSize();
                s_pMemoryForFxReverbHi = allocator.Alloc( reverbBufferSize );
                s_FxReverbHi.AssignWorkBuffer( s_pMemoryForFxReverbHi, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxReverbHi );
                NW_LOG( "REVREB_HI %d\n", result );
            }
            break;
        case EFFECT_TYPE_DELAY:
            {
                s_FxDelayParam.delay = 200.f;
                s_FxDelay.SetParam( s_FxDelayParam );
                size_t reverbBufferSize = s_FxDelay.GetRequiredMemSize();
                s_pMemoryForFxDelay = allocator.Alloc( reverbBufferSize );
                s_FxDelay.AssignWorkBuffer( s_pMemoryForFxDelay, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxDelay );
                NW_LOG( "DELAY %d\n", result );
            }
            break;
        case EFFECT_TYPE_CHORUS:
            {
                s_FxChorus.SetParam( s_FxChorusParam );
                size_t reverbBufferSize = s_FxChorus.GetRequiredMemSize();
                s_pMemoryForFxChorus = allocator.Alloc( reverbBufferSize );
                s_FxChorus.AssignWorkBuffer( s_pMemoryForFxChorus, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxChorus );
                NW_LOG( "CHORUS %d\n", result );
            }
            break;
        case EFFECT_TYPE_REVERB_STD_DRC:
            {
                s_FxReverbStdParam.fusedGain = 0.4f;
                s_FxReverbStd.SetParam( s_FxReverbStdParam );
                size_t reverbBufferSize = s_FxReverbStd.GetRequiredMemSize();
                s_pMemoryForFxReverbStd = allocator.Alloc( reverbBufferSize );
                s_FxReverbStd.AssignWorkBuffer( s_pMemoryForFxReverbStd, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxReverbStd, nw::snd::OUTPUT_DEVICE_DRC );
                NW_LOG( "REVREB_STD (DRC)%d\n", result );
            }
            break;
        case EFFECT_TYPE_REVERB_HI_DRC:
            {
                s_FxReverbHiParam.fusedGain = 0.4f;
                s_FxReverbHi.SetParam( s_FxReverbHiParam );
                size_t reverbBufferSize = s_FxReverbHi.GetRequiredMemSize();
                s_pMemoryForFxReverbHi = allocator.Alloc( reverbBufferSize );
                s_FxReverbHi.AssignWorkBuffer( s_pMemoryForFxReverbHi, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxReverbHi, nw::snd::OUTPUT_DEVICE_DRC );
                NW_LOG( "REVREB_HI (DRC)%d\n", result );
            }
            break;
        case EFFECT_TYPE_DELAY_DRC:
            {
                s_FxDelayParam.delay = 200.f;
                s_FxDelay.SetParam( s_FxDelayParam );
                size_t reverbBufferSize = s_FxDelay.GetRequiredMemSize();
                s_pMemoryForFxDelay = allocator.Alloc( reverbBufferSize );
                s_FxDelay.AssignWorkBuffer( s_pMemoryForFxDelay, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxDelay, nw::snd::OUTPUT_DEVICE_DRC );
                NW_LOG( "DELAY (DRC)%d\n", result );
            }
            break;
        case EFFECT_TYPE_CHORUS_DRC:
            {
                s_FxChorus.SetParam( s_FxChorusParam );
                size_t reverbBufferSize = s_FxChorus.GetRequiredMemSize();
                s_pMemoryForFxChorus = allocator.Alloc( reverbBufferSize );
                s_FxChorus.AssignWorkBuffer( s_pMemoryForFxChorus, reverbBufferSize );
                bool result = nw::snd::SoundSystem::AppendEffect(
                        nw::snd::AUX_BUS_A, &s_FxChorus, nw::snd::OUTPUT_DEVICE_DRC );
                NW_LOG( "CHORUS (DRC)%d\n", result );
            }
            break;
        default:
            {
                NW_LOG( "NONE\n");
            }
            break;
    }
}

void FinalizeEffect( int effectType )
{
    nw::demo::DefaultAllocator allocator;

    switch ( effectType )
    {
        case EFFECT_TYPE_REVERB_STD:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
            s_FxReverbStd.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxReverbStd );
            break;
        case EFFECT_TYPE_REVERB_HI:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
            s_FxReverbHi.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxReverbHi );
            break;
        case EFFECT_TYPE_DELAY:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
            s_FxDelay.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxDelay );
            break;
        case EFFECT_TYPE_CHORUS:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
            s_FxChorus.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxChorus );
            break;
        case EFFECT_TYPE_REVERB_STD_DRC:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
            s_FxReverbStd.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxReverbStd );
            break;
        case EFFECT_TYPE_REVERB_HI_DRC:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
            s_FxReverbHi.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxReverbHi );
            break;
        case EFFECT_TYPE_DELAY_DRC:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
            s_FxDelay.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxDelay );
            break;
        case EFFECT_TYPE_CHORUS_DRC:
            nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
            s_FxChorus.ReleaseWorkBuffer();
            allocator.Free( s_pMemoryForFxChorus );
            break;
        default:
            break;
    }
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &allocator;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(allocator.Alloc(sizeof(FSClient)));
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));
        NW_ASSERT_NOT_NULL(s_pFsClient);

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(allocator.Alloc(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        allocator.Free(block);
    }
#elif defined( NW_USE_NINTENDO_SDK )
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);
    // CONTENT_DIRのマウント
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            nn::fs::MountHost(MOUNT_NAME, buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return -1;
        }
    }
#endif

    InitializeSound();

    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/effect\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]          Start SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[X]          Start WSD  (SE_YOSHI)\n");
    NW_LOG("[Y]          Start STRM (STRM_MARIOKART)\n");
    NW_LOG("[B]          Stop Sound\n");
    NW_LOG("[LEFT/RIGHT] Change Effect\n");
    NW_LOG("[HOME]       Exit Application\n");
    NW_LOG("----------------------------------------\n");

    int effectType, preEffectType;
    effectType = preEffectType = 0;

    InitializeEffect( effectType );

    while ( !pDemo->IsExiting() ) {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }

        // サウンドの再生
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            s_SoundHandle.SetFxSend( nw::snd::AUX_BUS_A, 0.5f );
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
            NW_LOG("[SEQ] SEQ_MARIOKART ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            s_SoundHandle.SetFxSend( nw::snd::AUX_BUS_A, 0.5f );
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
            NW_LOG("[WSD] SE_YOSHI ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            s_SoundHandle.SetFxSend( nw::snd::AUX_BUS_A, 0.5f );
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
            NW_LOG("[STRM] STRM_MARIOKART ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) ) {
            s_SoundHandle.Stop( 0 );
        }

        // エフェクトの適用
        bool isChangeEffect = false;
        if ( pad->IsTrig( nw::demo::Pad::MASK_LEFT ) ) {
            preEffectType = effectType;
            effectType -= 1;
            if ( effectType < 0 ) {
                effectType = EFFECT_TYPE_MAX;
            }
            isChangeEffect = true;
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_RIGHT ) ) {
            preEffectType = effectType;
            effectType += 1;
            if ( effectType > EFFECT_TYPE_MAX ) {
                effectType = 0;
            }
            isChangeEffect = true;
        }
        if ( isChangeEffect )
        {
            FinalizeEffect( preEffectType );
            InitializeEffect( effectType );
        }


        s_SoundArchivePlayer.Update();
    }

    FinalizeEffect( effectType );
    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    allocator.Free( s_pMemoryForSoundHeap );
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
    allocator.Free( s_pMemoryForSoundSystem );

#if defined(NW_PLATFORM_CAFE)
    allocator.Free( s_pMemoryForFsSoundArchive );
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    allocator.Free(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

#if defined ( NW_USE_NINTENDO_SDK )
    nn::fs::Unmount(MOUNT_NAME);
#endif

    return 0;
}
