﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/snd/edit/sndedit_SoundEdit.h>
#include <nw/demo.h>
#include "common.fsid"

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#include <cafe/hio.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

//#define CPU_RENDERING

namespace
{
#if defined(NW_USE_NINTENDO_SDK)
const char MOUNT_NAME[] = "content";
const char SOUND_ARC_PATH[] = "content:snd/common/common.bfsar";
#else
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
#endif
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

nw::demo::DefaultAllocator  s_Allocator;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;
nw::snd::SoundHandle        s_SoundHandleHold;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;
void* s_pMemoryForLabelData;

#ifdef NW_SND_CONFIG_ENABLE_DEV
nw::snd::edit::SoundEdit s_SoundEdit;
void* s_pMemoryForSoundEdit;
#endif // NW_SND_CONFIG_ENABLE_DEV

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
FSCmdBlock* s_pFsCmdBlock;
char s_FsHfioMountPath[FS_MAX_MOUNTPATH_SIZE];
void* s_pMemoryForFsSoundArchive;
#endif

#if defined(NW_USE_NINTENDO_SDK)
    void* Allocate(size_t size)
    {
        return s_Allocator.Alloc(size);
    }

    void Deallocate(void* memory, size_t size)
    {
        NW_UNUSED_VARIABLE(size);
        s_Allocator.Free(memory);
    }
#endif

// --------------------------------------------------------------------------------------
bool
InitializeFs()
{
#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(s_Allocator.Alloc(sizeof(FSClient)));
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));
        NW_ASSERT_NOT_NULL(s_pFsClient);

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);

        s_pFsCmdBlock = reinterpret_cast<FSCmdBlock*>(s_Allocator.Alloc(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(s_pFsCmdBlock);
        FSInitCmdBlock(s_pFsCmdBlock);
    }

    // ディレクトリ移動
    {
        FSStatus status = FSChangeDir(s_pFsClient, s_pFsCmdBlock, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);
    }

    // HostFileIO デバイスのマウント（sndedit で利用）
    {
        FSMountSource mountSrc;
        FSGetMountSource(s_pFsClient, s_pFsCmdBlock, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR);
        FSMount(s_pFsClient, s_pFsCmdBlock, &mountSrc, s_FsHfioMountPath, sizeof(s_FsHfioMountPath), FS_RET_NO_ERROR);
    }
#elif defined( NW_USE_NINTENDO_SDK )
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);

    // マウント
    {
        nn::fs::MountHostRoot();

        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            nn::fs::MountHost(MOUNT_NAME, buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return false;
        }
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return false;
        }
    }
#endif

    return true;
}

// --------------------------------------------------------------------------------------
void
FinalizeFs()
{
#if defined( NW_PLATFORM_CAFE )
    // HostFileIO デバイスのアンマウント
    FSUnmount(s_pFsClient, s_pFsCmdBlock, s_FsHfioMountPath, FS_RET_NO_ERROR);

    // FS 終了処理
    s_Allocator.Free(s_pFsCmdBlock);
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    s_Allocator.Free(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();
#elif defined ( NW_USE_NINTENDO_SDK )
    nn::fs::Unmount(MOUNT_NAME);
    nn::fs::UnmountHostRoot();
#endif
}

// --------------------------------------------------------------------------------------
void
InitializeNwSound()
{
    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
    s_pMemoryForSoundSystem = s_Allocator.Alloc( workMemSize, 4 );

    nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );

    // サウンドアーカイブの初期化
    {
    #if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = s_Allocator.Alloc(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    #else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    #endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = s_Allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = s_Allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = s_Allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = s_Allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = s_Allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    // ラベルデータのロード
    {
        size_t setupSize = s_SoundArchive.GetLabelStringDataSize();
        s_pMemoryForLabelData = s_Allocator.Alloc( setupSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );

        s_SoundArchive.LoadLabelStringData(s_pMemoryForLabelData, setupSize);
    }

    // データのロード
    if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
    }
    if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
    }

#ifdef NW_SND_CONFIG_ENABLE_DEV
    // サウンドエディットの初期化
    {
        nw::snd::edit::SoundEdit::Option option;

        u32 memorySize = s_SoundEdit.GetRequiredMemorySize(option);
        s_pMemoryForSoundEdit = s_Allocator.Alloc(memorySize);

        nw::snd::edit::SoundEdit::InitializeArg args;
        args.buffer             = s_pMemoryForSoundEdit;
        args.bufferSize         = memorySize;
        args.soundArchive       = &s_SoundArchive;
        args.soundArchivePlayer = &s_SoundArchivePlayer;
#if defined( NW_PLATFORM_CAFE )
        args.fsClient           = s_pFsClient;
        args.fsHfioMountPath    = s_FsHfioMountPath;
#endif

        s_SoundEdit.Initialize(args, option);
    }

    s_SoundEdit.Start();
#endif // NW_SND_CONFIG_ENABLE_DEV
}

// --------------------------------------------------------------------------------------
void
FinalizeNwSound()
{
#ifdef NW_SND_CONFIG_ENABLE_DEV
    s_SoundEdit.Stop();
    s_SoundEdit.Finalize();
    s_Allocator.Free( s_pMemoryForSoundEdit );
#endif // NW_SND_CONFIG_ENABLE_DEV

    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    s_Allocator.Free( s_pMemoryForLabelData );
    s_Allocator.Free( s_pMemoryForSoundHeap );
    s_Allocator.Free( s_pMemoryForSoundArchivePlayer );
    s_Allocator.Free( s_pMemoryForStreamBuffer );
    s_Allocator.Free( s_pMemoryForSoundDataManager );
    s_Allocator.Free( s_pMemoryForInfoBlock );
#if defined(NW_PLATFORM_CAFE)
    s_Allocator.Free( s_pMemoryForFsSoundArchive );
#endif
    s_Allocator.Free( s_pMemoryForSoundSystem );
}

// --------------------------------------------------------------------------------------
void
PrintUsage()
{
    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/edit\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]     StartSound SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[X]     StartSound WSD  (SE_YOSHI)\n");
    NW_LOG("[Y]     StartSound STRM (STRM_MARIOKART)\n");
    NW_LOG("[B]     Stop Sound\n");
    NW_LOG("[RIGHT] HoldSound SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[UP]    HoldSound WSD  (SE_YOSHI)\n");
    NW_LOG("[LEFT]  HoldSound STRM (STRM_MARIOKART)\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S]     Exit Application\n");
#endif
    NW_LOG("---------------------------------------\n");
}

} // namespace

// --------------------------------------------------------------------------------------
int
NwDemoMain(int argc, char **argv)
{
    // SDK関連の初期化
#if defined( NW_PLATFORM_CAFE )
    AXInit();

#ifdef CPU_RENDERING
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_PPC );
#else
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
#endif

    HIOInit();
#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    // デモシステムの初期化
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &s_Allocator;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( s_Allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

    if (!InitializeFs())
    {
        return -1;
    }
    InitializeNwSound();

    PrintUsage();

#ifdef NW_SND_CONFIG_ENABLE_DEV
    bool isConnected = false;
#endif // NW_SND_CONFIG_ENABLE_DEV
    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::dev::Pad::MASK_A ) ) {
            s_SoundHandle.Stop( 0 );
            bool isEdited = false;
            bool isBankEdited = false;

#ifdef NW_SND_CONFIG_ENABLE_DEV
            isEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(SEQ_MARIOKART)) == nw::snd::edit::CACHE_STATE_CACHED;
            isBankEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(BANK_BGM)) == nw::snd::edit::CACHE_STATE_CACHED;
#endif // NW_SND_CONFIG_ENABLE_DEV

            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART)%s ... (%d)\n", isEdited || isBankEdited ? " with edit!" : "", result);
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_X ) ) {
            s_SoundHandle.Stop( 0 );
            bool isEdited = false;

#ifdef NW_SND_CONFIG_ENABLE_DEV
            isEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(SE_YOSHI)) == nw::snd::edit::CACHE_STATE_CACHED;
#endif // NW_SND_CONFIG_ENABLE_DEV

            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] StartSound(SE_YOSHI)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_Y ) ) {
            s_SoundHandle.Stop( 0 );
            bool isEdited = false;

#ifdef NW_SND_CONFIG_ENABLE_DEV
            isEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(STRM_MARIOKART)) == nw::snd::edit::CACHE_STATE_CACHED;
#endif // NW_SND_CONFIG_ENABLE_DEV

            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }
        if ( pad->IsTrig( nw::dev::Pad::MASK_B ) ) {
            s_SoundHandle.Stop( 0 );
        }

        // HoldSound
        if ( pad->IsHold( nw::dev::Pad::MASK_RIGHT ) ) {
            bool isEdited = false;
            bool isBankEdited = false;

#ifdef NW_SND_CONFIG_ENABLE_DEV
            isEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(SEQ_MARIOKART)) == nw::snd::edit::CACHE_STATE_CACHED;
            isBankEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(BANK_BGM)) == nw::snd::edit::CACHE_STATE_CACHED;
#endif // NW_SND_CONFIG_ENABLE_DEV

            bool result = s_SoundArchivePlayer
                .HoldSound( &s_SoundHandleHold, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] HoldSound(SEQ_MARIOKART)%s ... (%d)\n", isEdited || isBankEdited ? " with edit!" : "", result);
        }
        if ( pad->IsHold( nw::dev::Pad::MASK_UP ) ) {
            bool isEdited = false;

#ifdef NW_SND_CONFIG_ENABLE_DEV
            isEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(SE_YOSHI)) == nw::snd::edit::CACHE_STATE_CACHED;
#endif // NW_SND_CONFIG_ENABLE_DEV

            bool result = s_SoundArchivePlayer
                .HoldSound( &s_SoundHandleHold, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] HoldSound(SE_YOSHI)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }
        if ( pad->IsHold( nw::dev::Pad::MASK_LEFT ) ) {
            bool isEdited = false;

#ifdef NW_SND_CONFIG_ENABLE_DEV
            isEdited = s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(STRM_MARIOKART)) == nw::snd::edit::CACHE_STATE_CACHED;
#endif // NW_SND_CONFIG_ENABLE_DEV

            bool result = s_SoundArchivePlayer
                .HoldSound( &s_SoundHandleHold, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] HoldSound(STRM_MARIOKART)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }

        // Exit
        if ( pad->IsTrig( nw::dev::Pad::MASK_START ) ) {
            break;
        }

        s_SoundArchivePlayer.Update();

#ifdef NW_SND_CONFIG_ENABLE_DEV
        s_SoundEdit.Update();

        if(isConnected != s_SoundEdit.IsConnected())
        {
            if(isConnected)
            {
                isConnected = false;
                NW_LOG("disconnected.\n");
            }
            else
            {
                isConnected = true;
                NW_LOG("connected.\n");
            }
        }
#endif // NW_SND_CONFIG_ENABLE_DEV
    }

    FinalizeNwSound();
    FinalizeFs();

    // デモシステムの終了処理
    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();

    // アロケータの終了処理
    s_Allocator.Free( pDemo );

    // SDK関連の終了処理
#if defined( NW_PLATFORM_CAFE )
    AXQuit();
#endif

    return 0;
}
