﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "profile";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;

    nw::snd::SoundProfile s_SoundProfile[32];
    nw::snd::ProfileReader s_ProfileReader;
    nw::ut::Tick s_PrevAxFrameProcessBegin(0);
    nw::ut::Tick s_PrevNwFrameProcessBegin(0);

    bool s_ReadProfile = false;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データロード
        if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
        }
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }

        // プロファイルリーダーを登録
        nw::snd::SoundSystem::RegisterProfileReader(s_ProfileReader);
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        // プロファイルリーダの登録を解除
        nw::snd::SoundSystem::UnregisterProfileReader(s_ProfileReader);

        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]     StartSound SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]     StartSound WSD  (SE_YOSHI)\n");
        NW_LOG("[Y]     StartSound STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]     Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]     Exit Application\n");
#endif
        NW_LOG("[L]     Start/Stop Profile\n");
        NW_LOG("[R]     Print Usage (this)\n");
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            return false;
        }


        // プロファイル情報の取得/表示
        if (s_ReadProfile) {
            int profileCount = s_ProfileReader.Read(s_SoundProfile, 32);
            for( int i=0 ;i < profileCount ;i++)
            {
                nw::snd::SoundProfile prof = s_SoundProfile[i];

                NW_LOG("%6d  %8lld  %9lld  %7lld  %6lld  %9lld  %10lld %2.4f\n",
                    prof.axNumVoices,
                    prof.axFrameProcess.GetSpan().GetMicroSeconds(),
                    prof.nwFrameProcess.GetSpan().GetMicroSeconds(),
                    prof.auxProcess.GetSpan().GetMicroSeconds(),
                    prof.nwVoiceParamUpdate.GetSpan().GetMicroSeconds(),
                    prof.GetAxFrameProcessInterval(s_PrevAxFrameProcessBegin).GetMicroSeconds(),
                    prof.GetNwFrameProcessInterval(s_PrevNwFrameProcessBegin).GetMicroSeconds(),
                    static_cast<f32>(prof.dspCycles) / static_cast<f32>(AXGetMaxDspCycles()) * 100.0f
                );

                s_PrevAxFrameProcessBegin = prof.axFrameProcess.begin;
                s_PrevNwFrameProcessBegin = prof.nwFrameProcess.begin;
            }
        }

        // Usage 表示
        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) )
        {
            s_ReadProfile = false;
            PrintUsage();
        }

        // プロファイル情報取得/停止の切り替え
        if ( pad->IsTrig( nw::demo::Pad::MASK_L ) )
        {
            s_ReadProfile = !s_ReadProfile;

            NW_LOG("---------------------------------------------------------------------------\n");
            NW_LOG("voices  axFr(us)  sndFr(us)  aux(us)  cb(us)  axInt(us)  sndInt(us)  dsp(%%)\n");
            NW_LOG("---------------------------------------------------------------------------\n");
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void ProfileDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
