﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "multiListener";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;
    nw::snd::SoundHandle s_SoundHandleHold;

    nw::snd::Sound3DManager     s_Sound3DManager;
    nw::snd::Sound3DActor       s_Sound3DActor;

    const int LISTENER_COUNT = 2;
    nw::snd::Sound3DListener    s_Sound3DListener[LISTENER_COUNT];

    const f32 CONTROL_PAD_STEP = 0.2f;
    nw::math::VEC3              s_ActorPos;

    int s_FilterType;

    void* s_pMemoryFor3dManager;

    void CalcListenerMatrix( nw::math::MTX34& mtx, const nw::math::VEC3& pos  )
    {
        const nw::math::VEC3 upVec( 0.0f, 1.0f, 0.0f );  // リスナーのUp方向ベクトル
        const f32 degree = 0.0f;    // リスナーの向き

        // リスナーの方向ベクトル
        const nw::math::VEC3 direction(
            -nw::math::SinDeg( degree ), 0.0f, -nw::math::CosDeg( degree )
        );
        nw::math::VEC3 target = pos + direction;

        // リスナー行列生成
        nw::math::MTX34LookAt( &mtx, &pos, &upVec, &target );
    }

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        nw::snd::SoundSystem::SetOutputMode( nw::snd::OUTPUT_MODE_SURROUND );

        // 3D サウンドマネージャーの初期化
        {
            size_t setupSize = s_Sound3DManager.GetRequiredMemSize( &s_SoundArchive );
            s_pMemoryFor3dManager = allocator.Alloc( setupSize, 4 );
            s_Sound3DManager.Initialize( &s_SoundArchive, s_pMemoryFor3dManager, setupSize );
            s_Sound3DManager.SetMaxPriorityReduction( 32 );
            s_Sound3DManager.SetSonicVelocity( 340.0f / 60 );
        }

        // 3D サウンドリスナーの初期化
        for ( int i = 0; i < LISTENER_COUNT; i++ )
        {
            s_Sound3DManager.AddListener( &s_Sound3DListener[i] );

            // リスナー 2 つを、x = -3.0f (TV 出力用)  および
            //                  x =  3.0f (DRC 出力用) の位置に配置する
            nw::math::MTX34 listenerMtx;
            nw::math::VEC3 listenerPos( 0.0f, 0.0f, -3.0f );
            if ( i == 0 ) { listenerPos.x = -3.0f; }
            else { listenerPos.x = 3.0f; }

            CalcListenerMatrix( listenerMtx, listenerPos );
            s_Sound3DListener[i].SetMatrix( listenerMtx );
            s_Sound3DListener[i].SetMaxVolumeDistance( 2.0f );
            s_Sound3DListener[i].SetUnitDistance( 5.0f );
            s_Sound3DListener[i].SetInteriorSize( 5.0f );

            // 出力先属性を設定
            if ( i == 0 )
            {
                s_Sound3DListener[i].SetOutputTypeFlag( nw::snd::Sound3DListener::OUTPUT_TYPE_TV );
            }
            else
            {
                s_Sound3DListener[i].SetOutputTypeFlag( nw::snd::Sound3DListener::OUTPUT_TYPE_DRC0 );
            }
        }

        // 3D サウンドアクターの初期化
        {
            s_Sound3DActor.Initialize( s_SoundArchivePlayer, s_Sound3DManager );
            s_ActorPos = nw::math::VEC3::Zero();
            s_Sound3DActor.SetPosition( s_ActorPos );
        }

        // データのロード
        if ( ! s_SoundDataManager.LoadData( SE_IDLE, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_IDLE) failed." );
        }

        // 3D 関連パラメータの初期化
        {
            s_FilterType = 0;
            s_Sound3DManager.SetBiquadFilterType( s_FilterType );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        s_Sound3DActor.Finalize();
        s_Sound3DManager.Finalize();
        allocator.Free( s_pMemoryFor3dManager );

        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    const char* GetFilterTypeString( int type )
    {
        const char* FILTER_TYPE_STRING[] =
        {
            "NONE",
            "BIQUAD_LPF",
            "BIQUAD_HPF",
            "BIQUAD_BPF_512",
            "BIQUAD_BPF_1024",
            "BIQUAD_BPF_2048"
        };
        if ( type > nw::snd::BIQUAD_FILTER_TYPE_BPF2048 || type < 0 )
        {
            return "INVALID";
        }
        return FILTER_TYPE_STRING[ type ];
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A] HoldSound (Filter OFF)\n");
        NW_LOG("[X] HoldSound (Filter ON)\n");
        NW_LOG("[B] Reset Actor Position\n");
        NW_LOG("[L] Change Filter Type\n");
        NW_LOG("[LEFT/RIGHT] Move Actor Position.x\n");
        NW_LOG("[UP/DOWN]    Move Actor Position.z\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]       Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]          Exit Application\n");
#endif
       NW_LOG("[R]          Print Usage (this)\n");
       NW_LOG("---------------------------------------\n");
    }

    void PrintPosition()
    {
        NW_LOG("x(% 2.2f) z(% 2.2f)\n", s_ActorPos.x, s_ActorPos.z );
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // ドップラー効果・フィルター設定なしのサウンドをホールド再生
        if ( pad->IsHold( nw::demo::Pad::MASK_A ) )
        {
            s_Sound3DActor.HoldSound( &s_SoundHandle, SE_IDLE ).IsSuccess();
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
        }

        // ドップラー効果・フィルター設定ありのサウンドをホールド再生
        // ただし、リスナーが複数あるので、ドップラー効果は無効
        if ( pad->IsHold( nw::demo::Pad::MASK_X ) )
        {
            s_Sound3DActor.HoldSound( &s_SoundHandle, SE_IDLE_DOPPLER ).IsSuccess();
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
        }

        // 3D サウンドアクター座標をリセット
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_ActorPos.x = 0.0f;
            s_ActorPos.z = 0.0f;

            // 座標が飛ぶときは、ResetPosition 関数を呼び出す
            s_Sound3DActor.ResetPosition();
            PrintPosition();
        }

        // 3D サウンドアクターの座標変更
        if ( pad->IsHold( nw::demo::Pad::MASK_UP ) )
        {
            s_ActorPos.z += -CONTROL_PAD_STEP;
            PrintPosition();
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_DOWN ) )
        {
            s_ActorPos.z += CONTROL_PAD_STEP;
            PrintPosition();
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_LEFT ) )
        {
            s_ActorPos.x += -CONTROL_PAD_STEP;
            PrintPosition();
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_ActorPos.x += CONTROL_PAD_STEP;
            PrintPosition();
        }

        // フィルタタイプの変更
        if ( pad->IsTrig( nw::demo::Pad::MASK_L ) )
        {
            s_FilterType += 1;
            if ( s_FilterType > nw::snd::BIQUAD_FILTER_TYPE_BPF2048 )
            {
                s_FilterType = 0;
            }
            s_Sound3DManager.SetBiquadFilterType( s_FilterType );
            NW_LOG("FilterType(%s)\n",
                GetFilterTypeString( s_Sound3DManager.GetBiquadFilterType() ) );
        }

        // Usage 表示
        if ( pad->IsTrig( nw::demo::Pad::MASK_R ) )
        {
            PrintUsage();
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            return false;
        }

        s_Sound3DActor.SetPosition( s_ActorPos );
        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void MultiListenerDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
