﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>
#include <nw/snd.h>
#include <nw/snd/util/sndutil_FileReader.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

#if defined(NW_PLATFORM_CAFE)
#include <cafe.h>
#else
#include <winext/cafe.h>
#include "Shlwapi.h"
#endif

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "memorySoundArchive";
    const char EXTERNAL_FILE_ROOT[] = "snd/common/";

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::MemorySoundArchive s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHandle        s_SoundHandle;

    void* s_pMemoryForMemorySoundArchive;
    void* s_pMemoryForFsCommandBlockBufferPool;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);

        // メモリーサウンドアーカイブの初期化
        {
            // サウンドアーカイブファイルをメモリに展開
            nw::snd::util::FileReader fileReader;

            nw::snd::util::FileReader::Result result = fileReader.Initialize(
#if defined(NW_PLATFORM_CAFE)
                snddemo::GetFsClientPointer()
#endif
            );
            NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

            result = fileReader.Open(soundArchivePath);
            NW_ASSERTMSG( result == nw::snd::util::FileReader::RESULT_SUCCESS,
                "Failed to open file(%s) status(%d)\n", soundArchivePath, result );

            u32 size = fileReader.GetSize();
            s_pMemoryForMemorySoundArchive = allocator.Alloc(size, 64);
            s32 readSize = 0;
            fileReader.Read(s_pMemoryForMemorySoundArchive, size, &readSize);
            NW_ASSERT(size == static_cast<u32>(readSize));

            result = fileReader.Close();
            NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);
            result = fileReader.Finalize();
            NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

            // MemorySoundArchive インスタンスの初期化
            s_SoundArchive.Initialize( s_pMemoryForMemorySoundArchive );
        }

        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);

        // MemorySoundArchiveでストリームサウンドを使用するための処理です。
        // ストリームサウンドを使用しない場合はこの処理は不要です。
        {
            // 外部参照ファイルであるストリームサウンドバイナリを使用するために、
            // 外部参照ファイルのパスの基準となるディレクトリを設定します。
            // サウンドアーカイブバイナリ内部には相対パスでストリームサウンドバイナリのパスが埋め込まれているため、
            // 基本的にはサウンドアーカイブバイナリが存在しているディレクトリ
            // を設定することでストリームサウンドを再生できるようになります。
            char pathDir[64];
            snddemo::ConvertToPlatformDependentPath(EXTERNAL_FILE_ROOT, pathDir);

#ifdef NW_PLATFORM_CAFE
            // サウンドアーカイブがFSを使用できるようにするためにFS関連オブジェクトを設定します。
            size_t poolSize = s_SoundArchive.GetRequiredFsCommandBlockPoolMemSize();
            s_pMemoryForFsCommandBlockBufferPool = allocator.Alloc( poolSize );
            s_SoundArchive.CreateFsCommandBlockBufferPool(s_pMemoryForFsCommandBlockBufferPool, poolSize);
            s_SoundArchive.SetFsClient(snddemo::GetFsClientPointer());

            bool isFullPath = false;
            if ( pathDir[0] == '/' || pathDir[0] == '\\' )
            {
                isFullPath = true;
            }

            if ( isFullPath )
            {
                s_SoundArchive.SetExternalFileRoot( pathDir );
            }
            else
            {
                static const int FILE_PATH_MAX  = FS_MAX_FULLPATH_SIZE;
                char currentDir[FILE_PATH_MAX];

                // フルパスでカレントディレクトリを取得
                FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(allocator.Alloc(sizeof(FSCmdBlock)));
                NW_ASSERT_NOT_NULL(block);
                FSInitCmdBlock(block);

                FSStatus status = FSGetCwd(snddemo::GetFsClientPointer(), block, currentDir, FILE_PATH_MAX, FS_RET_NO_ERROR);
                NW_ASSERT(status == FS_STATUS_OK);
                allocator.Free(block);

                // currentDir <= currentDir + pathDir
                nw::ut::strncat( currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir) );
                s_SoundArchive.SetExternalFileRoot( currentDir );
            }
#elif defined( NW_USE_NINTENDO_SDK )
            s_SoundArchive.SetExternalFileRoot( pathDir );
#else
            if ( ! PathIsRelativeA(pathDir) )
            {
                s_SoundArchive.SetExternalFileRoot( pathDir );
            }
            else
            {
                static const int FILE_PATH_MAX = 256;
                char currentDir[FILE_PATH_MAX];

                // フルパスでカレントディレクトリを取得
                DWORD result = GetCurrentDirectoryA(FILE_PATH_MAX, currentDir);
                NW_ASSERT( result != 0 && result < FILE_PATH_MAX );
                if (currentDir[result - 1] != '\\')
                {
                    NW_ASSERT(result + 1 <= FILE_PATH_MAX);
                    currentDir[result + 1] = '\0';
                    currentDir[result] = '\\';
                }

                // currentDir <= currentDir + pathDir
                nw::ut::strncat( currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir) );
                s_SoundArchive.SetExternalFileRoot( currentDir );
            }
#endif
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
#if defined( NW_PLATFORM_CAFE )
        // 再生中に終了した場合には、SoundArchivePlayer::Finalize で使用されるため、
        // SoundArchivePlayer::Finalize() よりも後にで破棄処理を行います。
        s_SoundArchive.DestroyFsCommandBlockBufferPool();
#endif
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        s_SoundArchive.Finalize();
        snddemo::FinalizeSoundSystem(allocator);


#if defined( NW_PLATFORM_CAFE )
        allocator.Free( s_pMemoryForFsCommandBlockBufferPool );
#endif
        allocator.Free( s_pMemoryForMemorySoundArchive );
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]     Start SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]     Start WSD  (SE_YOSHI)\n");
        NW_LOG("[Y]     Start STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]     Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]     Exit Application\n");
#endif
        NW_LOG("----------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void MemorySoundArchiveDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();
        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
