﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>
#include <nw/snd/edit/sndedit_SoundEdit.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "edit";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;
    nw::snd::SoundHandle s_SoundHandleHold;

    void* s_pMemoryForStringBlock;

#ifdef NW_SND_CONFIG_ENABLE_DEV
    nw::snd::edit::SoundEdit s_SoundEdit;
    void* s_pMemoryForSoundEdit;
    bool s_IsConnected = false;
#endif // NW_SND_CONFIG_ENABLE_DEV

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データロード
        if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
        }
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void InitializeSndEdit(nw::ut::IAllocator& allocator)
    {
        // ラベル文字列のロード（sndedit で利用）
        size_t stringBlockSize = s_SoundArchive.GetLabelStringDataSize();
        s_pMemoryForStringBlock = allocator.Alloc( stringBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadLabelStringData( s_pMemoryForStringBlock, stringBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load stringBlock" );
        }

#ifdef NW_SND_CONFIG_ENABLE_DEV
        nw::snd::edit::SoundEdit::Option option;

        u32 memorySize = s_SoundEdit.GetRequiredMemorySize(option);
        s_pMemoryForSoundEdit = allocator.Alloc(memorySize);

        nw::snd::edit::SoundEdit::InitializeArg args;
        args.buffer             = s_pMemoryForSoundEdit;
        args.bufferSize         = memorySize;
        args.soundArchive       = &s_SoundArchive;
        args.soundArchivePlayer = &s_SoundArchivePlayer;
#if defined( NW_PLATFORM_CAFE )
        args.fsClient           = snddemo::GetFsClientPointer();
        args.fsHfioMountPath    = snddemo::GetHostFileIoMountPath();
#endif

        s_SoundEdit.Initialize(args, option);

        s_SoundEdit.Start();
#endif // NW_SND_CONFIG_ENABLE_DEV
    }

    void FinalizeSndEdit(nw::ut::IAllocator& allocator)
    {
#ifdef NW_SND_CONFIG_ENABLE_DEV
        s_SoundEdit.Stop();

        s_SoundEdit.Finalize();
        allocator.Free( s_pMemoryForSoundEdit );
#endif // NW_SND_CONFIG_ENABLE_DEV

        // ラベル文字列用メモリの解放
        allocator.Free(s_pMemoryForStringBlock);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]     StartSound SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]     StartSound WSD  (SE_YOSHI)\n");
        NW_LOG("[Y]     StartSound STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]     Stop Sound\n");
        NW_LOG("[RIGHT] HoldSound SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[UP]    HoldSound WSD  (SE_YOSHI)\n");
        NW_LOG("[LEFT]  HoldSound STRM (STRM_MARIOKART)\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]     Exit Application\n");
#endif
        NW_LOG("---------------------------------------\n");
    }

    bool IsEdited(nw::snd::SoundArchive::ItemId soundId)
    {
#ifdef NW_SND_CONFIG_ENABLE_DEV
        nw::snd::edit::CacheState cacheState =
            s_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
                s_SoundArchive.GetItemLabel(soundId));

        return cacheState == nw::snd::edit::CACHE_STATE_CACHED;
#else
        return false;
#endif
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool isEdited = IsEdited(SEQ_MARIOKART);
            bool isBankEdited = IsEdited(BANK_BGM);

            bool result = s_SoundArchivePlayer.StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART)%s ... (%d)\n", isEdited || isBankEdited ? " with edit!" : "", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool isEdited = IsEdited(SE_YOSHI);

            bool result = s_SoundArchivePlayer.StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] StartSound(SE_YOSHI)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool isEdited = IsEdited(STRM_MARIOKART);

            bool result = s_SoundArchivePlayer.StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // HoldSound
        if ( pad->IsHold( nw::demo::Pad::MASK_RIGHT ) )
        {
            bool isEdited = IsEdited(SEQ_MARIOKART);
            bool isBankEdited = IsEdited(BANK_BGM);

            bool result = s_SoundArchivePlayer.HoldSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] HoldSound(SEQ_MARIOKART)%s ... (%d)\n", isEdited || isBankEdited ? " with edit!" : "", result);
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_UP ) )
        {
            bool isEdited = IsEdited(SE_YOSHI);

            bool result = s_SoundArchivePlayer.HoldSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] HoldSound(SE_YOSHI)%s ... (%d)\n", isEdited ? " with edit!" : "", result);
        }
        if ( pad->IsHold( nw::demo::Pad::MASK_LEFT ) )
        {
            bool isEdited = IsEdited(STRM_MARIOKART);

            bool result = s_SoundArchivePlayer.HoldSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] HoldSound(STRM_MARIOKART)%s ... (%d)\n", isEdited ? " with edit!" : "", result);       }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            return false;
        }

        s_SoundArchivePlayer.Update();

#ifdef NW_SND_CONFIG_ENABLE_DEV
        s_SoundEdit.Update();

        if(s_IsConnected != s_SoundEdit.IsConnected())
        {
            if(s_IsConnected)
            {
                s_IsConnected = false;
                NW_LOG("disconnected.\n");
            }
            else
            {
                s_IsConnected = true;
                NW_LOG("connected.\n");
            }
        }
#endif // NW_SND_CONFIG_ENABLE_DEV

        return true;
    }
}

namespace snddemo
{

    void EditDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

#if defined( NW_PLATFORM_CAFE )
        // HostFileIO デバイスのマウント（sndedit で利用）
        snddemo::InitializeHostFileIo();
#endif

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        // SndEdit の初期化
        InitializeSndEdit(allocator);

        PrintUsage();

        s_IsConnected = false;

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // SndEdit の終了処理
        FinalizeSndEdit(allocator);

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();

#if defined( NW_PLATFORM_CAFE )
        // HostFileIO デバイスのアンマウント
        snddemo::FinalizeHostFileIo();
#endif
    }

}
