﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd.h>
#include <nw/demo.h>

#include "common/SampleUtility.h"
#if defined( NW_PLATFORM_CAFE )
#include <cafe/hio.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#elif defined( NW_PLATFORM_WIN32 )
#include <Windows.h>
#endif

namespace
{
#if defined( NW_USE_NINTENDO_SDK )
    const char MOUNT_NAME[] = "content";
#else
    const char CONTENT_DIR[] = "/vol/content";
#endif

#if defined(NW_USE_NINTENDO_SDK)
    void* Allocate(size_t size)
    {
        nw::demo::DefaultAllocator allocator;
        return allocator.Alloc(size);
    }

    void Deallocate(void* memory, size_t size)
    {
        NW_UNUSED_VARIABLE(size);

        nw::demo::DefaultAllocator allocator;
        allocator.Free(memory);
    }
#endif

}

namespace snddemo
{

#if defined(NW_PLATFORM_CAFE)
    static FSClient* s_pFsClient = NULL;
    static FSCmdBlock* s_pFsCmdBlock = NULL;
    static char s_FsHfioMountPath[FS_MAX_MOUNTPATH_SIZE];
#endif

    bool InitializeFs(nw::ut::IAllocator& allocator)
    {
#if defined(NW_PLATFORM_CAFE)
        // FS 初期化
        FSInit();

        // FS クライアントの初期化
        {
            s_pFsClient = reinterpret_cast<FSClient*>(allocator.Alloc(sizeof(FSClient)));
            std::memset(s_pFsClient, 0x00, sizeof(FSClient));
            NW_ASSERT_NOT_NULL(s_pFsClient);

            FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
        }

        // コマンドブロックの初期化
        {
            s_pFsCmdBlock = reinterpret_cast<FSCmdBlock*>(allocator.Alloc(sizeof(FSCmdBlock)));
            NW_ASSERT_NOT_NULL(s_pFsCmdBlock);
            FSInitCmdBlock(s_pFsCmdBlock);
        }

        // ディレクトリ移動
        {
            FSStatus status = FSChangeDir(s_pFsClient, s_pFsCmdBlock, CONTENT_DIR, FS_RET_NO_ERROR);
            NW_ASSERT(status == FS_STATUS_OK);
        }
#elif defined( NW_USE_NINTENDO_SDK )
        NW_UNUSED_VARIABLE(allocator);
        // nn::fs の初期化
        nn::fs::SetAllocator(Allocate, Deallocate);
        // CONTENT_DIRのマウント
        {
            static const int cFileNameLen = 512;
            char buf[cFileNameLen];
            if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
            {
                nn::fs::MountHost(MOUNT_NAME, buf);
            }
            else
            {
                NW_LOG("cannot read CAFE_CONTENT_DIR\n");
                return false;
            }
        }
#else
        NW_UNUSED_VARIABLE(allocator);
        // ディレクトリ移動
        {
            static const int32_t cFileNameLen = 512;
            char buf[cFileNameLen];
            if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
            {
                SetCurrentDirectoryA(buf);
            }
            else
            {
                NW_LOG("cannot read CAFE_CONTENT_DIR\n");
                return false;
            }
        }
#endif

        return true;
    }

    void FinalizeFs(nw::ut::IAllocator& allocator)
    {
#if defined( NW_PLATFORM_CAFE )
        allocator.Free(s_pFsCmdBlock);
        FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
        allocator.Free(s_pFsClient);
        s_pFsClient = NULL;
        FSShutdown();
#elif defined( NW_USE_NINTENDO_SDK )
        NW_UNUSED_VARIABLE(allocator);
        nn::fs::Unmount(MOUNT_NAME);
#else
        NW_UNUSED_VARIABLE(allocator);
#endif
    }

    void InitializeSdkSound(u32 rendererSelect)
    {
#if defined( NW_PLATFORM_CAFE )
        AXInit();
        AXSetDefaultRenderer(rendererSelect);
#else
        NW_UNUSED_VARIABLE(rendererSelect);
#endif
    }

    void FinalizeSdkSound()
    {
#if defined( NW_PLATFORM_CAFE )
        AXQuit();
#endif
    }

    void WaitForVBlank(nw::demo::DemoSystem* pDemo)
    {
#if defined( NW_PLATFORM_CAFE )
        NW_UNUSED_VARIABLE(pDemo);
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif
    }

    void ConvertToPlatformDependentPath(const char* path, char* platformDependentPath)
    {
#if defined( NW_USE_NINTENDO_SDK )
        std::sprintf(platformDependentPath, "%s:%s", MOUNT_NAME, path);
#else
        std::sprintf(platformDependentPath, "%s", path);
#endif
    }

#if defined( NW_PLATFORM_CAFE )
    void InitializeHostFileIo()
    {
        NW_ASSERT_NOT_NULL(s_pFsClient);
        NW_ASSERT_NOT_NULL(s_pFsCmdBlock);

        HIOInit();

        FSMountSource mountSrc;
        FSGetMountSource(s_pFsClient, s_pFsCmdBlock, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR);
        FSMount(s_pFsClient, s_pFsCmdBlock, &mountSrc, s_FsHfioMountPath, sizeof(s_FsHfioMountPath), FS_RET_NO_ERROR);
    }

    void FinalizeHostFileIo()
    {
        NW_ASSERT_NOT_NULL(s_pFsClient);
        NW_ASSERT_NOT_NULL(s_pFsCmdBlock);

        FSUnmount(s_pFsClient, s_pFsCmdBlock, s_FsHfioMountPath, FS_RET_NO_ERROR);
    }

    const char* GetHostFileIoMountPath()
    {
        return s_FsHfioMountPath;
    }

    FSClient* GetFsClientPointer()
    {
        NW_ASSERT_NOT_NULL(s_pFsClient);
        return s_pFsClient;
    }
#endif

}
