﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>
#if defined( NW_PLATFORM_CAFE )
#include <nw/sndaac.h>
#endif

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "aac";
    const s32 STREAM_READ_CACHE_SIZE = 16 * 1024 * 5;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;

    nw::snd::SoundHandle s_SoundHandle;

#if defined(NW_PLATFORM_CAFE)
    void* s_pMemoryForAacDec;
#endif

    void* s_pMemoryForSoundArchivePlayer;
    void* s_pMemoryForStreamBuffer;
    void* s_pMemoryForStreamCacheBuffer;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);

#if defined(NW_PLATFORM_CAFE)
        // AAC初期化
        {
            size_t aacDecBufferSize = nw::snd::aac::GetAacDecBufferSize(&s_SoundArchive);
            s_pMemoryForAacDec = allocator.Alloc( aacDecBufferSize, 64 );
            nw::snd::aac::Initialize( s_pMemoryForAacDec, aacDecBufferSize );
        }
#endif

        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);

        // サウンドアーカイブプレイヤーの初期化
        {
            size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
            s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
            size_t setupStrmBufferSize =
                s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
            s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );

            u32 setupStrmCacheSize =
                s_SoundArchivePlayer.GetRequiredStreamCacheSize( &s_SoundArchive, STREAM_READ_CACHE_SIZE );
            s_pMemoryForStreamCacheBuffer = allocator.Alloc(setupStrmCacheSize, 64);

            nw::snd::SoundArchivePlayer::InitializeParam param;
            param.soundArchive = &s_SoundArchive;
            param.soundDataManager = &s_SoundDataManager;
            param.setupBuffer = s_pMemoryForSoundArchivePlayer;
            param.setupBufferSize = setupSize;
            param.streamBuffer = s_pMemoryForStreamBuffer;
            param.streamBufferSize = setupStrmBufferSize;
            param.streamCacheBuffer = s_pMemoryForStreamCacheBuffer;
            param.streamCacheSize = setupStrmCacheSize;
            bool result = s_SoundArchivePlayer.Initialize(param);
            NW_ASSERT( result );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        s_SoundArchivePlayer.Finalize();
        allocator.Free( s_pMemoryForStreamCacheBuffer );
        allocator.Free( s_pMemoryForStreamBuffer );
        allocator.Free( s_pMemoryForSoundArchivePlayer );

        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
#if defined(NW_PLATFORM_CAFE)
        nw::snd::aac::Finalize();
        allocator.Free( s_pMemoryForAacDec );
#endif
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]     Play STRM_MARIOKART_AAC\n");
        NW_LOG("[Y]     Play STRM_MARIOKART_ADPCM\n");
        NW_LOG("[B]     Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]     Exit Application\n");
#endif
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART_AAC ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART_AAC) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART_ADPCM) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void AacDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
