﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#include <nw/lyt.h>
#include <nw/font.h>

//------------------------------------------------------------------------------
//   変数の宣言
//------------------------------------------------------------------------------

// アロケーター
const u32 c_HeapSize = 16 * 1024 * 1024;
nw::ut::MemoryAllocator s_Allocator;

// デモシステム
nw::demo::DemoSystem* s_Demo = NULL;

// グラフィックスコンテキスト
nw::gfnd::GraphicsContext s_GraphicsContext;

// レイアウト
nw::lyt::Layout* s_Layout_1 = NULL;
nw::lyt::Layout* s_Layout_2 = NULL;
u8* s_LayoutArchiveBinary = NULL;
// レイアウトで使用するフォント
nw::font::ResFont* s_Font = NULL;
u8* s_FontBinary = NULL;
// レイアウトアーカイブリソースアクセサ
nw::lyt::ArcResourceAccessor* s_ArcResourceAccessor = NULL;
// レイアウトグラフィックスリソース
nw::lyt::GraphicsResource s_GraphicsResource;
// 描画時に使用する情報
nw::lyt::DrawInfo s_DrawInfo;
#if defined(NW_PLATFORM_CAFE)
// レイアウトシェーダセットアップヘルパー(Cafeでのみ必要)
nw::lyt::ShaderSetupHelper s_ShaderSetupHelper;
#endif

#if defined(NW_PLATFORM_CAFE)
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002.bffnt";
#else
static const char* const DEMO_FONT_FILENAME = "nintendo_NTLG-DB_002_Nw4f.bffnt";
#endif

//------------------------------------------------------------------------------
//   レイアウトの処理
//------------------------------------------------------------------------------
/*
 *  レイアウトの初期化処理
 */
void
InitializeLayout()
{
    // レイアウトライブラリの初期化
    nw::lyt::Initialize(&s_Allocator);

    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

    // グラフィックスリソースの初期化
    {
        u8* pFontBinary = NULL;
        u8* pLayoutBinary = NULL;
#if defined(NW_PLATFORM_CAFE)
        // Cafeではシェーダバイナリを読み込んで初期化する必要がある
        nw::ut::MemoryRange fontShaderBinary;
        nw::ut::MemoryRange lytShaderBinary;
        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = "font_BuildinShader.gsh";
            loadArg.allocator = &s_Allocator;
            pFontBinary = fileSystem->Load( loadArg );
            fontShaderBinary = nw::ut::MakeMemoryRange(pFontBinary, loadArg.readSize);
        }
        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = "lyt_BuildinShader.gsh";
            loadArg.allocator = &s_Allocator;
            pLayoutBinary = fileSystem->Load( loadArg );
            lytShaderBinary = nw::ut::MakeMemoryRange(pLayoutBinary, loadArg.readSize);
        }
#endif
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
#if defined(NW_PLATFORM_CAFE)
            s_GraphicsResource.Setup(512, fontShaderBinary);
            s_ShaderSetupHelper.Setup(&s_GraphicsResource, lytShaderBinary);
#else
            s_GraphicsResource.Setup(512);
#endif
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
        nw::ut::SafeFree(pFontBinary, &s_Allocator);
        nw::ut::SafeFree(pLayoutBinary, &s_Allocator);
    }

    // レイアウトアーカイブの読み込みとリソースアクセサへのアタッチ
    // 複数のレイアウトから同じリソースを共有しています
    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "SharedResource.arc";
        loadArg.allocator = &s_Allocator;
        loadArg.alignment = nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT;
        s_LayoutArchiveBinary = fileSystem->Load( loadArg );

        s_ArcResourceAccessor = new( s_Allocator.Alloc( sizeof( nw::lyt::ArcResourceAccessor ) ) ) nw::lyt::ArcResourceAccessor();
        bool ret = s_ArcResourceAccessor->Attach(s_LayoutArchiveBinary, ".");
        NW_ASSERT(ret);
    }

    // フォントの読み込み
    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "sample.bffnt";
        loadArg.allocator = &s_Allocator;
        loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
        s_FontBinary = fileSystem->Load( loadArg );
    }

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    // フォントの初期化
    {
        s_Font = new( s_Allocator.Alloc( sizeof( nw::font::ResFont ) ) ) nw::font::ResFont();
        // リソースのセット。内部でテクスチャを初期化するため、コンテキストのロックが必要
        bool ret = s_Font->SetResource(s_FontBinary);
        NW_ASSERT(ret);

        // リソースアクセサにアタッチ
        s_ArcResourceAccessor->RegistFont("sample.bffnt", s_Font);
    }
    // レイアウトの初期化(リソースを共有しているため、同じリソースを渡します)
    {
        s_Layout_1 = new( s_Allocator.Alloc( sizeof( nw::lyt::Layout ) ) ) nw::lyt::Layout();
        s_Layout_1->BuildWithName("Layout_1.bflyt", s_ArcResourceAccessor);
        s_Layout_2 = new( s_Allocator.Alloc( sizeof( nw::lyt::Layout ) ) ) nw::lyt::Layout();
        s_Layout_2->BuildWithName("Layout_2.bflyt", s_ArcResourceAccessor);
    }
    // 描画に使用する情報の設定
    {
        nw::math::MTX44 projection;
        nw::ut::Rect layoutRect = s_Layout_1->GetLayoutRect();

        // 2つのレイアウトの描画サイズが等しいのでDrawInfoを共用させています
        // 一般的にはレイアウトの数だけDrawInfoを用意すべきです
        {
            nw::ut::Rect layoutRect2 = s_Layout_2->GetLayoutRect();
            NW_ASSERT(layoutRect.GetX() == layoutRect2.GetX());
            NW_ASSERT(layoutRect.GetY() == layoutRect2.GetY());
            NW_ASSERT(layoutRect.GetWidth() == layoutRect2.GetWidth());
            NW_ASSERT(layoutRect.GetHeight() == layoutRect2.GetHeight());
        }

        nw::math::MTX44Ortho(&projection, layoutRect.left, layoutRect.right, layoutRect.bottom, layoutRect.top, 0.0f, 300.0f);
        nw::math::MTX34 view;
        // zが1の位置からxy平面上のポリゴンを見る設定
        nw::math::VEC3 pos(0.f, 0.f, 1.f);
        nw::math::VEC3 up(0.f, 1.f, 0.f);
        nw::math::VEC3 target(0.f, 0.f, 0.f);
        nw::math::MTX34LookAt(&view, &pos, &up, &target);

        s_DrawInfo.SetGraphicsResource(&s_GraphicsResource);
        s_DrawInfo.SetProjectionMtx(projection);
        s_DrawInfo.SetViewMtx(view);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//------------------------------------------------------------------------------
/*
 *  レイアウトの更新処理
 */
void
UpdateLayout()
{
    // 描画時に使用するマトリックスを計算
    s_Layout_1->CalculateMtx(s_DrawInfo);
    s_Layout_2->CalculateMtx(s_DrawInfo);
}

//------------------------------------------------------------------------------
/*
 *  レイアウトの描画処理
 */
void
DrawLayout()
{
    // 描画設定
    // レイアウトは基本的にデプステストなし、カリングなしで描画する
    // これらの設定はレイアウト内では行っていないため、Drawを呼び出す前に設定しておく必要がある
    // 逆にアルファテスト及びブレンドの設定はレイアウト内で書き換えるため、注意が必要。
    s_GraphicsContext.SetDepthEnable(false, false);
    s_GraphicsContext.SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);
    s_GraphicsContext.Apply();

    // 複数のレイアウトを重ねて描画(後ろのレイアウトから順に呼び出します)
    s_Layout_1->Draw(s_DrawInfo);
    s_Layout_2->Draw(s_DrawInfo);
}

//------------------------------------------------------------------------------
/*
 *  レイアウトの終了処理
 */
void
FinalizeLayout()
{
    // レイアウトの破棄(この時点ではリソースは解放されません)
    nw::ut::SafeFreeWithDestruct( s_Layout_1, &s_Allocator );
    nw::ut::SafeFreeWithDestruct( s_Layout_2, &s_Allocator );

    // リソースアクセサの破棄
    nw::ut::SafeFreeWithDestruct( s_ArcResourceAccessor, &s_Allocator );
    nw::ut::SafeFree(s_LayoutArchiveBinary, &s_Allocator);

    // フォントの破棄
    nw::ut::SafeFreeWithDestruct( s_Font, &s_Allocator );
    nw::ut::SafeFree(s_FontBinary, &s_Allocator);

    // グラフィックスリソースの終了処理
    s_GraphicsResource.Finalize();

#if defined(NW_PLATFORM_CAFE)
    // シェーダの破棄
    s_ShaderSetupHelper.Finalize();
#endif
}

//------------------------------------------------------------------------------
//   デモフレームワークの処理
//------------------------------------------------------------------------------
/*
 *  1 フレーム分の処理
 */
void
ProcFrame()
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
    NW_NULL_ASSERT(graphics);

    // フレームの開始処理
    s_Demo->BeginFrame();

    // デモパッドの更新
    s_Demo->UpdatePad();

    // レイアウトの計算処理
    s_Demo->GetMeterCalc().BeginMeasure();
    UpdateLayout();
    s_Demo->GetMeterCalc().EndMeasure();

    graphics->LockDrawContext();
    {
        // TV 描画
        {
#if defined(NW_PLATFORM_CAFE)
            // GPUのreadキャッシュを全域クリアする
            GX2Invalidate(static_cast<GX2InvalidateType>(GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE | GX2_INVALIDATE_CONSTANT_BUFFER | GX2_INVALIDATE_SHADER),
                0x00000000, 0xffffffff );
#endif

            // フレームバッファのクリア
            s_Demo->ClearFrameBuffers();

            s_Demo->SetViewport( 0.f, 0.f, static_cast<f32>( s_Demo->GetWidth() ), static_cast<f32>( s_Demo->GetHeight() ), 0.f, 1.f );
            s_Demo->SetScissor( 0.f, 0.f, static_cast<f32>( s_Demo->GetWidth() ), static_cast<f32>( s_Demo->GetHeight() ) );

            // このデモのGPU負荷計測では、レイアウトの処理のみ計測する
            // 実際にはフレームバッファのクリア、スキャンバッファへの転送などの処理も必要だが、その処理は表示されない。
            s_Demo->GetMeterGPU().BeginMeasure();
            s_Demo->GetMeterDraw().BeginMeasure();
            {
                // レイアウトの描画処理
                DrawLayout();
            }
            s_Demo->GetMeterDraw().EndMeasure();
            s_Demo->GetMeterGPU().EndMeasure();

            // 負荷計測メーターを描画します。
            s_Demo->DrawLoadMeters();
        }
    }
    graphics->UnlockDrawContext();

    // バッファのスワップ
    s_Demo->SwapBuffer();

    // フレームの終了処理
    s_Demo->EndFrame();

    //  VBlank 待ち
    s_Demo->WaitForVBlank();
}

//------------------------------------------------------------------------------
/*
 *  main() 関数
 */
int
NwDemoMain(int argc, char **argv)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    //============== 初期化処理 ==============//
    // アロケーターの初期化
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    void* addr = malloc( c_HeapSize );
#else
    void* addr = MEMAllocFromDefaultHeap( c_HeapSize );
#endif
    NW_NULL_ASSERT(addr);
    s_Allocator.Initialize( addr, c_HeapSize );

    // デモシステムの作成
    nw::demo::DemoSystem::CreateArg arg;
    arg.allocator = &s_Allocator;
    arg.waitVBlank = 1;
    arg.width = 1280;
    arg.height = 720;
    arg.drawMeter = true;
    // デモ文字列描画用のフォントのパスを指定します。
    arg.fontPath = DEMO_FONT_FILENAME;
#if defined(NW_PLATFORM_CAFE)
    // デモ文字列描画用のシェーダーのパスを指定します。（ PC 版では不要です。 ）
    arg.fontShaderPath = "font_BuildinShader.gsh";
    // PrimitiveRenderer で用いるシェーダーのパスを指定します。（ PC 版では不要です。 ）
    arg.primitiveRendererShaderPath = "dev_PrimitiveRenderer.gsh";
#else
    arg.primitiveRendererInitialize = true;
#endif
    arg.fileDeviceManagerInitialize = true;

    s_Demo = new( s_Allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) ) nw::demo::DemoSystem( arg );
    NW_NULL_ASSERT(s_Demo);
    // デモシステムの初期化
    s_Demo->Initialize();

    // グラフィックシステムの初期化
    s_Demo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    // GL では座標を Cafe と同じ中心に設定する。
    nw::ut::DynamicCast<nw::demo::WPadWin*>( s_Demo->GetWPad() )->SetPointerCenterOrigin( true );
    nw::ut::DynamicCast<nw::demo::MouseWin*>( s_Demo->GetMouse() )->SetPointerCenterOrigin( true );
#endif

    // レイアウトの初期化処理
    InitializeLayout();

    //============== メインループ ==============//
    NW_LOG("Start demo.\n");
    // L,R,X同時押しで抜ける
    while ( ! s_Demo->GetPad()->IsHoldAll( nw::demo::Pad::MASK_L | nw::demo::Pad::MASK_R | nw::demo::Pad::MASK_X ) && ! s_Demo->IsExiting())
    {
        ProcFrame();
    }

    //============== 終了処理 ==============//
    // レイアウトの終了処理
    FinalizeLayout();

    // グラフィックシステムの終了処理
    s_Demo->FinalizeGraphicsSystem();
    // デモシステムの終了処理
    s_Demo->Finalize();
    nw::ut::SafeFree( s_Demo, &s_Allocator );

    // アロケータの破棄
    s_Allocator.Finalize();
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    free( addr );
#else
    MEMFreeToDefaultHeap( addr );
#endif

    NW_LOG("End demo.\n");
    return 0;
}
