﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3ddemo_GfxUtility.h>

#include <cstdarg>
#include <cctype>
#include <algorithm>
#include <nw/g3d/ut/g3d_Inlines.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/fnd/g3d_GLUtility.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <stdio.h>

namespace nw { namespace g3d { namespace demo {

//--------------------------------------------------------------------------------------------------
// Graphics

void ColorBufferTexture::Setup()
{
    texture.CalcSize();
    texture.Setup();
    nw::g3d::DebugPrint("(ColorBuffer) base size: %d, mip size: %d, alignment: %d\n",
        texture.GetBaseSize(), texture.GetMipSize(), texture.GetAlignment());

    renderBuffer.SetTexture(&texture);
    if (isFTV)
    {
        renderBuffer.UpdateRegsFTV();
    }
    else
    {
        renderBuffer.UpdateRegs();
    }
    renderBuffer.CalcAuxSize();
    nw::g3d::DebugPrint("(AuxBuffer) aux size: %d, alignment: %d\n",
        renderBuffer.GetAuxSize(), renderBuffer.GetAuxAlignment());
}

void ColorBufferTexture::Cleanup()
{
    texture.Cleanup();
}

void ColorBufferTexture::Alloc(FuncAlloc funcAlloc, bit32 bufferType /*= ALL_BUFFER*/)
{
    NW_G3D_ASSERT_NOT_NULL(funcAlloc);

    if (bufferType & COLOR_BUFFER)
    {
        size_t size = texture.GetBaseSize() + texture.GetMipSize();
        void* imagePtr = funcAlloc(size, texture.GetAlignment());
        NW_G3D_ASSERT_NOT_NULL(imagePtr);
        texture.SetImagePtrs(imagePtr, NULL);
        renderBuffer.SetImagePtrs(&texture);
    }
    if ((bufferType & AUX_BUFFER) && renderBuffer.GetAuxSize() > 0)
    {
        void* auxPtr = funcAlloc(renderBuffer.GetAuxSize(), renderBuffer.GetAuxAlignment());
        NW_G3D_ASSERT_NOT_NULL(auxPtr);
        renderBuffer.SetAuxPtr(auxPtr);
    }
}

void ColorBufferTexture::Free(FuncFree funcFree, bit32 bufferType /*= ALL_BUFFER*/)
{
    NW_G3D_ASSERT_NOT_NULL(funcFree);

    if ((bufferType & COLOR_BUFFER) && texture.GetBasePtr())
    {
        funcFree(texture.GetBasePtr());
    }
    if ((bufferType & AUX_BUFFER) && renderBuffer.GetAuxPtr())
    {
        funcFree(renderBuffer.GetAuxPtr());
    }
}

//--------------------------------------------------------------------------------------------------

void DepthBufferTexture::Setup()
{
    texture.CalcSize();
    texture.Setup();
    nw::g3d::DebugPrint("(DepthBuffer) base size: %d, mip size: %d, alignment: %d\n",
        texture.GetBaseSize(), texture.GetMipSize(), texture.GetAlignment());

    renderBuffer.SetTexture(&texture);
    renderBuffer.UpdateRegs();
    renderBuffer.SetClearDepthStencil(1.0f, 0);
    renderBuffer.CalcHiZSize();
    nw::g3d::DebugPrint("(HiZBuffer) hiz size: %d, alignment: %d\n",
        renderBuffer.GetHiZSize(), renderBuffer.GetHiZAlignment());
}

void DepthBufferTexture::Cleanup()
{
    texture.Cleanup();
}

void DepthBufferTexture::Alloc(FuncAlloc funcAlloc, bit32 bufferType /*= ALL_BUFFER*/)
{
    NW_G3D_ASSERT_NOT_NULL(funcAlloc);

    if (bufferType & DEPTH_BUFFER)
    {
        size_t size = texture.GetBaseSize() + texture.GetMipSize();
        void* imagePtr = funcAlloc(size, texture.GetAlignment());
        NW_G3D_ASSERT_NOT_NULL(imagePtr);
        texture.SetImagePtrs(imagePtr, NULL);
        renderBuffer.SetImagePtrs(&texture);
    }

    if ((bufferType & HIZ_BUFFER) && renderBuffer.GetHiZSize() > 0)
    {
        void* hiZPtr = funcAlloc(renderBuffer.GetHiZSize(), renderBuffer.GetHiZAlignment());
        NW_G3D_ASSERT_NOT_NULL(hiZPtr);
        renderBuffer.SetHiZPtr(hiZPtr);
        renderBuffer.SetHiZEnable(GX2_TRUE);
    }
}

void DepthBufferTexture::Free(FuncFree funcFree, bit32 bufferType /*= ALL_BUFFER*/)
{
    NW_G3D_ASSERT_NOT_NULL(funcFree);

    if (bufferType & DEPTH_BUFFER && texture.GetBasePtr())
    {
        funcFree(texture.GetBasePtr());
    }

    if ((bufferType & HIZ_BUFFER) && renderBuffer.GetHiZPtr())
    {
        funcFree(renderBuffer.GetHiZPtr());
    }
}

//--------------------------------------------------------------------------------------------------

size_t FrameBuffer::CalcSize(const InitArg& arg)
{
    NW_G3D_ASSERT(0 <= arg.colorBufferCount && arg.colorBufferCount <= MAX_RENDER_TARGET);
    NW_G3D_ASSERT(arg.width <= MAX_WIDTH);
    NW_G3D_ASSERT(arg.height <= MAX_HEIGHT);

    size_t size = sizeof(ColorBufferTexture) * arg.colorBufferCount;
    if (arg.useDepthBuffer)
    {
        size += sizeof(DepthBufferTexture);
    }
    return size;
}

bool FrameBuffer::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    if (bufferSize > CalcSize(arg))
    {
        return false;
    }

    m_Handle = 0;
    m_Width = arg.width;
    m_Height = arg.height;
    m_NumRenderTarget = arg.colorBufferCount;
    m_pColorBuffer = arg.colorBufferCount > 0 ? static_cast<ColorBufferTexture*>(pBuffer) : NULL;
    m_pDepthBuffer = arg.useDepthBuffer ? AddOffset<DepthBufferTexture>(
        pBuffer, sizeof(ColorBufferTexture) * arg.colorBufferCount) : NULL;

    for (int renderTarget = 0; renderTarget < m_NumRenderTarget; ++renderTarget)
    {
        ColorBufferTexture* pBuffer = new(&m_pColorBuffer[renderTarget]) ColorBufferTexture();
        pBuffer->isFTV = arg.colorBufferFTV;
        GX2Surface& surface = pBuffer->texture.GetGX2Texture()->surface;
        surface.width = m_Width;
        surface.height = m_Height;
        surface.format = (arg.colorBufferFormat != GX2_SURFACE_FORMAT_INVALID) ? arg.colorBufferFormat : surface.format;
    }
    if (m_pDepthBuffer)
    {
        DepthBufferTexture* pBuffer = new(m_pDepthBuffer) DepthBufferTexture();
        GX2Surface& surface = pBuffer->texture.GetGX2Texture()->surface;
        surface.width = m_Width;
        surface.height = m_Height;
        surface.format = GX2_SURFACE_FORMAT_TCD_R32_FLOAT;
        surface.use = GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE;
    }

    SetMipLevel(0);

    return true;
}

void FrameBuffer::Setup()
{
    if (m_Handle == 0)
    {
        glGenFramebuffers(1, &m_Handle);
        static const GLenum s_TblDrawBuffer[] = {
            GL_COLOR_ATTACHMENT0,
            GL_COLOR_ATTACHMENT1,
            GL_COLOR_ATTACHMENT2,
            GL_COLOR_ATTACHMENT3,
            GL_COLOR_ATTACHMENT4,
            GL_COLOR_ATTACHMENT5,
            GL_COLOR_ATTACHMENT6,
            GL_COLOR_ATTACHMENT7,
        };

        glBindFramebuffer(GL_FRAMEBUFFER, m_Handle);
        glDrawBuffers(m_NumRenderTarget, s_TblDrawBuffer);
        glBindFramebuffer(GL_FRAMEBUFFER, 0);
        NW_G3D_GL_ASSERT();
    }

    for (int renderTarget = 0; renderTarget < m_NumRenderTarget; ++renderTarget)
    {
        m_pColorBuffer[renderTarget].Setup();
    }
    if (m_pDepthBuffer)
    {
        m_pDepthBuffer->Setup();
    }
}

void FrameBuffer::Cleanup()
{
    for (int renderTarget = 0; renderTarget < m_NumRenderTarget; ++renderTarget)
    {
        m_pColorBuffer[renderTarget].Cleanup();
    }
    if (m_pDepthBuffer)
    {
        m_pDepthBuffer->Cleanup();
    }

    if (m_Handle != 0)
    {
        glDeleteFramebuffers(1, &m_Handle);
        m_Handle = 0;
        NW_G3D_GL_ASSERT();
    }
}

void FrameBuffer::Alloc(FuncAlloc funcAlloc, bit32 bufferType /*= ALL_BUFFER*/)
{
    NW_G3D_ASSERT_NOT_NULL(funcAlloc);

    int numRenderTarget = (bufferType & COLOR_AUX_BUFFER) ? m_NumRenderTarget : 0;
    for (int renderTarget = 0; renderTarget < numRenderTarget; ++renderTarget)
    {
        m_pColorBuffer[renderTarget].Alloc(funcAlloc, bufferType);
    }

    if ((bufferType & DEPTH_HIZ_BUFFER) && m_pDepthBuffer)
    {
        m_pDepthBuffer->Alloc(funcAlloc, bufferType);
    }
}

void FrameBuffer::Free(FuncFree funcFree, bit32 bufferType /*= ALL_BUFFER*/)
{
    NW_G3D_ASSERT_NOT_NULL(funcFree);

    int numRenderTarget = (bufferType & COLOR_AUX_BUFFER) ? m_NumRenderTarget : 0;
    for (int renderTarget = 0; renderTarget < numRenderTarget; ++renderTarget)
    {
        m_pColorBuffer[renderTarget].Free(funcFree, bufferType);
    }

    if ((bufferType & DEPTH_HIZ_BUFFER) && m_pDepthBuffer)
    {
        m_pDepthBuffer->Free(funcFree, bufferType);
    }
}

void FrameBuffer::SetMipLevel(u32 mipLevel)
{
    for (int renderTarget = 0; renderTarget < m_NumRenderTarget; ++renderTarget)
    {
        m_pColorBuffer[renderTarget].renderBuffer.GetGX2ColorBuffer()->viewMip = mipLevel;
    }
    if (m_pDepthBuffer)
    {
        m_pDepthBuffer->renderBuffer.GetGX2DepthBuffer()->viewMip = mipLevel;
    }
    u32 width = std::max(1u, m_Width >> mipLevel);
    u32 height = std::max(1u, m_Height >> mipLevel);

    m_Viewport.SetViewport(0, 0, static_cast<float>(width), static_cast<float>(height), 0.0f, 1.0f);
    m_Scissor.SetScissor(0, 0, width, height);
}

void FrameBuffer::Load()
{
    for (int renderTarget = 0; renderTarget < m_NumRenderTarget; ++renderTarget)
    {
        m_pColorBuffer[renderTarget].renderBuffer.Load(
            static_cast<GX2RenderTarget>(renderTarget), m_Handle);
    }
    if (m_pDepthBuffer)
    {
        m_pDepthBuffer->renderBuffer.Load(m_Handle);
    }

    m_Viewport.Load();
    m_Scissor.Load();

    glBindFramebuffer(GL_FRAMEBUFFER, m_Handle);
    //glEnable(GL_FRAMEBUFFER_SRGB);
    NW_G3D_GL_ASSERT();
}

void SimpleShader::Setup(const Path& path)
{
    nw::g3d::GfxShader::SetupArg arg;
    void* src[] = { NULL, NULL, NULL, NULL, NULL };
    if (path.vertexShader)
    {
        src[0] = LoadFile(path.vertexShader);
        arg.pVertexShader = static_cast<char*>(src[0]);
    }
    if (path.fragmentShader)
    {
        src[2] = LoadFile(path.fragmentShader);
        arg.pFragmentShader = static_cast<char*>(src[2]);
    }

    nw::g3d::GfxShader::Setup(arg);
    for (int i = 0; i < sizeof(src) / sizeof(void*); ++i)
    {
        if (src[i])
        {
            FreeMem2(src[i]);
        }
    }
}

void SimpleShader::Cleanup()
{
    nw::g3d::GfxShader::Cleanup();
}

}}} // namespace nw::g3d::demo
