﻿/*---------------------------------------------------------------------------*
  Project:  NintendoWare

  Copyright (C)Nintendo/HAL Laboratory, Inc.  All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *---------------------------------------------------------------------------*/

#include <common.glsl>

#define c_Pos_p         ( 0 )
#define c_Nrm_l         ( c_Pos_p + 1 )
#define c_Nrm_w         ( c_Nrm_l + 1 )
#define c_Nrm_v         ( c_Nrm_w + 1 )
#define c_Tan_l         ( c_Nrm_v + 1 )
#define c_Binrm_l       ( c_Tan_l + 1 )
#define c_TanSpace      ( c_Binrm_l + 1 )
#define c_SkinCnt       ( c_TanSpace + 1 )
#define c_Influence     ( c_SkinCnt + 1 )
#define c_Color         ( c_Influence + 1 )
#define c_UV            ( c_Color + 1 )
#define c_Albedo        ( c_UV + 1 )
#define c_NrmMap        ( c_Albedo + 1 )

#define OUTPUT c_Pos_p // @@ id="output" choice="c_Pos_p, c_Nrm_l, c_Nrm_w, c_Nrm_v, c_Tan_l, c_Binrm_l, c_TanSpace, c_SkinCnt, c_Influence, c_Color, c_UV, c_Albedo, c_NrmMap" type="dynamic"

//--------------------------------------------------------------------------------------------------

#if NW_VERTEX_SHADER

layout(location = 0)  in vec4   i_Position;     // @@ id="_p0"   hint="position0"
layout(location = 1)  in vec3   i_Normal;       // @@ id="_n0"   hint="normal0"
layout(location = 2)  in vec4   i_Tangent;      // @@ id="_t0"   hint="tangent0"
layout(location = 3)  in vec4   i_Binormal;     // @@ id="_b0"   hint="binormal0"
layout(location = 4)  in ivec4  i_Index0;       // @@ id="_i0"   hint="blendindex0"
layout(location = 5)  in ivec4  i_Index1;       // @@ id="_i1"   hint="blendindex1"
layout(location = 6)  in vec4   i_Weight0;      // @@ id="_w0"   hint="blendweight0"
layout(location = 7)  in vec4   i_Weight1;      // @@ id="_w1"   hint="blendweight1"
layout(location = 8)  in vec4   i_Color;        // @@ id="_c0"   hint="color0"
layout(location = 9)  in vec4   i_UV;           // @@ id="_u0"   hint="uv0"

#endif // NW_VERTEX_SHADER

#if !NW_GEOMETRY_SHADER

NW_RASTER
{
    vec4 pos_p;
    vec3 vec;
    vec4 color;
    vec2 uv;
} nw_raster;

#endif // !NW_GEOMETRY_SHADER

#if NW_FRAGMENT_SHADER

layout(location = 0)    out vec4 o_Color;

#endif // NW_FRAGMENT_SHADER

//--------------------------------------------------------------------------------------------------

uniform sampler2D s_AlbedoTex; // @@ id="_a0" hint="albedo0"
uniform sampler2D s_NormalTex; // @@ id="_n0" hint="normal0"

layout(std140) uniform Skeleton // @@ id="skel" type="skeleton"
{
    vec4 mtxPalette[3 * 128]; // RADEON では配列のサイズを超えてアクセスできない。
};

layout(std140) uniform Shape // @@ id="shape" type="shape"
{
    vec4 shapeMtx[3];
    int vtxSkinCount;
};

layout(std140) uniform View // @@ id="view"
{
    vec4 projMtx[4];
    vec4 cameraMtx[3];
};

//--------------------------------------------------------------------------------------------------

#if NW_VERTEX_SHADER

void main()
{
    vec4 pos_w = vec4(0, 0, 0, 1);
    vec3 nrm_w = vec3(0, 0, 0);

    int influence = 0;
    if (vtxSkinCount == 0)
    {
        // rigid body
        pos_w.xyz = NW_G3D_TRANSFORM_POS(shapeMtx, i_Position);
        nrm_w.xyz = NW_G3D_TRANSFORM_VEC(shapeMtx, i_Normal);
    }
    else if (vtxSkinCount == 1)
    {
        // rigid skinning
        int offset = i_Index0.x * 3;
        pos_w.xyz = NW_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
        nrm_w.xyz = NW_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
        influence = 1;
    }
    else if (vtxSkinCount <= 4)
    {
        // smooth skinning [1, 4]
        for (int i = 0; i < vtxSkinCount; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NW_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NW_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            if (weight > 0)
            {
                ++influence;
            }
        }
    }
    else if (vtxSkinCount <= 8)
    {
        // smooth skinning [5, 8]
        for (int i = 0; i < 4; ++i)
        {
            int offset = i_Index0[i] * 3;
            float weight = i_Weight0[i];
            pos_w.xyz += weight * NW_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NW_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            if (weight > 0)
            {
                ++influence;
            }
        }
        for (int i = 4; i < vtxSkinCount; ++i)
        {
            int offset = i_Index1[i - 4] * 3;
            float weight = i_Weight1[i - 4];
            pos_w.xyz += weight * NW_G3D_TRANSFORM_POS_OFFSET(mtxPalette, offset, i_Position);
            nrm_w.xyz += weight * NW_G3D_TRANSFORM_VEC_OFFSET(mtxPalette, offset, i_Normal);
            if (weight > 0)
            {
                ++influence;
            }
        }
    }

    vec4 pos_v = vec4(NW_G3D_TRANSFORM_POS(cameraMtx, pos_w), 1);
    vec3 nrm_v = NW_G3D_TRANSFORM_VEC(cameraMtx, nrm_w);
    gl_Position = NW_G3D_PROJECT(projMtx, pos_v);

    const vec3[9] tblSkinColor = vec3[9](
        vec3(0, 0, 0), vec3(1, 0, 0), vec3(0, 1, 0), vec3(0, 0, 1),
        vec3(1, 1, 0), vec3(1, 0, 1), vec3(0, 1, 1), vec3(1, 1, 1),
        vec3(0.5, 0.5, 0.5)
    );

#if OUTPUT == c_Pos_p
    OUT.pos_p = gl_Position;
#elif OUTPUT == c_Nrm_l
    OUT.vec = i_Normal;
#elif OUTPUT == c_Nrm_w
    OUT.vec = nrm_w.xyz;
#elif OUTPUT == c_Nrm_v
    OUT.vec = nrm_v;
#elif OUTPUT == c_Tan_l
    OUT.vec = i_Tangent.xyz;
#elif OUTPUT == c_Binrm_l
    OUT.vec = i_Binormal.xyz;
#elif OUTPUT == c_TanSpace
    OUT.color = vec4(clamp(i_Tangent.w, 0, 1), clamp(-i_Tangent.w, 0, 1), 0, 1);
#elif OUTPUT == c_SkinCnt
    OUT.color = vec4(tblSkinColor[vtxSkinCount], 1);
#elif OUTPUT == c_Influence
    OUT.color = vec4(tblSkinColor[influence], 1);
#elif OUTPUT == c_Color
    OUT.color = i_Color;
#elif OUTPUT == c_UV
    OUT.color = clamp(i_UV, 0, 1);
#elif OUTPUT >= c_Albedo
    OUT.uv = i_UV.xy;
#endif
}

#endif // NW_VERTEX_SHADER

//--------------------------------------------------------------------------------------------------

#if NW_FRAGMENT_SHADER

void main()
{
    vec4 color = vec4(0.5, 0.5, 0.5, 1);
#if OUTPUT == c_Pos_p
    color.rgb = IN.pos_p.xyz / IN.pos_p.w * 0.5 + vec3(0.5);
#elif c_Nrm_l <= OUTPUT && OUTPUT <= c_Binrm_l
    color.rgb = normalize(IN.vec) * 0.5 + vec3(0.5);
#elif c_TanSpace <= OUTPUT && OUTPUT <= c_UV
    color = IN.color;
#elif OUTPUT == c_Albedo
    color = texture(s_AlbedoTex, IN.uv.xy);
#elif OUTPUT == c_NrmMap
    vec4 nrm = texture(s_NormalTex, IN.uv.xy);
    float lengthSq = clamp(1 - nrm.x * nrm.x - nrm.z * nrm.z, 0, 1);
    nrm.z = sqrt(lengthSq);
    color.rgb = nrm.xyz * 0.5 + vec3(0.5);
#endif
    o_Color = color;
}

#endif // NW_FRAGMENT_SHADER
