﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NAudio.Wave;
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Linq;

namespace NintendoWare.Spy
{
    public class WaveformWaveSource : ObservableObject, IWaveSource
    {
        /// <summary>
        /// WaveformSpyModel のサンプルレートが分からないときの代わりの値。
        /// </summary>
        private const int FallbackSampleRate = 32000;

        private static readonly WaveSourceOutputInfo MainInfo = new WaveSourceOutputInfo(WaveformSpyModel.MainOutName, WaveformSpyModel.MainOutChannelCount);

        private readonly WeakReference<WaveformSpyModel> _model = new WeakReference<WaveformSpyModel>(null);
        private SpyGlobalTime _beginTime = SpyGlobalTime.InvalidValue;
        private SpyGlobalTime _endTime = SpyGlobalTime.InvalidValue;
        private SpyGlobalTime _currentTime = SpyGlobalTime.InvalidValue;

        //-----------------------------------------------------------------

        public int SamplingRate
        {
            get
            {
                var waveformModel = this.WaveformSpyModel;
                if (waveformModel == null || !waveformModel.IsWaveformMetadataReady)
                {
                    return FallbackSampleRate;
                }
                return waveformModel.SampleRate;
            }
        }

        public int BitsPerSample
        {
            get { return WaveformSpyModel.BitsPerSample; }
        }

        public SpyGlobalTime BeginTime
        {
            get { return _beginTime; }
            set
            {
                if (this.SetPropertyValue(ref _beginTime, value))
                {
                    IsPlaybackRegionChanged = true;
                }
            }
        }

        public SpyGlobalTime EndTime
        {
            get { return _endTime; }
            set
            {
                if (this.SetPropertyValue(ref _endTime, value))
                {
                    IsPlaybackRegionChanged = true;
                }
            }
        }

        public SpyGlobalTime CurrentTime
        {
            get { return _currentTime; }
            set
            {
                if (this.SetPropertyValue(ref _currentTime, value))
                {
                    IsPlaybackRegionChanged = true;
                }
            }
        }

        /// <summary>
        /// 先頭および末尾の波形情報のないフレームを無音で再生します。
        /// </summary>
        public bool PlayPrecedingAndSucceedingMissingFrame { get; set; } = false;

        /// <summary>
        /// 中間区間の波形情報のないフレームを無音で再生します。
        /// </summary>
        public bool PlayMissingFrame { get; set; } = true;

        public WaveformSpyModel WaveformSpyModel
        {
            get { return _model.GetTarget(); }
            set
            {
                if (this.WaveformSpyModel == value)
                {
                    return;
                }

                _model.SetTarget(value);
                this.NotifyPropertyChanged();
            }
        }

        /// <summary>
        /// BeginTime, EndTime, CurrentTime が変更されたことを示します。
        /// </summary>
        private bool IsPlaybackRegionChanged { get; set; }

        /// <summary>
        /// 波形データが終了したことを示します。
        /// </summary>
        private bool IsEndOfStream { get; set; }

        //-----------------------------------------------------------------

        public WaveformWaveSource()
        {
        }

        public IEnumerable<WaveSourceOutputInfo> EnumerateWaveSourceOutputInfos()
        {
            yield return MainInfo;
        }

        int IWaveSource.Mix(byte[] buffer, int offset, int count, string targetName, int channelCount)
        {
            Ensure.Argument.True(offset == 0); // NAudio.WaveBuffer は offset を扱えない。

            Ensure.True<NotSupportedException>(targetName == WaveformSpyModel.MainOutName);

            if (count <= 0)
            {
                return 0;
            }

            var currentTime = this.CurrentTime;
            if (this.IsPlaybackRegionChanged)
            {
                this.IsPlaybackRegionChanged = false;
                this.IsEndOfStream = false;

                currentTime = ValidateTime(currentTime);
            }

            if (currentTime == SpyGlobalTime.InvalidValue)
            {
                return 0;
            }

            // 再生範囲外なら、なにも返さない
            if (this.IsEndOfStream ||
                currentTime < this.BeginTime ||
                this.EndTime < currentTime)
            {
                return 0;
            }

            WaveBuffer waveBuffer = new WaveBuffer(buffer);
            var requiredSampleCount = count / sizeof(short) / channelCount; // 16 bit 固定
            var waveformModel = this.WaveformSpyModel;

            // Waveform データが存在しない場合、無音区間とみなし 0 を埋めて返す
            if (!IsWaveformAvailable(waveformModel))
            {
                this.ClearSamples(waveBuffer, 0, requiredSampleCount * channelCount);
                currentTime += SpyGlobalTime.FromSeconds(requiredSampleCount * this.SamplingRate);
                this.SetCurrentTime(currentTime);
                return requiredSampleCount * channelCount * sizeof(short);
            }

            var remainSampleCount = requiredSampleCount;
            var currentSampleIndex = this.CalcSampleIndex(waveformModel.BeginTime.Timestamp, currentTime);

            if (currentSampleIndex < 0)
            {
                if (this.PlayPrecedingAndSucceedingMissingFrame)
                {
                    // Waveform データより前のフレームは無音区間とみなし 0 を埋めて返す
                    var copyCount = (int)Math.Min(remainSampleCount, -currentSampleIndex);
                    this.ClearSamples(waveBuffer, 0, copyCount * channelCount);
                    currentSampleIndex += copyCount;

                    remainSampleCount -= copyCount;
                    if (remainSampleCount == 0)
                    {
                        this.UpdateCurrentTime(waveformModel.BeginTime.Timestamp, currentSampleIndex);
                        return copyCount * channelCount * sizeof(short);
                    }

                    Assertion.Operation.True(currentSampleIndex == 0);
                }
                else
                {
                    // Waveform データより前の無音区間をスキップします。
                    currentSampleIndex = 0;
                }
            }

            // 現在時刻を区間に含むか、それ以降の波形データのインデックスを取得します。
            int waveformIndex = waveformModel.GetWaveformIndex(currentTime);
            if (waveformIndex < 0)
            {
                waveformIndex = ~waveformIndex;
            }

            // Waveform データ内のサンプルをコピーする
            int destIndex = 0;
            while (remainSampleCount > 0)
            {
                var waveform = waveformModel.GetWaveform(waveformIndex++);
                if (waveform == null)
                {
                    currentSampleIndex += remainSampleCount;

                    if (this.PlayPrecedingAndSucceedingMissingFrame)
                    {
                        // Waveform データより後のフレームは無音区間とみなし０で埋めます。
                        this.ClearSamples(waveBuffer, destIndex, remainSampleCount * channelCount);
                        destIndex += remainSampleCount * channelCount;
                        remainSampleCount = 0;
                        break;
                    }
                    else
                    {
                        // Waveform データより後の無音区間をスキップします。
                        // TODO: SIGLO-47401 適切な範囲まで波形データを出力し確実に完了状態に遷移する
                        // HACK: SIGLO-47403 次の Mix() の呼び出しで再生範囲外に達したと判断させるために CurrentTime > EndTime にします
                        this.SetCurrentTime(SpyGlobalTime.FromMicroSeconds(this.EndTime.MicroSeconds + 1));
                        return destIndex * sizeof(short);
                    }
                }

                var waveformSampleIndex = waveform.SampleIndex;
                if (currentSampleIndex < waveformSampleIndex)
                {
                    if (this.PlayMissingFrame)
                    {
                        // 無音区間とみなし０で埋めます。
                        var copyCount = (int)Math.Min(remainSampleCount, waveformSampleIndex - currentSampleIndex);
                        Assertion.Operation.True(copyCount > 0);
                        this.ClearSamples(waveBuffer, destIndex, copyCount * channelCount);
                        destIndex += copyCount * channelCount;
                        remainSampleCount -= copyCount;
                        currentSampleIndex += copyCount;

                        if (remainSampleCount == 0)
                        {
                            break;
                        }
                    }
                    else
                    {
                        currentSampleIndex = waveformSampleIndex;
                    }
                }

                if (currentSampleIndex < waveformSampleIndex + waveform.SampleCount)
                {
                    int startIndex = (int)(currentSampleIndex - waveformSampleIndex);
                    int copyCount = Math.Min(remainSampleCount, waveform.SampleCount - startIndex);
                    Assertion.Operation.True(startIndex + copyCount <= waveform.SampleCount);
                    var sampleData = waveform.SampleData;

                    // ミックス処理
                    switch (channelCount)
                    {
                        case 2:
                            foreach (int sampleIndex in Enumerable.Range(startIndex, copyCount))
                            {
                                var l = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainFrontLeft][sampleIndex];
                                var r = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainFrontRight][sampleIndex];

                                waveBuffer.ShortBuffer[destIndex++] = l;
                                waveBuffer.ShortBuffer[destIndex++] = r;
                            }
                            break;

                        case 6:
                            foreach (int sampleIndex in Enumerable.Range(startIndex, copyCount))
                            {
                                var l = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainFrontLeft][sampleIndex];
                                var r = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainFrontRight][sampleIndex];
                                var rl = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainRearLeft][sampleIndex];
                                var rr = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainRearRight][sampleIndex];
                                var fc = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainFrontCenter][sampleIndex];
                                var lfe = sampleData.Samples[(int)WaveformSpyModel.ChannelIndex.MainLfe][sampleIndex];

                                waveBuffer.ShortBuffer[destIndex++] = l;
                                waveBuffer.ShortBuffer[destIndex++] = r;
                                waveBuffer.ShortBuffer[destIndex++] = fc;
                                waveBuffer.ShortBuffer[destIndex++] = lfe;
                                waveBuffer.ShortBuffer[destIndex++] = rl;
                                waveBuffer.ShortBuffer[destIndex++] = rr;
                            }
                            break;

                        default:
                            foreach (int sampleIndex in Enumerable.Range(startIndex, copyCount))
                            {
                                for (int i = 0; i < channelCount; ++i)
                                {
                                    waveBuffer.ShortBuffer[destIndex++] = sampleData.Samples[i][sampleIndex];
                                }
                            }
                            break;
                    }

                    remainSampleCount -= copyCount;
                    currentSampleIndex += copyCount;
                }
            }

            this.UpdateCurrentTime(waveformModel.BeginTime.Timestamp, currentSampleIndex);
            return destIndex * sizeof(short);
        }

        private static bool IsWaveformAvailable(WaveformSpyModel waveformModel)
        {
            return waveformModel != null && waveformModel.IsWaveformMetadataReady;
        }

        private void ClearSamples(WaveBuffer buffer, int startIndex, int sampleCount)
        {
            foreach (var sampleIndex in Enumerable.Range(startIndex, sampleCount))
            {
                buffer.ShortBuffer[sampleIndex] = 0;
            }
        }

        private long CalcSampleIndex(SpyGlobalTime baseTime, SpyGlobalTime currentTime)
        {
            return (long)Math.Ceiling((currentTime - baseTime).Seconds * this.SamplingRate);
        }

        /// <summary>
        /// 現在の baseTime と sampleIndex に基づいて CurrentTime を更新します。
        /// CurrentTime は更新しますが、IsPlaybackRegionChanged フラグは設定しません。
        /// </summary>
        private void UpdateCurrentTime(SpyGlobalTime baseTime, long sampleIndex)
        {
            var value = baseTime + SpyGlobalTime.FromSeconds(sampleIndex / (double)this.SamplingRate);
            this.SetCurrentTime(value);
        }

        /// <summary>
        /// CurrentTime は更新しますが、IsPlaybackRegionChanged フラグは設定しません。
        /// </summary>
        private void SetCurrentTime(SpyGlobalTime value)
        {
            this.SetPropertyValue(ref _currentTime, value, nameof(this.CurrentTime));
        }

        private SpyGlobalTime ValidateTime(SpyGlobalTime time)
        {
            if (time == SpyGlobalTime.InvalidValue)
            {
                return this.BeginTime;
            }

            if (time < this.BeginTime)
            {
                return this.BeginTime;
            }

            if (this.EndTime < time)
            {
                return this.EndTime;
            }

            return time;
        }

        private T Min<T>(T lhs, T rhs)
            where T : IComparable<T>
        {
            return lhs.CompareTo(rhs) <= 0 ? lhs : rhs;
        }
    }
}
