﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.IO;
using System.Web;

namespace NintendoWare.Spy
{
    /// <summary>
    /// パスに関するユーティリティクラスです。
    /// </summary>
    public static class PathUtility
    {
        /// <summary>
        /// 相対パスを生成します。
        /// </summary>
        /// <param name="path">対象の絶対パスを指定します。</param>
        /// <param name="basePath">相対パスの基準となるパスを指定します。</param>
        /// <returns>生成した相対パスを返します。</returns>
        public static string MakeRelativePath(string path, string basePath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(path);
            Ensure.Argument.StringIsNotNullOrEmpty(basePath);

            string relativeUri = new Uri(basePath).MakeRelativeUri(new Uri(path)).ToString();

            // URLデコードして、区切り文字を変更します。
            return HttpUtility.UrlDecode(relativeUri).Replace('/', '\\');
        }

        /// <summary>
        /// 絶対パスを生成します。
        /// </summary>
        /// <param name="path">対象の相対パスを指定します。</param>
        /// <param name="basePath">相対パスの基準となるパスを指定します。</param>
        /// <returns>生成した絶対パスを返します。</returns>
        public static string MakeAbsolutePath(string path, string basePath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(path);
            Ensure.Argument.StringIsNotNullOrEmpty(basePath);

            return Path.Combine(basePath, path).GetFullPath();
        }

        public static string GetFullPath(this string path)
        {
            string sourcePath = path;

            // Path.GetFullPath() を直接使用すると "../" 等を含む冗長なパス長に制約がかかるため、
            // Uri.LocalPath で先に最短パスに変換してから、Path.GetFullPath() にてパス長を検証しています。
            // ただし、path が相対パスの場合は Uri で例外が発生するため、path が絶対パスの場合にのみ実行します。
            if (Path.IsPathRooted(path))
            {
                sourcePath = new Uri(path).LocalPath;
            }

            return Path.GetFullPath(sourcePath);
        }
    }
}
