﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.ComponentModel;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Spy の時間情報を格納します。
    /// </summary>
    [TypeConverter(typeof(SpyTimeTypeConverter))]
    public sealed class SpyTime
    {
        public static readonly SpyTime Zero = new SpyTime(SpyGlobalTime.Zero, Frame.Zero, Frame.Zero);
        public static readonly SpyTime InvalidValue = new SpyTime(SpyGlobalTime.InvalidValue, Frame.InvalidValue, Frame.InvalidValue);

        private static readonly char[] Splitter = new char[] { ',' };

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="timestamp">実時間を指定します。</param>
        /// <param name="appFrame">アプリケーションフレームを指定します。</param>
        /// <param name="audioFrame">オーディオフレームを指定します。</param>
        public SpyTime(SpyGlobalTime timestamp, Frame appFrame, Frame audioFrame)
        {
            this.Timestamp = timestamp;
            this.AppFrame = appFrame;
            this.AudioFrame = audioFrame;
        }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="timestamp">実時間を指定します。</param>
        /// <param name="belongingFrame">フレームを指定します。</param>
        public SpyTime(SpyGlobalTime timestamp, SpyTime belongingFrame)
        {
            this.Timestamp = timestamp;
            this.AppFrame = belongingFrame.AppFrame;
            this.AudioFrame = belongingFrame.AudioFrame;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 実時間を取得します。
        /// </summary>
        public SpyGlobalTime Timestamp { get; private set; }

        /// <summary>
        /// アプリケーションフレームを取得します。
        /// </summary>
        public Frame AppFrame { get; private set; }

        /// <summary>
        /// オーディオフレームを取得します。
        /// </summary>
        public Frame AudioFrame { get; private set; }

        //-----------------------------------------------------------------

        public static bool operator ==(SpyTime left, SpyTime right)
        {
            if (object.ReferenceEquals(left, null) && object.ReferenceEquals(right, null))
            {
                return true;
            }

            if (object.ReferenceEquals(left, null))
            {
                return false;
            }

            return left.Equals(right);
        }

        public static bool operator !=(SpyTime left, SpyTime right)
        {
            return !(left == right);
        }

        public override int GetHashCode()
        {
            return this.Timestamp.GetHashCode() ^ this.AppFrame.GetHashCode() ^ this.AudioFrame.GetHashCode();
        }

        public override bool Equals(object obj)
        {
            var other = obj as SpyTime;

            if (other == null)
            {
                return false;
            }

            return this.Timestamp == other.Timestamp && this.AppFrame == other.AppFrame && this.AudioFrame == other.AudioFrame;
        }

        public override string ToString()
        {
            return string.Format(
                "Timestamp={0}, AppFrame={1} , AudioFrame={2}",
                this.Timestamp.ToString(),
                this.AppFrame.ToString(),
                this.AudioFrame.ToString());
        }

        public long GetMicroSeconds(long defaultValue = 0)
        {
            return this.Timestamp.IsValid ? this.Timestamp.MicroSeconds : defaultValue;
        }

        /// <summary>
        /// 時間の表示単位に応じたフレーム値を取得します。
        /// </summary>
        /// <param name="timeUnit"></param>
        /// <param name="defaultValue"></param>
        /// <returns></returns>
        public long SelectFrameValue(SpyTimeUnit timeUnit, long defaultValue = 0)
        {
            switch (timeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    return (this.AppFrame.IsValid) ? this.AppFrame.Value : defaultValue;

                case SpyTimeUnit.AudioFrame:
                    return (this.AudioFrame.IsValid) ? this.AudioFrame.Value : defaultValue;

                case SpyTimeUnit.Timestamp:
                case SpyTimeUnit.TimestampUsec:
                    return (this.Timestamp.IsValid) ? this.Timestamp.MicroSeconds : defaultValue;

                default:
                    throw new ArgumentException(string.Format("unexpected value ({0})", timeUnit), "timeUnit");
            }
        }

        /// <summary>
        /// 時間の表示単位にしたがってフレーム値を表示用の文字列に変換します。
        /// </summary>
        /// <param name="frameValue"></param>
        /// <param name="timeUnit"></param>
        /// <returns></returns>
        public static string ConvertFrameValueToLabelString(long frameValue, SpyTimeUnit timeUnit)
        {
            switch (timeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    return new Frame(frameValue).ToLabelString();

                case SpyTimeUnit.AudioFrame:
                    return new Frame(frameValue).ToLabelString();

                case SpyTimeUnit.Timestamp:
                    return SpyGlobalTime.FromMicroSeconds(frameValue).ToLabelString();

                case SpyTimeUnit.TimestampUsec:
                    return SpyGlobalTime.FromMicroSeconds(frameValue).ToMicroSecondsLabelString();

                default:
                    throw new ArgumentException(string.Format("unexpected value ({0})", timeUnit), "timeUnit");
            }
        }

        /// <summary>
        /// 時間単位に応じて SpyTime を表示用の文字列に変換します。
        /// </summary>
        /// <param name="timeUnit">時間単位</param>
        /// <returns>文字列を返します</returns>
        public string ToLabelString(SpyTimeUnit timeUnit)
        {
            switch (timeUnit)
            {
                case SpyTimeUnit.AppFrame:
                    return this.AppFrame.ToLabelString();

                case SpyTimeUnit.AudioFrame:
                    return this.AudioFrame.ToLabelString();

                case SpyTimeUnit.Timestamp:
                    return this.Timestamp.ToLabelString();

                case SpyTimeUnit.TimestampUsec:
                    return this.Timestamp.ToMicroSecondsLabelString();

                default:
                    throw new ArgumentException(string.Format("unexpected value ({0})", timeUnit), "timeUnit");
            }
        }

        /// <summary>
        /// 時間単位に応じた SpyTime のプロパティを使って SpyTime の時間を比較します。
        /// </summary>
        /// <param name="x">比較する最初のオブジェクト</param>
        /// <param name="y">比較する２番目のオブジェクト</param>
        /// <param name="timeUnit">比較に用いる時間単位</param>
        /// <returns>
        /// x &lt; y なら負の値を返します。
        /// x == y なら 0 を返します。
        /// x &gt; y なら正の値を返します。
        /// </returns>
        /// <remarks>
        /// timeUnit に SpyTimeUnit.Timestamp が指定された場合、 SpyTime.Timestamp のマイクロ秒の桁は無視されます。
        /// </remarks>
        public static int Compare(SpyTime x, SpyTime y, SpyTimeUnit timeUnit)
        {
            return x.SelectFrameValue(timeUnit).CompareTo(y.SelectFrameValue(timeUnit));
        }

        /// <summary>
        /// 文字列をパースし SpyTime の生成を試みます。
        /// フォーマットは "long,long,long" です。
        /// </summary>
        /// <param name="text"></param>
        /// <param name="time"></param>
        /// <returns></returns>
        /// <exception cref="ArgumentNullException"><c>text</c> 引数が null の場合。</exception>
        public static bool TryParse(string text, out SpyTime time)
        {
            Ensure.Argument.NotNull(text);

            time = SpyTime.InvalidValue;

            var values = text.Split(Splitter, 3);

            if (values == null || values.Length != 3)
            {
                return false;
            }

            long timestamp;
            long appFrame;
            long audioFrame;

            if (!long.TryParse(values[0], out timestamp) ||
                !long.TryParse(values[1], out appFrame) ||
                !long.TryParse(values[2], out audioFrame))
            {
                return false;
            }

            time = new SpyTime(
                new SpyGlobalTime(timestamp),
                new Frame(appFrame),
                new Frame(audioFrame));

            return true;
        }

        /// <summary>
        /// 文字列をパースし SpyTime を生成します。
        /// フォーマットは "long,long,long" です。
        /// </summary>
        /// <param name="text"></param>
        /// <returns></returns>
        /// <exception cref="ArgumentNullException"><c>text</c> 引数が null の場合。</exception>
        /// <exception cref="FormatException"><c>text</c> 引数が不正なフォーマットの場合。</exception>
        public static SpyTime Parse(string text)
        {
            Ensure.Argument.NotNull(text);

            SpyTime time;
            if (!TryParse(text, out time))
            {
                throw new FormatException("Invalid format. Should be \"long,long,long\".");
            }

            return time;
        }
    }
}
