﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Text.RegularExpressions;

namespace NintendoWare.Spy
{
    /// <summary>
    /// 時間を管理します。
    /// </summary>
    public struct SpyGlobalTime : IComparable<SpyGlobalTime>, IComparable
    {
        public static SpyGlobalTime Zero = new SpyGlobalTime(0);
        public static SpyGlobalTime InvalidValue = new SpyGlobalTime(long.MinValue);

        private static readonly Regex RegexPattern = new Regex(@"^(?<minus>-)?((?<sec>\d+(\.\d+)?)|((?<hour>\d+):)?(?<min>\d{1,2}):(?<sec>\d{1,2}(\.\d+)?))?$");

        /// <summary>
        /// InvalidValue から時間を取得しようとした場合の例外です。
        /// </summary>
        public class InvalidValueException : InvalidOperationException { }

        //-----------------------------------------------------------------

        private long _value;

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="value">時間[usec]を指定します。</param>
        public SpyGlobalTime(long value)
        {
            _value = value;
        }

        /// <summary>
        /// 指定した時間を表す SpyGlobalTime を生成します。
        /// </summary>
        /// <param name="value">秒単位の時間</param>
        /// <returns></returns>
        public static SpyGlobalTime FromSeconds(double value)
        {
            return new SpyGlobalTime((long)(value * 1000 * 1000));
        }

        /// <summary>
        /// 指定した時間を表す SpyGlobalTime を生成します。
        /// </summary>
        /// <param name="value">マイクロ秒単位の時間</param>
        /// <returns></returns>
        public static SpyGlobalTime FromMicroSeconds(long value)
        {
            return new SpyGlobalTime(value);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 時間を取得します。
        /// 値の単位は[usec]です。
        /// </summary>
        public long Value
        {
            get { return _value; }
        }

        /// <summary>
        /// 秒単位の値を取得します。
        /// </summary>
        /// <exception cref="InvalidValueException" />
        public double Seconds
        {
            get
            {
                if (!IsValid)
                {
                    throw new InvalidValueException();
                }

                return this.Value * 0.001 * 0.001;
            }
        }

        /// <summary>
        /// ミリ秒に丸められた値を取得します。
        /// </summary>
        /// <exception cref="InvalidValueException" />
        public long RoundedMilliSeconds
        {
            get
            {
                if (!IsValid)
                {
                    throw new InvalidValueException();
                }

                // NOTE:
                // マイクロ秒の単位は、より以前の時間（負の方向）に丸めます。
                //  1.000 →  1.0 [msec]
                //  0.999 →  0.0
                //  0.000 →  0.0
                // -0.001 → -1.0
                // -1.000 → -1.0
                // -1.001 → -2.0
                if (this.Value >= 0)
                {
                    return this.Value / 1000;
                }
                else
                {
                    return (this.Value - 999) / 1000;
                }
            }
        }

        /// <summary>
        /// マイクロ秒単位の値を取得します。
        /// </summary>
        /// <exception cref="InvalidValueException" />
        public long MicroSeconds
        {
            get
            {
                if (!IsValid)
                {
                    throw new InvalidValueException();
                }

                return this.Value;
            }
        }

        /// <summary>
        /// 有効な数値かどうかを調べます。
        /// </summary>
        public bool IsValid
        {
            get { return _value != InvalidValue.Value; }
        }

        //-----------------------------------------------------------------

        public override string ToString()
        {
            return this.Value.ToString();
        }

        public override bool Equals(object obj)
        {
            if (obj is SpyGlobalTime)
            {
                return this == (SpyGlobalTime)obj;
            }
            else
            {
                return false;
            }
        }

        public bool Equals(SpyGlobalTime other)
        {
            return this.Value == other.Value;
        }

        public override int GetHashCode()
        {
            return this.Value.GetHashCode();
        }

        public static bool operator ==(SpyGlobalTime left, SpyGlobalTime right)
        {
            return left.Value == right.Value;
        }

        public static bool operator !=(SpyGlobalTime left, SpyGlobalTime right)
        {
            return left.Value != right.Value;
        }

        public static bool operator <(SpyGlobalTime left, SpyGlobalTime right)
        {
            return left.Value < right.Value;
        }

        public static bool operator >(SpyGlobalTime left, SpyGlobalTime right)
        {
            return left.Value > right.Value;
        }

        public static bool operator <=(SpyGlobalTime left, SpyGlobalTime right)
        {
            return left.Value <= right.Value;
        }

        public static bool operator >=(SpyGlobalTime left, SpyGlobalTime right)
        {
            return left.Value >= right.Value;
        }

        public static SpyGlobalTime operator +(SpyGlobalTime left, SpyGlobalTime right)
        {
            return new SpyGlobalTime(left.Value + right.Value);
        }

        public static SpyGlobalTime operator -(SpyGlobalTime left, SpyGlobalTime right)
        {
            return new SpyGlobalTime(left.Value - right.Value);
        }

        public int CompareTo(SpyGlobalTime other)
        {
            return this.Value.CompareTo(other.Value);
        }

        public int CompareTo(object obj)
        {
            return this.CompareTo((SpyGlobalTime)obj);
        }

        /// <summary>
        /// 0:00:00.000 形式の文字列を時間に変換します。
        /// </summary>
        /// <param name="s">文字列</param>
        /// <returns>時間を返します</returns>
        public static SpyGlobalTime Parse(string s)
        {
            Ensure.Argument.NotNull(s);

            var match = RegexPattern.Match(s.Trim());
            if (!match.Success)
            {
                throw new FormatException("String must 0:00:00.000 format.");
            }

            double sec = 0;

            if (match.Groups["hour"].Success)
            {
                sec += long.Parse(match.Groups["hour"].Value) * 60 * 60;
            }

            if (match.Groups["min"].Success)
            {
                sec += long.Parse(match.Groups["min"].Value) * 60;
            }

            if (match.Groups["sec"].Success)
            {
                sec += double.Parse(match.Groups["sec"].Value);
            }

            if (match.Groups["minus"].Success)
            {
                sec = -sec;
            }

            return SpyGlobalTime.FromSeconds(sec);
        }

        /// <summary>
        /// 時間を 0:00:00.000 形式の文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        /// <remarks>
        /// [usec]の単位の情報は失われます。
        /// </remarks>
        public string ToLabelString()
        {
            if (!this.IsValid)
            {
                return "?";
            }
            else
            {
                var value = this.RoundedMilliSeconds;

                var msec = Math.Abs(value);
                var sec = msec / 1000;
                var minute = sec / 60;
                var hour = minute / 60;

                sec = sec % 60;
                minute = minute % 60;

                byte[] buff = new byte[10];
                buff[9] = (byte)(0x30 + msec % 10);
                buff[8] = (byte)(0x30 + (msec / 10) % 10);
                buff[7] = (byte)(0x30 + (msec / 100) % 10);
                buff[6] = 0x2e; // .
                buff[5] = (byte)(0x30 + sec % 10);
                buff[4] = (byte)(0x30 + (sec / 10) % 10);
                buff[3] = 0x3a; // :
                buff[2] = (byte)(0x30 + minute % 10);
                buff[1] = (byte)(0x30 + (minute / 10) % 10);
                buff[0] = 0x3a; // :

                var str = System.Text.Encoding.ASCII.GetString(buff);

                if (value >= 0)
                {
                    return hour.ToString() + str;
                }
                else
                {
                    return "-" + hour.ToString() + str;
                }
            }
        }

        /// <summary>
        /// 時間を 0:00:00.000000 形式の文字列に変換します。
        /// </summary>
        /// <returns>文字列を返します。</returns>
        public string ToMicroSecondsLabelString()
        {
            if (!this.IsValid)
            {
                return "?";
            }
            else
            {
                var usec = Math.Abs(_value);
                var msec = usec / 1000;
                var sec = msec / 1000;
                var minute = sec / 60;
                var hour = minute / 60;

                sec = sec % 60;
                minute = minute % 60;

                byte[] buff = new byte[13];
                buff[12] = (byte)(0x30 + usec % 10);
                buff[11] = (byte)(0x30 + (usec / 10) % 10);
                buff[10] = (byte)(0x30 + (usec / 100) % 10);
                buff[9] = (byte)(0x30 + msec % 10);
                buff[8] = (byte)(0x30 + (msec / 10) % 10);
                buff[7] = (byte)(0x30 + (msec / 100) % 10);
                buff[6] = 0x2e; // .
                buff[5] = (byte)(0x30 + sec % 10);
                buff[4] = (byte)(0x30 + (sec / 10) % 10);
                buff[3] = 0x3a; // :
                buff[2] = (byte)(0x30 + minute % 10);
                buff[1] = (byte)(0x30 + (minute / 10) % 10);
                buff[0] = 0x3a; // :

                var str = System.Text.Encoding.ASCII.GetString(buff);

                if (_value >= 0)
                {
                    return hour.ToString() + str;
                }
                else
                {
                    return "-" + hour.ToString() + str;
                }
            }
        }
    }
}
