﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;

namespace NintendoWare.Spy
{
    /// <summary>
    /// キーとなる時間情報を格納する Spy モデルです。
    /// </summary>
    public sealed class FrameSyncSpyModel : SpyModel
    {
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_2_0_0 = new Version(0, 2, 0, 0);

        /// <summary>
        /// バージョン 0.3.0.0
        /// </summary>
        /// <remarks>
        /// パケットフォーマット：
        /// <code>
        /// struct TimePacket {
        ///     s32 applicationFrame;
        ///     s32 audioFrame;
        /// };
        /// </code>
        /// </remarks>
        [SuppressMessage("StyleCop.CSharp.NamingRules", "SA1310:FieldNamesMustNotContainUnderscore", Justification = "バージョン番号のため")]
        private static readonly Version Version_0_3_0_0 = new Version(0, 3, 0, 0);

        /// <summary>
        /// 非サポートバージョン。
        /// 最新のサポートバージョンよりマイナーバージョンを１つ大きくします。
        /// </summary>
        private static readonly Version VersionUnexpected = new Version(0, 4, 0, 0);

        private readonly AllFrameList _allFrames = new AllFrameList();
        private readonly AppFrameList _appFrames = new AppFrameList();
        private readonly AudioFrameList _audioFrames = new AudioFrameList();
        private readonly List<SpyTime> _pendingFrames = new List<SpyTime>();

        private SpyTime _minimumFrame;
        private SpyTime _maximumFrame;

        private bool _errorUnexpectedDataVersion = false;

        /// <summary>
        /// フレーム同期情報の全リストを保持するクラスです。
        /// AppFrame または AudioFrame の変化した時間が昇順に記録されています。
        /// </summary>
        public interface IAllFrameList : IList<SpyTime>
        {
            /// <summary>
            /// 指定のタイムスタンプを持つ要素のインデックスを取得します。
            /// </summary>
            /// <param name="time"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            bool TryGetIndexOf(SpyGlobalTime time, out int index);

            /// <summary>
            /// 指定のタイムスタンプを区間に含む要素のインデックスを取得します。
            /// ( this[index].Time &lt;= timestamp &lt; this[index + 1].Time )
            /// 最後の区間は無限長として扱います。
            /// </summary>
            /// <param name="timestamp"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            bool TryFindIndexOf(SpyGlobalTime timestamp, out int index);

            /// <summary>
            /// 指定のタイムスタンプを持つ要素を取得します。
            /// </summary>
            /// <param name="timestamp"></param>
            /// <param name="value"></param>
            /// <returns></returns>
            bool TryGetValueOf(SpyGlobalTime timestamp, out SpyTime value);

            /// <summary>
            /// 指定のタイムスタンプを区間に含む要素を取得します。
            /// ( this[index].Time &lt;= timestamp &lt; this[index + 1].Time )
            /// 最後の区間は無限長として扱います。
            /// </summary>
            /// <param name="timestamp"></param>
            /// <param name="value"></param>
            /// <returns></returns>
            bool TryFindValueOf(SpyGlobalTime timestamp, out SpyTime value);
        }

        /// <summary>
        /// フレーム同期情報の AppFrame のリストを保持するクラスです。
        /// AppFrame の変化した時間が昇順に記録されています。
        /// </summary>
        public interface IAppFrameList : IList<SpyTime>
        {
            /// <summary>
            /// 指定の AppFrame を持つ要素のインデックスを取得します。
            /// </summary>
            /// <param name="appFrame"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            bool TryGetIndexOf(Frame appFrame, out int index);

            /// <summary>
            /// 指定の AppFrame を区間に含む要素のインデックスを取得します。
            /// ( Item[index].AppFrame &lt;= appFrame &lt; Item[index + 1].AppFrame )
            /// 最後の区間は無限長として扱います。
            /// </summary>
            /// <param name="appFrame"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            bool TryFindIndexOf(Frame appFrame, out int index);

            /// <summary>
            /// 指定の AppFrame を持つ要素を取得します。
            /// </summary>
            /// <param name="appFrame"></param>
            /// <param name="value"></param>
            /// <returns></returns>
            bool TryGetValueOf(Frame appFrame, out SpyTime value);

            /// <summary>
            /// 指定の AppFrame を区間に含む要素を取得します。
            /// ( Item[index].AppFrame &lt;= appFrame &lt; Item[index + 1].AppFrame )
            /// 最後の区間は無限長として扱います。
            /// </summary>
            /// <param name="appFrame"></param>
            /// <param name="value"></param>
            /// <returns></returns>
            bool TryFindValueOf(Frame appFrame, out SpyTime value);
        }

        /// <summary>
        /// フレーム同期情報の AudioFrame のリストを保持するクラスです。
        /// AudioFrame の変化した時間が昇順に記録されています。
        /// </summary>
        public interface IAudioFrameList : IList<SpyTime>
        {
            /// <summary>
            /// 指定の AudioFrame を持つ要素のインデックスを取得します。
            /// </summary>
            /// <param name="audioFrame"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            bool TryGetIndexOf(Frame audioFrame, out int index);

            /// <summary>
            /// 指定の AudioFrame を区間に含む要素のインデックスを取得します。
            /// ( Item[index].AudioFrame &lt;= audioFrame &lt; Item[index + 1].AudioFrame )
            /// 最後の区間は無限長として扱います。
            /// </summary>
            /// <param name="audioFrame"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            bool TryFindIndexOf(Frame audioFrame, out int index);

            /// <summary>
            /// 指定の AudioFrame を持つ要素を取得します。
            /// </summary>
            /// <param name="audioFrame"></param>
            /// <param name="value"></param>
            /// <returns></returns>
            bool TryGetValueOf(Frame audioFrame, out SpyTime value);

            /// <summary>
            /// 指定の AudioFrame を区間に含む要素を取得します。
            /// ( Item[index].AudioFrame &lt;= audioFrame &lt; Item[index + 1].AudioFrame )
            /// 最後の区間は無限長として扱います。
            /// </summary>
            /// <param name="audioFrame"></param>
            /// <param name="value"></param>
            /// <returns></returns>
            bool TryFindValueOf(Frame audioFrame, out SpyTime value);
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 最初の SpyTime です。
        /// </summary>
        public SpyTime MinimumFrame
        {
            get { return _minimumFrame; }
            private set { this.SetPropertyValue(ref _minimumFrame, value); }
        }

        /// <summary>
        /// 最後の SpyTime です。
        /// </summary>
        public SpyTime MaximumFrame
        {
            get { return _maximumFrame; }
            private set { this.SetPropertyValue(ref _maximumFrame, value); }
        }

        /// <summary>
        /// フレーム同期情報のリストです。
        /// AppFrame または AudioFrame の変化した時間が昇順に記録されています。
        /// </summary>
        public IAllFrameList AllFrames { get { return _allFrames; } }

        /// <summary>
        /// フレーム同期情報のリストです。
        /// AppFrame の変化した時間が昇順に記録されています。
        /// </summary>
        public IAppFrameList AppFrames { get { return _appFrames; } }

        /// <summary>
        /// フレーム同期情報のリストです。
        /// AudioFrame の変化した時間が昇順に記録されています。
        /// </summary>
        public IAudioFrameList AudioFrames { get { return _audioFrames; } }

        /// <summary>
        /// サポートしないバージョンのデータを受信すると true に設定されます。
        /// </summary>
        public bool ErrorUnexpectedDataVersion
        {
            get { return _errorUnexpectedDataVersion; }
            private set { this.SetPropertyValue(ref _errorUnexpectedDataVersion, value); }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// フレーム値の時間単位を変換します。
        /// </summary>
        /// <param name="fromFrame">元のフレーム値です。</param>
        /// <param name="fromUnit">元のフレーム値(fromFrame)の時間単位です。</param>
        /// <param name="toUnit">変換後の時間単位です。</param>
        /// <returns>変換されたフレーム値です。</returns>
        public long ConvertFrame(long fromFrame, SpyTimeUnit fromUnit, SpyTimeUnit toUnit)
        {
            Ensure.Argument.True(fromUnit != toUnit);

            if (this.AllFrames.Count == 0)
            {
                // FrameSync 情報が無いときは変換不能。
                return 0;
            }

            // fromFrameに対応する実時間(timestamp)を求める。
            long timestamp;
            switch (fromUnit)
            {
                case SpyTimeUnit.AppFrame:
                    {
                        int index;
                        if (!this.AppFrames.TryFindIndexOf(new Frame(fromFrame), out index))
                        {
                            return 0;
                        }
                        else
                        {
                            if (this.AppFrames[index].AppFrame.Value == fromFrame || index + 1 == this.AppFrames.Count)
                            {
                                timestamp = this.AppFrames[index].Timestamp.MicroSeconds;
                            }
                            else
                            {
                                timestamp = InterpolateFrame(
                                    fromFrame,
                                    SpyTimeUnit.AppFrame,
                                    SpyTimeUnit.TimestampUsec,
                                    this.AppFrames[index],
                                    this.AppFrames[index + 1]);
                            }
                        }
                    }
                    break;

                case SpyTimeUnit.AudioFrame:
                    {
                        int index;
                        if (!this.AudioFrames.TryFindIndexOf(new Frame(fromFrame), out index))
                        {
                            return 0;
                        }
                        else
                        {
                            if (this.AudioFrames[index].AudioFrame.Value == fromFrame || index + 1 == this.AudioFrames.Count)
                            {
                                timestamp = this.AudioFrames[index].Timestamp.MicroSeconds;
                            }
                            else
                            {
                                timestamp = InterpolateFrame(
                                    fromFrame,
                                    SpyTimeUnit.AudioFrame,
                                    SpyTimeUnit.TimestampUsec,
                                    this.AudioFrames[index],
                                    this.AudioFrames[index + 1]);
                            }
                        }
                    }
                    break;

                case SpyTimeUnit.Timestamp:
                case SpyTimeUnit.TimestampUsec:
                    timestamp = Math.Max(fromFrame, this.AllFrames[0].Timestamp.MicroSeconds);
                    break;

                default:
                    throw new ArgumentException(string.Format("unexpected value ({0})", fromUnit), "fromUnit");
            }

            // 実時間(timestamp)に対応するtoUnitの値を返す。
            switch (toUnit)
            {
                case SpyTimeUnit.TimestampUsec:
                case SpyTimeUnit.Timestamp:
                    return timestamp;

                case SpyTimeUnit.AppFrame:
                    {
                        SpyTime time;
                        if (!this.AllFrames.TryFindValueOf(SpyGlobalTime.FromMicroSeconds(timestamp), out time))
                        {
                            Ensure.Operation.Fail();
                        }

                        int index;
                        if (!this.AppFrames.TryGetIndexOf(time.AppFrame, out index))
                        {
                            Ensure.Operation.Fail();
                        }

                        if (this.AppFrames[index].Timestamp.MicroSeconds == timestamp || index + 1 == this.AppFrames.Count)
                        {
                            return this.AppFrames[index].AppFrame.Value;
                        }
                        else
                        {
                            return InterpolateFrame(
                                timestamp,
                                SpyTimeUnit.TimestampUsec,
                                SpyTimeUnit.AppFrame,
                                this.AppFrames[index],
                                this.AppFrames[index + 1]);
                        }
                    }

                case SpyTimeUnit.AudioFrame:
                    {
                        SpyTime time;
                        if (!this.AllFrames.TryFindValueOf(SpyGlobalTime.FromMicroSeconds(timestamp), out time))
                        {
                            Ensure.Operation.Fail();
                        }

                        int index;
                        if (!this.AudioFrames.TryGetIndexOf(time.AudioFrame, out index))
                        {
                            Ensure.Operation.Fail();
                        }

                        if (this.AudioFrames[index].Timestamp.MicroSeconds == timestamp || index + 1 == this.AudioFrames.Count)
                        {
                            return this.AudioFrames[index].AudioFrame.Value;
                        }
                        else
                        {
                            return InterpolateFrame(
                                timestamp,
                                SpyTimeUnit.TimestampUsec,
                                SpyTimeUnit.AudioFrame,
                                this.AudioFrames[index],
                                this.AudioFrames[index + 1]);
                        }
                    }

                default:
                    throw new ArgumentException(string.Format("unexpected value ({0})", toUnit), "toUnit");
            }
        }

        /// <summary>
        /// １つのフレーム区間内で線形補間によりフレーム値の時間単位を変換します。
        /// </summary>
        /// <param name="fromFrame"></param>
        /// <param name="fromUnit"></param>
        /// <param name="toUnit"></param>
        /// <param name="time0"></param>
        /// <param name="time1"></param>
        /// <returns></returns>
        private static long InterpolateFrame(long fromFrame, SpyTimeUnit fromUnit, SpyTimeUnit toUnit, SpyTime time0, SpyTime time1)
        {
            var f0 = time0.SelectFrameValue(fromUnit);
            var f1 = time1.SelectFrameValue(fromUnit);
            var t0 = time0.SelectFrameValue(toUnit);
            var t1 = time1.SelectFrameValue(toUnit);

            if (fromFrame == f0)
            {
                return t0;
            }
            else
            {
                Ensure.Operation.True(f0 != f1);
                return Convert.ToInt64((double)(fromFrame - f0) * (t1 - t0) / (f1 - f0)) + t0;
            }
        }

        protected override void OnPushData(SpyDataBlock dataBlock)
        {
            if (this.DataVersion >= VersionUnexpected)
            {
                this.ErrorUnexpectedDataVersion = true;
                return;
            }

            var reader = CreateDataReader(dataBlock);

            // 登録されるフレームは常に新しくなければなりません。
            if (MaximumFrame != null && MaximumFrame.Timestamp > dataBlock.Timestamp)
            {
                Assertion.Operation.Fail();
                return;
            }

            if (this.DataVersion < Version_0_2_0_0)
            {
                // 読み捨てる
                Frame spyFrame = new Frame(reader.ReadUInt32());
            }

            Frame appFrame;
            Frame audioFrame;

            if (this.DataVersion < Version_0_3_0_0)
            {
                appFrame = new Frame(reader.ReadUInt32());
                audioFrame = new Frame(reader.ReadUInt32());
            }
            else
            {
                appFrame = new Frame(reader.ReadInt32());
                audioFrame = new Frame(reader.ReadInt32());
            }

            var frame = new SpyTime(dataBlock.Timestamp, appFrame, audioFrame);

            if (this.MaximumFrame == null)
            {
                AddNewFrame(frame);
            }
            else
            {
                // 登録されるフレームは常に新しくなければなりません。
                if (appFrame <= this.MaximumFrame.AppFrame &&
                    audioFrame <= this.MaximumFrame.AudioFrame)
                {
                    return;
                }

                // 登録されるフレームは appFrame と audioFrame の一方だけが変化する必要があります。
                // 同時に変化しているときはマルチスレッド実行によりパケット送信順が前後しているので、
                // 正しい順番のパケットが到着するまで登録を保留します。
                if (this.MaximumFrame.AppFrame < appFrame &&
                    this.MaximumFrame.AudioFrame < audioFrame)
                {
                    var index = _pendingFrames.BinarySearch(frame, new PendingFrameComparer());
                    if (index >= 0)
                    {
                        // 同じ内容のフレーム同期情報は許されません。
                        Assertion.Operation.Fail();
                        return;
                    }

                    index = ~index;
                    _pendingFrames.Insert(index, frame);

                    return;
                }

                AddNewFrame(frame);

                while (!_pendingFrames.IsEmpty())
                {
                    var pendingHead = _pendingFrames[0];

                    if (this.MaximumFrame.AppFrame == pendingHead.AppFrame ||
                        this.MaximumFrame.AudioFrame == pendingHead.AudioFrame)
                    {
                        AddNewFrame(pendingHead);
                        _pendingFrames.RemoveAt(0);

                        continue;
                    }

                    Assertion.Operation.True(
                        this.MaximumFrame.AppFrame < pendingHead.AppFrame &&
                        this.MaximumFrame.AudioFrame < pendingHead.AudioFrame);

                    break;
                }
            }
        }

        private void AddNewFrame(SpyTime frame)
        {
            if (this.MaximumFrame == null || this.MaximumFrame.AppFrame < frame.AppFrame)
            {
                _appFrames.Add(frame);
            }

            if (this.MaximumFrame == null || this.MaximumFrame.AudioFrame < frame.AudioFrame)
            {
                _audioFrames.Add(frame);
            }

            _allFrames.Add(frame);

            if (this.MinimumFrame == null)
            {
                this.MinimumFrame = frame;
            }

            this.MaximumFrame = frame;
        }

        /// <summary>
        /// キーにより要素のインデックスを素早く取得できるようにしたリスト型です。
        /// </summary>
        /// <typeparam name="TKey"></typeparam>
        /// <typeparam name="TValue"></typeparam>
        private class KeyIndexList<TKey, TValue> : List<TValue>
        {
            protected readonly Dictionary<TKey, int> Index = new Dictionary<TKey, int>();

            /// <summary>
            /// キーと要素を登録します。
            /// </summary>
            /// <remarks>
            /// 同じキーを持つ要素を登録することはできますが、
            /// ２番目以降に登録された要素のインデックスをキーから取得することは出来ません。
            /// </remarks>
            /// <param name="key"></param>
            /// <param name="value"></param>
            public void Add(TKey key, TValue value)
            {
                if (!this.Index.ContainsKey(key))
                {
                    this.Index.Add(key, this.Count);
                }

                base.Add(value);
            }

            public bool TryGetIndexOf(TKey key, out int index)
            {
                return this.Index.TryGetValue(key, out index);
            }

            public bool TryGetValueOf(TKey key, out TValue value)
            {
                int index;
                if (this.TryGetIndexOf(key, out index))
                {
                    value = this[index];
                    return true;
                }
                else
                {
                    value = default(TValue);
                    return false;
                }
            }
        }

        /// <summary>
        /// 連続してならぶ区間を表すリスト型です。
        /// KeyedList のキーが区間の開始位置を表します。
        /// </summary>
        /// <typeparam name="TKey"></typeparam>
        /// <typeparam name="TValue"></typeparam>
        private class FrameList<TKey, TValue> : KeyIndexList<TKey, TValue>
        {
            /// <summary>
            /// 要素からキーを取得する選択子です。
            /// </summary>
            protected Func<TValue, TKey> Selector { get; private set; }

            public FrameList(Func<TValue, TKey> selector)
            {
                this.Selector = selector;
            }

            /// <summary>
            /// リストに要素を追加します。
            /// </summary>
            /// <param name="item"></param>
            public new void Add(TValue item)
            {
                base.Add(this.Selector(item), item);
            }

            /// <summary>
            /// 指定したキーを含む区間のインデックスを取得します。
            /// </summary>
            /// <param name="key"></param>
            /// <param name="index"></param>
            /// <returns></returns>
            public bool TryFindIndexOf(TKey key, out int index)
            {
                if (this.TryGetIndexOf(key, out index))
                {
                    return true;
                }
                else
                {
                    // ひとつ小さい SpyTime を探します。
                    var result = ~BinarySearchUtility.BinarySearch(this, key, this.Selector) - 1;
                    if (result >= 0)
                    {
                        index = result;
                        return true;
                    }
                    else
                    {
                        return false;
                    }
                }
            }

            public bool TryFindValueOf(TKey key, out TValue value)
            {
                int index;
                if (this.TryFindIndexOf(key, out index))
                {
                    value = this[index];
                    return true;
                }
                else
                {
                    value = default(TValue);
                    return false;
                }
            }
        }

        private class AllFrameList : FrameList<SpyGlobalTime, SpyTime>, IAllFrameList
        {
            public AllFrameList()
                : base(i => i.Timestamp)
            {
            }
        }

        private class AppFrameList : FrameList<Frame, SpyTime>, IAppFrameList
        {
            public AppFrameList()
                : base(i => i.AppFrame)
            {
            }
        }

        private class AudioFrameList : FrameList<Frame, SpyTime>, IAudioFrameList
        {
            public AudioFrameList()
                : base(i => i.AudioFrame)
            {
            }
        }

        private class PendingFrameComparer : IComparer<SpyTime>
        {
            public int Compare(SpyTime x, SpyTime y)
            {
                var result = x.Timestamp.CompareTo(y.Timestamp);
                if (result != 0)
                {
                    return result;
                }

                var appResult = x.AppFrame.CompareTo(y.AppFrame);
                var audioResult = x.AudioFrame.CompareTo(y.AudioFrame);

                if (appResult < 0 || audioResult < 0)
                {
                    return -1;
                }
                else if (appResult == 0 && audioResult == 0)
                {
                    return 0;
                }
                else
                {
                    return 1;
                }
            }
        }
    }
}
