﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Binary;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace NintendoWare.Spy
{
    /// <summary>
    /// Spy ファイルを管理します。
    /// </summary>
    internal sealed class SpyFileManager : DisposableObject, ISpyDataReader
    {
        private const long InvalidDataBlockPosition = -1;

        private readonly Dictionary<string, string> _listFilePathCaches = new Dictionary<string, string>();

        /// <summary>
        /// データファイルパス辞書です。（Key=データ名, Value=.spydat ファイルパス）
        /// </summary>
        private readonly Dictionary<string, string> _dataFilePathCaches = new Dictionary<string, string>();

        private readonly Dictionary<string, SpyDataBinaryWriter> _binaryWriters = new Dictionary<string, SpyDataBinaryWriter>();
        private readonly ConcurrentDictionary<string, SpyDataBinaryReader> _binaryReaders = new ConcurrentDictionary<string, SpyDataBinaryReader>();

        /// <summary>
        /// データブロックID -> データブロック位置への辞書です。
        /// </summary>
        private readonly Dictionary<long, long> _dataBlockPositions = new Dictionary<long, long>();

        private string _directoryPath = string.Empty;

        private uint _currentDataBlockID = 0;
        private bool? _isLittleEndian = null;

        /// <summary>
        /// リード中のデータファイルのコンテナバージョン。
        /// すべてのデータファイルで一致する必要があります。
        /// </summary>
        private Version _containerVersion;

        /// <summary>
        /// 受信ディレクトリを作った時刻です。
        /// 次の受信ディレクトリを作るときに、前の受信ディレクトリをリネームするのに使います。
        /// </summary>
        private DateTime? _directoryLastCreateTime;

        //-----------------------------------------------------------------

        public bool IsOpened
        {
            get { return !string.IsNullOrEmpty(_directoryPath); }
        }

        //-----------------------------------------------------------------

        public void Create(string directoryPath, bool isLittleEndian)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(directoryPath);

            this.Close();

            _directoryLastCreateTime = DateTime.Now;

            // Directory.Delete() でフォルダごと削除して、直後に CreateDirectory() すると、
            // ディレクトリが作成されないことがあるようなので、該当ディレクトリの中身だけを
            // 削除するようにしています。
            Directory.CreateDirectory(directoryPath);
            this.DeleteDirectoryContents(directoryPath);

            _directoryPath = directoryPath;
            _isLittleEndian = isLittleEndian;
        }

        public void Open(string directoryPath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(directoryPath);
            Ensure.True<DirectoryNotFoundException>(Directory.Exists(directoryPath));

            this.Close();

            _directoryPath = directoryPath;
        }

        public void Close()
        {
            foreach (var writer in _binaryWriters.Values)
            {
                writer.Dispose();
            }

            foreach (var reader in _binaryReaders.Values)
            {
                lock (reader)
                {
                    reader.Close();
                }
            }

            _binaryWriters.Clear();
            _binaryReaders.Clear();
            _dataBlockPositions.Clear();

            _isLittleEndian = null;

            _listFilePathCaches.Clear();
            _dataFilePathCaches.Clear();

            _currentDataBlockID = 0;
            _containerVersion = null;

            if (_directoryLastCreateTime.HasValue)
            {
                // これまで開いていた受信ディレクトリの名前を日時に変更します。
                var parentDir = Directory.GetParent(_directoryPath).FullName;
                var newName = DirectoryUtility.GetDateTimePath(parentDir, _directoryLastCreateTime.Value);
                Directory.Move(_directoryPath, newName);
                _directoryLastCreateTime = null;
            }

            _directoryPath = string.Empty;
        }

        public void CloseDataFile(string filePath)
        {
            SpyDataBinaryReader reader;

            if (!_binaryReaders.TryRemove(filePath, out reader))
            {
                return;
            }

            lock (reader)
            {
                reader.Close();
            }
        }

        public void FlushFiles()
        {
            _binaryWriters.Values.ForEach(writer => writer.WriteTotalDataBlockLength());
            _binaryWriters.Values.ForEach(writer => writer.Dispose());
            _binaryWriters.Clear();
        }

        public SpyDataBinaryHeader ReadDataFileHeader(string filePath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(filePath);

            Ensure.Operation.True(this.IsOpened);
            Ensure.Operation.True(Path.GetDirectoryName(filePath) == _directoryPath);

            if (Path.GetExtension(filePath) != ConstConfig.SpyDataFileExtention)
            {
                throw new InvalidDataException();
            }

            var result = this.GetSpyDataBinaryReader(filePath).Header;
            Assertion.Operation.NotNull(result);

            if (_isLittleEndian.HasValue)
            {
                Ensure.Operation.True(result.IsLittleEndian == _isLittleEndian.Value);
            }
            else
            {
                _isLittleEndian = result.IsLittleEndian;
            }

            // リード用のすべてのデータファイルでコンテナバージョンが一致していることを確認します。
            {
                if (_containerVersion == null)
                {
                    _containerVersion = result.ContainerVersion;
                }

                Ensure.Operation.AreEqual(_containerVersion, result.ContainerVersion);
            }

            return result;
        }

        public IEnumerable<SpyDataBlock> ReadDataFileBlocks(string filePath)
        {
            Ensure.Argument.StringIsNotNullOrEmpty(filePath);

            Ensure.Operation.True(this.IsOpened);
            Ensure.Operation.True(Path.GetDirectoryName(filePath) == _directoryPath);

            if (Path.GetExtension(filePath) != ConstConfig.SpyDataFileExtention)
            {
                throw new InvalidDataException();
            }

            var reader = this.GetSpyDataBinaryReader(filePath);

            while (true)
            {
                long currentPosition = reader.GetCurrentPosition();
                var binaryBlock = reader.ReadNextBlock();

                if (binaryBlock == null)
                {
                    break;
                }

                var dataBlock = new SpyDataBlock(binaryBlock.ID, binaryBlock.RawData, binaryBlock.Timestamp, currentPosition);

                if (_dataBlockPositions.ContainsKey(dataBlock.ID))
                {
                    // 古いデータファイルから読み込んだ場合は ID が一意で無いため、
                    // 誤ったデータを読めないようにしておきます。
                    _dataBlockPositions[dataBlock.ID] = InvalidDataBlockPosition;
                }
                else
                {
                    _dataBlockPositions.Add(dataBlock.ID, currentPosition);
                }

                yield return dataBlock;
            }
        }

        /// <summary>
        /// データファイルが作成済みかどうかを調べます。
        /// </summary>
        /// <param name="dataName"></param>
        /// <returns></returns>
        public bool IsDataFileMade(string dataName)
        {
            return _binaryWriters.ContainsKey(this.GetDataFilePath(dataName));
        }

        /// <summary>
        /// データファイルを作成します。
        /// </summary>
        /// <param name="dataID">データ ID を指定します。</param>
        /// <param name="dataName">データ名を指定します。</param>
        /// <param name="dataVersion">データバージョンを指定します。</param>
        public void MakeDataFile(uint dataID, string dataName, Version dataVersion)
        {
            Assertion.Argument.True(dataID > 0);
            Assertion.Argument.StringIsNotNullOrEmpty(dataName);
            Assertion.Argument.NotNull(dataVersion);

            Ensure.Operation.True(_isLittleEndian.HasValue);

            SpyDataBinaryWriter writer = null;
            var filePath = this.GetDataFilePath(dataName);

            // ファイルが開かれていない前提
            Ensure.Operation.False(_binaryWriters.TryGetValue(filePath, out writer));

            var fileStream = File.Open(filePath, FileMode.Create, FileAccess.ReadWrite, FileShare.Read);

            try
            {
                writer = new SpyDataBinaryWriter();
                writer.Open(fileStream, _isLittleEndian.Value);

                writer.WriteHeader(dataName, dataVersion);

                _binaryWriters.Add(filePath, writer);
            }
            catch
            {
                fileStream.Dispose();
                throw;
            }
        }

        /// <summary>
        /// データブロックを書き込みます。
        /// 戻り値のSpyDataBlockにデータブロック ID が記録されます。
        /// </summary>
        /// <param name="dataName">データ名を指定します。</param>
        /// <param name="rawData">データ配列を指定します。</param>
        /// <param name="timestamp">データの送信時間[usec]です。</param>
        public SpyDataBlock WriteDataBlock(string dataName, byte[] rawData, long timestamp)
        {
            var writer = this.GetSpyDataBinaryWriter(dataName);

            var id = this.NextDataBlockID();
            var position = writer.GetNextBlockPosition();

            writer.WriteBlock(id, rawData, timestamp);
            _dataBlockPositions.Add(id, position);

            return new SpyDataBlock(id, rawData, timestamp, position);
        }

        /// <summary>
        /// リストファイルを作成します。
        /// </summary>
        /// <param name="baseName">リストファイルのファイル名（拡張子を除く）を指定します。</param>
        /// <returns>
        /// 作成したリストファイルパスを返します。
        /// 対象のデータファイルが存在せず、リストファイルを作成しなかった場合は null を返します。
        /// </returns>
        public string MakeListFile(string baseName)
        {
            if (_binaryWriters.Count == 0)
            {
                return null;
            }

            var paths = new SortedList<string, string>();

            foreach (var item in _binaryWriters)
            {
                paths.Add(item.Key, item.Key);
            }

            string filePath = this.GetListFilePath(baseName);

            using (var fileStream = File.Open(
                filePath,
                FileMode.Create,
                FileAccess.ReadWrite,
                FileShare.Read))
            {
                using (var listWriter = new SpyDataListWriter(fileStream))
                {
                    foreach (var path in paths.Values)
                    {
                        listWriter.WriteItem(path);
                    }
                }
            }

            return filePath;
        }

        /// <summary>
        /// データファイルをディレクトリからディレクトリにコピーします。
        /// </summary>
        public void CopyDataFileTo(IEnumerable<string> dataNames, string sourceDirectory, string destinationDirectory)
        {
            foreach (string dataName in dataNames)
            {
                string fileName = this.GetDataFileName(dataName);
                string sourceFilePath = Path.Combine(sourceDirectory, fileName);
                string destinationFilePath = Path.Combine(destinationDirectory, fileName);

                if (File.Exists(sourceFilePath) != false)
                {
                    File.Copy(sourceFilePath, destinationFilePath, true);
                }
            }
        }

        bool? ISpyDataReader.IsLittleEndian
        {
            get
            {
                return _isLittleEndian;
            }
        }

        SpyDataBlock ISpyDataReader.ReadDataBlock(string dataName, long dataBlockID)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(dataName);

            long dataBlockPosition = 0;

            if (!_dataBlockPositions.TryGetValue(dataBlockID, out dataBlockPosition))
            {
                return null;
            }

            if (dataBlockPosition == InvalidDataBlockPosition)
            {
                return null;
            }

            SpyDataBinaryReader reader = this.GetSpyDataBinaryReaderFromDataName(dataName);

            SpyDataBinaryBlock binaryBlock = null;
            try
            {
                lock (reader)
                {
                    if (reader.IsDisposed)
                    {
                        return null;
                    }

                    reader.SetCurrentPosition(dataBlockPosition);
                    binaryBlock = reader.ReadNextBlock();
                }

                if (binaryBlock == null)
                {
                    return null;
                }
            }
            catch (EndOfStreamException)
            {
                return null;
            }

            Ensure.True(dataBlockID == binaryBlock.ID, () => new InvalidDataException());

            return new SpyDataBlock(dataBlockID, binaryBlock.RawData, (long)(binaryBlock.Timestamp), dataBlockPosition);
        }

        protected override void Dispose(bool disposing)
        {
            this.Close();
            base.Dispose(disposing);
        }

        private uint NextDataBlockID()
        {
            return _currentDataBlockID++;
        }

        private void DeleteDirectoryContents(string directoryPath)
        {
            if (!Directory.Exists(directoryPath))
            {
                return;
            }

            try
            {
                foreach (var path in Directory.EnumerateDirectories(directoryPath))
                {
                    Directory.Delete(path, true);
                }

                foreach (var path in Directory.EnumerateFiles(directoryPath))
                {
                    File.Delete(path);
                }
            }
            catch
            {
            }
        }

        private SpyDataBinaryWriter GetSpyDataBinaryWriter(string dataName)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(dataName);

            SpyDataBinaryWriter writer = null;
            var filePath = this.GetDataFilePath(dataName);

            // MakeDataFile() でファイルが開かれている前提
            Ensure.Operation.True(_binaryWriters.TryGetValue(filePath, out writer));
            return writer;
        }

        private SpyDataBinaryReader GetSpyDataBinaryReader(string filePath)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(filePath);

            SpyDataBinaryReader reader = null;

            if (_binaryReaders.TryGetValue(filePath, out reader))
            {
                return reader;
            }

            var fileStream = File.Open(filePath, FileMode.Open, FileAccess.Read, FileShare.ReadWrite);

            try
            {
                reader = new SpyDataBinaryReader();
                reader.Open(fileStream);

                reader.ReadHeader();

                if (!_binaryReaders.TryAdd(filePath, reader))
                {
                    // 別のスレッドに先を越された場合。
                    fileStream.Dispose();
                    _binaryReaders.TryGetValue(filePath, out reader);
                }
            }
            catch
            {
                fileStream.Dispose();
                throw;
            }

            return reader;
        }

        private SpyDataBinaryReader GetSpyDataBinaryReaderFromDataName(string dataName)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(dataName);
            return this.GetSpyDataBinaryReader(this.GetDataFilePath(dataName));
        }

        private string GetListFilePath(string baseName)
        {
            Assertion.Operation.StringIsNotNullOrEmpty(baseName);
            Assertion.Operation.StringIsNotNullOrEmpty(_directoryPath);

            string result;

            // Path.GetFullPath() が遅いのでキャッシュ
            if (_listFilePathCaches.TryGetValue(baseName, out result))
            {
                return result;
            }

            result = Path.GetFullPath(
                Path.Combine(_directoryPath, baseName + ConstConfig.SpyDataListFileExtention));

            _listFilePathCaches.Add(baseName, result);
            return result;
        }

        private string GetDataFileName(string dataName)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(dataName);
            return dataName + ConstConfig.SpyDataFileExtention;
        }

        private string GetDataFilePath(string dataName)
        {
            Assertion.Argument.StringIsNotNullOrEmpty(_directoryPath);

            string result;

            // Path.GetFullPath() が遅いのでキャッシュ
            if (_dataFilePathCaches.TryGetValue(dataName, out result))
            {
                return result;
            }

            result = Path.GetFullPath(
                Path.Combine(_directoryPath, this.GetDataFileName(dataName)));

            _dataFilePathCaches.Add(dataName, result);
            return result;
        }
    }
}
