﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Media;
using System.Windows.Threading;

namespace NintendoWare.Spy.Windows.Primitives
{
    public class PlotGridItemsControl : ItemsControl
    {
        private const int MaximumGridCount = 500;

        public static readonly DependencyProperty OriginXProperty = DependencyProperty.Register(
            nameof(OriginX),
            typeof(double),
            typeof(PlotGridItemsControl),
            new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnOriginXChanged(e)));

        public static readonly DependencyProperty ScaleXProperty = DependencyProperty.Register(
            nameof(ScaleX),
            typeof(double),
            typeof(PlotGridItemsControl),
            new FrameworkPropertyMetadata(double.NaN, (d, e) => Self(d).OnScaleXChanged(e)));

        public static readonly DependencyProperty GridFrequencyProperty = DependencyProperty.Register(
            nameof(GridFrequency),
            typeof(double),
            typeof(PlotGridItemsControl),
            new FrameworkPropertyMetadata(100.0, (d, e) => Self(d).OnGridDistanceChanged(e)));

        public static readonly DependencyProperty LabelTemplateProperty = DependencyProperty.Register(
            nameof(LabelTemplate),
            typeof(DataTemplate),
            typeof(PlotGridItemsControl),
            new FrameworkPropertyMetadata(null));

        public static readonly DependencyProperty GridStrokeProperty = DependencyProperty.Register(
            nameof(GridStroke),
            typeof(Brush),
            typeof(PlotGridItemsControl),
            new FrameworkPropertyMetadata(Brushes.DimGray));

        private readonly MergedRequestDispatcher _updateGridsDispatcher = new MergedRequestDispatcher();

        public double OriginX
        {
            get { return (double)this.GetValue(OriginXProperty); }
            set { this.SetValue(OriginXProperty, value); }
        }

        public double ScaleX
        {
            get { return (double)this.GetValue(ScaleXProperty); }
            set { this.SetValue(ScaleXProperty, value); }
        }

        private double ValidatedOriginX
        {
            get { return PlotCanvas.GetValidatedOriginX(this.OriginX); }
        }

        private double ValidatedScaleX
        {
            get { return PlotCanvas.GetValidatedScaleX(this.ScaleX); }
        }

        /// <summary>
        /// 保証すべきグリッド間隔(ピクセル)です。
        /// 実際のグリッド間隔はこの値以上になります。
        /// </summary>
        public double GridFrequency
        {
            get { return (double)this.GetValue(GridFrequencyProperty); }
            set { this.SetValue(GridFrequencyProperty, value); }
        }

        public DataTemplate LabelTemplate
        {
            get { return (DataTemplate)this.GetValue(LabelTemplateProperty); }
            set { this.SetValue(LabelTemplateProperty, value); }
        }

        public Brush GridStroke
        {
            get { return (Brush)this.GetValue(GridStrokeProperty); }
            set { this.SetValue(GridStrokeProperty, value); }
        }

        public PlotGridItemsControl()
        {
        }

        protected override void OnRenderSizeChanged(SizeChangedInfo sizeInfo)
        {
            base.OnRenderSizeChanged(sizeInfo);

            this.RequestUpdateGrids();
        }

        protected override DependencyObject GetContainerForItemOverride()
        {
            return new PlotGridItem();
        }

        protected override bool IsItemItsOwnContainerOverride(object item)
        {
            return item is PlotGridItem;
        }

        protected override void PrepareContainerForItemOverride(DependencyObject element, object item)
        {
            base.PrepareContainerForItemOverride(element, item);

            var itemContainer = (PlotGridItem)item;
            itemContainer.Stroke = this.GridStroke;

            if (this.LabelTemplate != null)
            {
                itemContainer.ContentTemplate = this.LabelTemplate;
            }
        }

        private void OnOriginXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGrids();
        }

        private void OnScaleXChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGrids();
        }

        private void OnGridDistanceChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGrids();
        }

        private void OnTimeUnitChanged(DependencyPropertyChangedEventArgs e)
        {
            this.RequestUpdateGrids();
        }

        private void RequestUpdateGrids()
        {
            _updateGridsDispatcher.Request(UpdateGrids, DispatcherPriority.Render);
        }

        private void UpdateGrids()
        {
            if (this.ActualWidth <= 0)
            {
                this.Items.Clear();
                return;
            }

            var originX = this.ValidatedOriginX;
            var scaleX = this.ValidatedScaleX;
            var reciprocalScaleX = 1.0 / scaleX;
            var screenLeft = originX * scaleX;

            var gridWidthX = CalcGridDistance();

            var startX = GetLowerMultipleOf(originX, gridWidthX);
            var endX = GetUpperMultipleOf(originX + this.ActualWidth * reciprocalScaleX, gridWidthX);

            if (endX <= startX)
            {
                this.Items.Clear();
                return;
            }

            var gridCount = Math.Min(MaximumGridCount, (int)((endX - startX) / gridWidthX) + 1);

            for (int i = 0; i < gridCount; i++)
            {
                Assertion.Operation.True(i <= this.Items.Count);

                if (this.Items.Count == i)
                {
                    this.Items.Add(new PlotGridItem());
                }

                var x = startX + i * gridWidthX;

                var item = (PlotGridItem)this.Items[i];
                item.Visibility = Visibility.Visible;
                item.SetValue(PlotCanvas.LeftXProperty, x);
                item.Content = x;
            }

            // 非表示のグリッドのうち、数の多すぎる部分を削除します。
            var maxGridCount = Math.Max(gridCount * 2, 100);
            while (maxGridCount < this.Items.Count)
            {
                this.Items.RemoveAt(this.Items.Count - 1);
            }

            // 未使用のグリッドを非表示にします。
            this.Items.Cast<PlotGridItem>().Skip(gridCount).ForEach(grid => grid.Visibility = Visibility.Hidden);
        }

        /// <summary>
        /// 目盛間隔を求めます。
        /// </summary>
        private double CalcGridDistance()
        {
            // グリッド間隔として保証する最低の幅 (X座標)
            var widthX = Math.Max(1, this.GridFrequency) / this.ValidatedScaleX;

            // widthX 以下の 10 のべき乗のうち、最も大きい値を求めます。
            var widthBase = Math.Pow(10.0, Math.Floor(Math.Log10(widthX)));

            if (widthX <= widthBase * 1)
            {
                return widthBase * 1;
            }
            else if (widthX <= widthBase * 2)
            {
                return widthBase * 2;
            }
            else if (widthX <= widthBase * 5)
            {
                return widthBase * 5;
            }
            else
            {
                return widthBase * 10;
            }
        }

        private static double GetUpperMultipleOf(double value, double round)
        {
            return Math.Ceiling(value / round) * round;
        }

        private static double GetLowerMultipleOf(double value, double round)
        {
            return Math.Floor(value / round) * round;
        }

        private static PlotGridItemsControl Self(DependencyObject d)
        {
            return (PlotGridItemsControl)d;
        }
    }
}
