﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.Contracts;
using System;
using System.Globalization;
using System.Windows;
using System.Windows.Data;

namespace NintendoWare.Spy.Windows
{
    /// <summary>
    /// データサイズの単位を変換します。
    /// </summary>
    [ValueConversion(typeof(object), typeof(string))]
    public sealed class DataSizeUnitConverter : IValueConverter
    {
        /// <summary>
        /// 単位間のスケール
        /// </summary>
        private const int ScalePerUnit = 1024;

        /// <summary>
        /// サイズの単位を示します。
        /// </summary>
        public enum UnitType : int
        {
            /// <summary>
            /// 自動でできるだけ大きい単位に調整します。
            /// DestinationUnit にのみ指定できます。
            /// </summary>
            Auto = 0,

            Byte = 1,
            KB,
            MB,
            GB,
            TB,
        }

        /// <summary>
        /// 変換元の単位を取得または設定します。
        /// デフォルト値は、Byte です。
        /// </summary>
        public UnitType SourceUnit { get; set; } = UnitType.Byte;

        /// <summary>
        /// 変換後の単位を取得または設定します。
        /// デフォルト値は、Auto です。
        /// </summary>
        public UnitType DestinationUnit { get; set; } = UnitType.Auto;

        /// <summary>
        /// 変換後の double 値を文字列化する書式を取得または設定します。
        /// デフォルト値は、"0.00" です。
        /// </summary>
        public string DestinationValueFormat { get; set; } = "0.00";

        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            Ensure.Operation.True(this.SourceUnit != UnitType.Auto);
            Ensure.Operation.True(this.DestinationUnit == UnitType.Auto || this.SourceUnit <= this.DestinationUnit);

            if (value == null)
            {
                return DependencyProperty.UnsetValue;
            }

            var doubleValue = ValidateValue(value);
            var destinationUnit = this.GetValidDestinationUnit(doubleValue);
            var destinationValue = this.CalculateValue(doubleValue, this.SourceUnit, destinationUnit);

            return $"{destinationValue.ToString(this.DestinationValueFormat)} {destinationUnit.ToString()}";
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return DependencyProperty.UnsetValue;
        }

        private double ValidateValue(object value)
        {
            return System.Convert.ToDouble(value);
        }

        private double CalculateValue(double value, UnitType sourceUnit, UnitType destinationUnit)
        {
            Ensure.Argument.True(destinationUnit != UnitType.Auto);

            if (double.IsNaN(value) || double.IsInfinity(value))
            {
                return 0.0d;
            }

            var unitDelta = (int)this.SourceUnit - (int)destinationUnit;

            if (unitDelta == 0)
            {
                return value;
            }

            return value * Math.Pow(ScalePerUnit, unitDelta);
        }

        private UnitType GetValidDestinationUnit(double value)
        {
            if (this.DestinationUnit != UnitType.Auto)
            {
                return this.DestinationUnit;
            }

            if (double.IsNaN(value) || double.IsInfinity(value))
            {
                return this.SourceUnit;
            }

            // SourceUnit ⇔ UnitType.TB の間に収める
            var power = Math.Log(Math.Abs(value), ScalePerUnit);
            var unitDelta = (int)Math.Floor(MathUtility.Clamp(power, 0d, UnitType.TB - this.SourceUnit));
            return this.SourceUnit + unitDelta;
        }
    }
}
