﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using System;

namespace NintendoWare.Spy.Framework
{
    public abstract class ApplicationFramework : ObservableObject
    {
        private IServiceProvider _rootServiceProvider;
        private ApplicationPresenter _applicationPresenter;

        //-----------------------------------------------------------------

        /// <summary>
        /// フレームワークを初期化します。
        /// </summary>
        public void Initialize(IApplicationFrameworkSettings settings)
        {
            Ensure.Argument.NotNull(settings);

            this.InitializeRootServiceProvider(settings);
            this.InitializePresenter(settings);
        }

        /// <summary>
        /// フレームワークの終了処理を行います。
        /// </summary>
        public void Uninitialize()
        {
            this.UninitializePresenter();
            this.UninitializeRootServiceProvider();
        }

        /// <summary>
        /// アプリケーションを実行します。
        /// </summary>
        /// <remarks>
        /// このメソッドはアプリケーションの終了通知を検知するまで制御を返します。
        /// </remarks>
        public void Run()
        {
            this.RunApplication();
        }

        /// <summary>
        /// アプリケーションを終了します。
        /// </summary>
        public void Exit()
        {
            int exitCode = 0;
            System.Windows.Application.Current.Shutdown(exitCode);
        }

        /// <summary>
        /// アプリケーションを終了可能かどうか調べます。
        /// </summary>
        /// <returns>終了可能な場合は true、不可能な場合は false を返します。</returns>
        public bool CanExit()
        {
            return false;
        }

        private void InitializeRootServiceProvider(IApplicationFrameworkSettings settings)
        {
            Assertion.Argument.NotNull(settings);

            _rootServiceProvider = settings.CreateRootServiceProvider();
            Ensure.Operation.NotNull(_rootServiceProvider);
        }

        private void InitializePresenter(IApplicationFrameworkSettings settings)
        {
            Assertion.Argument.NotNull(settings);

            _applicationPresenter = settings.CreateApplicationPresenterFunc();
            Ensure.Operation.NotNull(_applicationPresenter);

            _applicationPresenter.Initialize(_rootServiceProvider, null);
        }

        private void UninitializeRootServiceProvider()
        {
            var disposableServiceProvider = _rootServiceProvider as IDisposable;
            Disposer.SafeDispose(ref disposableServiceProvider);

            _rootServiceProvider = null;
        }

        private void UninitializePresenter()
        {
            if (_applicationPresenter == null)
            {
                return;
            }

            _applicationPresenter.Uninitialize();
            Disposer.SafeDispose(ref _applicationPresenter);
        }

        private void RunApplication()
        {
            Ensure.Operation.NotNull(_applicationPresenter);
            _applicationPresenter.Run();
        }
    }
}
