﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace NintendoWare.Spy.Foundation.Commands
{
    /// <summary>
    /// コマンドの処理結果を示します。
    /// </summary>
    public sealed class CommandResult
    {
        private readonly Code _code;
        private readonly string _message;
        private readonly object _returnValue;

        //-----------------------------------------------------------------

        public enum Code : int
        {
            Succeeded = 0,
            Canceled = 1,
            Failed = -1,
            CommandNotFound = -2,
            CommandTargetNotFound = -3,
            CommandHandlerNotFound = -4,
            CommandIDNotSupported = -5,
            InvalidArguments = -6,
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="code">コマンドの処理結果コードを指定します。</param>
        /// <param name="returnValue">コマンドの戻り値を指定します。</param>
        /// <param name="message">メッセージを指定します。</param>
        public CommandResult(Code code, object returnValue = null, string message = null)
        {
            _code = code;
            _message = message ?? string.Empty;
            _returnValue = returnValue;
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// コマンドの成否を取得します。
        /// </summary>
        public bool IsSucceeded
        {
            get { return _code == Code.Succeeded; }
        }

        /// <summary>
        /// コマンドがキャンセルされたかどうかを取得します。
        /// </summary>
        public bool IsCanceled
        {
            get { return _code == Code.Canceled; }
        }

        /// <summary>
        /// エラーが発生したかどうかを取得します。
        /// </summary>
        public bool IsError
        {
            get { return (int)_code < 0; }
        }

        /// <summary>
        /// 詳細な処理結果コードを取得します。
        /// </summary>
        public Code DetailCode
        {
            get { return _code; }
        }

        /// <summary>
        /// メッセージを取得します。
        /// </summary>
        public string Message
        {
            get { return _message; }
        }

        /// <summary>
        /// コマンドの戻り値を取得します。
        /// </summary>
        public object ReturnValue
        {
            get { return _returnValue; }
        }

        //-----------------------------------------------------------------

        /// <summary>
        /// 成功を示す CommandResult を作成します。
        /// </summary>
        /// <param name="returnValue">コマンドの戻り値を指定します。</param>
        /// <param name="message">メッセージを指定します。</param>
        /// <returns>成功を示す CommandResult を返します。</returns>
        public static CommandResult CreateSucceeded(object returnValue = null, string message = null)
        {
            return new CommandResult(Code.Succeeded, returnValue, message);
        }

        /// <summary>
        /// キャンセルを示す CommandResult を作成します。
        /// </summary>
        /// <param name="message">メッセージを指定します。</param>
        /// <returns>キャンセルを示す CommandResult を返します。</returns>
        public static CommandResult CreateCanceled(string message = null)
        {
            return new CommandResult(Code.Canceled, null, message);
        }
    }
}
