﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.Spy.Windows
{
    public class PlotNodeData : ObservableObject
    {
        private readonly ObservableKeyedList<string, PlotFloatData> _floats = new ObservableKeyedList<string, PlotFloatData>(item => item.Name);
        private IList<PlotFloatData> _filteredFloats;
        private readonly ObservableKeyedList<string, PlotStateData> _states = new ObservableKeyedList<string, PlotStateData>(it => it.Name);
        private IList<PlotStateData> _filteredStates;
        private bool _isChecked = true;
        private double _currentFrame = double.NaN;
        private double _minimumFrame = double.NaN;
        private double _maximumFrame = double.NaN;
        private string _itemNameFilterText;
        private double _scaleY = 0.9;

        public string Name { get; }

        public double Height { get; set; } = double.NaN;

        public bool IsGlobal { get; set; }

        public bool IsChecked
        {
            get { return _isChecked; }
            set { this.SetPropertyValue(ref _isChecked, value); }
        }

        public ObservableCollection<PlotFloatData> Floats
        {
            get { return _floats; }
        }

        public IList<PlotFloatData> FilteredFloats
        {
            get { return _filteredFloats; }
            private set { this.SetPropertyValue(ref _filteredFloats, value); }
        }

        public ObservableCollection<PlotStateData> States
        {
            get { return _states; }
        }

        public IList<PlotStateData> FilteredStates
        {
            get { return _filteredStates; }
            private set { this.SetPropertyValue(ref _filteredStates, value); }
        }

        public double CurrentFrame
        {
            get { return _currentFrame; }
            set
            {
                if (this.SetPropertyValue(ref _currentFrame, value))
                {
                    foreach (var state in _states)
                    {
                        state.CurrentFrame = value;
                    }
                }
            }
        }

        public double MinimumFrame
        {
            get { return _minimumFrame; }
            set
            {
                if (this.SetPropertyValue(ref _minimumFrame, value))
                {
                    foreach (var floats in _floats)
                    {
                        floats.MinimumX = value;
                    }
                }
            }
        }

        public double MaximumFrame
        {
            get { return _maximumFrame; }
            set
            {
                if (this.SetPropertyValue(ref _maximumFrame, value))
                {
                    foreach (var floats in _floats)
                    {
                        floats.MaximumX = value;
                    }
                }
            }
        }

        public string ItemNameFilterText
        {
            get { return _itemNameFilterText; }
            set
            {
                if (this.SetPropertyValue(ref _itemNameFilterText, value))
                {
                    if (string.IsNullOrEmpty(_itemNameFilterText))
                    {
                        this.FilteredFloats = _floats;
                    }
                    else
                    {
                        var filterText = _itemNameFilterText;
                        this.FilteredFloats = new WhereListDecorator<PlotFloatData>(
                            item => item.Name.IndexOf(filterText, StringComparison.OrdinalIgnoreCase) >= 0,
                            _floats as IReadOnlyList<PlotFloatData>,
                            dependentPropertyNames: null);
                    }
                }
            }
        }

        public double ScaleY
        {
            get { return _scaleY; }
            set { this.SetPropertyValue(ref _scaleY, value); }
        }

        public PlotNodeData(string name)
        {
            this.Name = name;

            // グループ無しを手軽に使えるように、デフォルトの高さを大きめに。
            this.Height = string.IsNullOrEmpty(name) ? 200d : 100d;

            _filteredFloats = _floats;
            _filteredStates = _states;
        }

        protected override void DisposeManagedInstance()
        {
            _floats.ForEach(it => it.Dispose());
            _states.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public void NewPlotFloat(PlotSpyModel.PlotFloat item)
        {
            if (_floats.ContainsKey(item.Name) == false)
            {
                var itemData = new PlotFloatData(item)
                {
                    MinimumX = this.MinimumFrame,
                    MaximumX = this.MaximumFrame,
                };
                _floats.Add(itemData);
            }
        }

        public void NewPlotState(PlotSpyModel.PlotState state)
        {
            if (_states.ContainsKey(state.Name) == false)
            {
                var stateData = new PlotStateData(state)
                {
                    CurrentFrame = this.CurrentFrame,
                };
                _states.Add(stateData);
            }
        }

        public void CheckAllFloats()
        {
            foreach (var item in _floats)
            {
                item.IsVisible = true;
            }
        }

        public void UncheckAllFloats()
        {
            foreach (var item in _floats)
            {
                item.IsVisible = false;
            }
        }
    }
}
