﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public sealed class TimelineViewModel : ObservableObject
    {
        private readonly WeakReference<SoundStatusInfoSpyModel> _infoModel = new WeakReference<SoundStatusInfoSpyModel>(null);
        private readonly WeakReference<SoundDataInfoSpyModel> _labelModel = new WeakReference<SoundDataInfoSpyModel>(null);
        private readonly ObservableCollection<TimelineItemViewModel> _items = new ObservableCollection<TimelineItemViewModel>();
        private readonly List<TimelineItemViewModel> _activeItems = new List<TimelineItemViewModel>();

        private SpyGlobalTime _lastTimestamp;
        private SpyTime _startTime = SpyTime.InvalidValue;
        private SpyTime _endTime = SpyTime.InvalidValue;

        /// <summary>
        /// モデル
        /// </summary>
        public SoundStatusInfoSpyModel InfoModel
        {
            get { return _infoModel.GetTarget(); }
        }

        /// <summary>
        /// モデル
        /// </summary>
        public SoundDataInfoSpyModel LabelModel
        {
            get { return _labelModel.GetTarget(); }
        }

        /// <summary>
        /// トーンイベントリスト
        /// </summary>
        public IList<TimelineItemViewModel> Items
        {
            get { return _items; }
        }

        /// <summary>
        /// 開始時刻
        /// </summary>
        public SpyTime StartTime
        {
            get { return _startTime; }
            set
            {
                this.SetPropertyValue(ref _startTime, value);
            }
        }

        /// <summary>
        /// 終了時刻
        /// </summary>
        public SpyTime EndTime
        {
            get { return _endTime; }
            set
            {
                this.SetPropertyValue(ref _endTime, value);
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TimelineViewModel()
        {
            _items = new ObservableCollection<TimelineItemViewModel>();
        }

        protected override void DisposeManagedInstance()
        {
            Items.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        /// <summary>
        /// モデルのセット
        /// </summary>
        /// <param name="model"></param>
        public void SetupModel(SoundStatusInfoSpyModel model)
        {
            var targetModel = _infoModel.GetTarget();

            if (targetModel == model)
            {
                return;
            }

            _infoModel.SetTarget(model);

            if (InfoModel != null)
            {
                InfoModel.SoundInfoChanged += UpdateInfo;
                InfoModel.ReadAllSoundStatusInfos();
            }
        }

        public void SetupModel(SoundDataInfoSpyModel model)
        {
            var targetModel = _labelModel.GetTarget();

            if (targetModel == model)
            {
                return;
            }

            _labelModel.SetTarget(model);

            if (LabelModel != null)
            {
                UpdateLabels();
            }
        }

        public void CloseStatusModel()
        {
            this.Clear();
        }

        private void Clear()
        {
            Items.ForEach(it => it.Dispose());
            Items.Clear();
            _activeItems.Clear();
            _lastTimestamp = SpyGlobalTime.Zero;
        }

        /// <summary>
        /// ラベルリストが変更されたら各ラインの名前を更新する
        /// </summary>
        private void UpdateLabels()
        {
            foreach (var line in Items)
            {
                line.Name = LabelModel.GetLabel(line.SoundId);
                line.PlayerId = LabelModel.GetPlayerId(line.SoundId);
                line.PlayerName = LabelModel.GetPlayerName(line.PlayerId);
            }
        }

        /// <summary>
        /// ランタイムからフレーム毎のサウンド情報が送信される度に呼ばれる。
        /// イベントが何もなくても呼ばれる。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="args"></param>
        private void UpdateInfo(object sender, SoundStatusInfoSpyModel.SoundInfoChangedEventArgs args)
        {
            if (args.Timestamp <= _lastTimestamp)
            {
                this.Clear();
            }

            _lastTimestamp = args.Timestamp;

            var checkedInfos = new List<SoundStatusInfoSpyModel.SoundInfo>();

            for (int i = _activeItems.Count - 1; i >= 0; i--)
            {
                var item = _activeItems[i];

                bool isUpdate = false;
                foreach (var info in args.SoundInfos)
                {
                    if (item.InstanceId == info.InstanceId)
                    {
                        var activeState = item.ActiveState;

                        if (activeState.State == TimelineItemStateViewModel.GetStateType(info))
                        {
                            item.Update(args.BelongingFrame);
                        }
                        else
                        {
                            item.AppendState(new TimelineItemStateViewModel(item, info));
                        }

                        checkedInfos.Add(info);
                        isUpdate = true;
                        break;
                    }
                }

                // イベントが終了した場合
                if (!isUpdate)
                {
                    item.Close(args.BelongingFrame);
                    _activeItems.Remove(item);
                }
            }

            // 新規に開始したイベント
            foreach (var info in args.SoundInfos)
            {
                bool isPass = false;
                foreach (var checkedInfo in checkedInfos)
                {
                    if (info == checkedInfo)
                    {
                        isPass = true;
                        checkedInfos.Remove(checkedInfo);
                        break;
                    }
                }
                if (isPass)
                {
                    continue;
                }

                var newItem = new TimelineItemViewModel()
                {
                    SoundId = info.SoundId,
                    InstanceId = info.InstanceId,
                    Name = LabelModel == null ? info.SoundId.ToString() : LabelModel.GetLabel(info.SoundId),
                    PlayerId = LabelModel == null ? 0 : LabelModel.GetPlayerId(info.SoundId),
                };
                newItem.PlayerName = LabelModel == null ? "Player" : LabelModel.GetPlayerName(newItem.PlayerId);
                newItem.AppendState(new TimelineItemStateViewModel(newItem, info));

                Items.Add(newItem);

                _activeItems.Add(newItem);
            }
        }
    }
}
