﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.NwSoundSpyPlugin.Plugins;
using NintendoWare.Spy;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Settings;
using NintendoWare.Spy.Windows;
using System;
using System.Windows.Controls;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public sealed class TimelinePanelPresenter : SpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private readonly WeakReference<FrameSyncSpyModel> _frameSync = new WeakReference<FrameSyncSpyModel>(null);
        private TimelineViewModel _viewModel;
        private TimelinePanel _view;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private int _grabToken = SpyPlaybackService.TokenGrabInvalid;

        private FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSync.GetTarget(); }
            set
            {
                _frameSync.SetTarget(value);
            }
        }

        protected override void OnInitialize()
        {
            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => UpdateGlobalTimelineCurrent());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Begin,
                    (target, args) => UpdateGlobalTimelineBegin());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.End,
                    (target, args) => UpdateGlobalTimelineEnd());

            base.OnInitialize();
        }

        protected override Control CreateContent()
        {
            Assertion.Operation.Null(_viewModel);
            Assertion.Operation.Null(_view);

            _viewModel = new TimelineViewModel();

            _view = new TimelinePanel()
            {
                DataContext = _viewModel,
            };

            _view.SetPresenter(this);
            _view.SetViewModel(_viewModel.Items);
            _view.DragCurrentFrame += this.HandleDragCurrentFrame;

            this.UpdateGlobalTimelineBegin();
            this.UpdateGlobalTimelineEnd();

            // 初期状態のカレントフレームを設定します。
            // View.ActualWidth が確定し、ViewModel.StartFrame, EndFrame が View に伝わるまで遅延します。
            _view.Loaded += HandleViewLoaded;

            return _view;
        }

        protected override void OnUninitialize()
        {
            _view.Loaded -= HandleViewLoaded;

            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        private void HandleViewLoaded(object sender, System.Windows.RoutedEventArgs e)
        {
            _view.Loaded -= HandleViewLoaded;
            this.UpdateGlobalTimelineCurrent();
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            var soundInfoSpyModel = model as SoundStatusInfoSpyModel;

            if (soundInfoSpyModel != null && _viewModel.InfoModel != soundInfoSpyModel)
            {
                _viewModel.SetupModel(soundInfoSpyModel);
            }

            var soundLabelModel = model as SoundDataInfoSpyModel;

            if (soundLabelModel != null && _viewModel.LabelModel != soundLabelModel)
            {
                _viewModel.SetupModel(soundLabelModel);
            }

            if (dataName == SoundStatusInfoSpyModelPlugin.SpyDataName && model == null)
            {
                // 閉じられたら消す
                _viewModel.CloseStatusModel();
            }

            if (dataName == FrameSyncSpyModelPlugin.SpyDataName)
            {
                var frameSync = model as FrameSyncSpyModel;
                this.FrameSyncModel = frameSync;
            }
        }

        /// <summary>
        /// グローバルタイムラインが更新されたら
        /// </summary>
        private void UpdateGlobalTimelineCurrent()
        {
            if (_view != null)
            {
                var playbackService = this.GetPlaybackService();
                var current = playbackService.Current;
                if (current.Timestamp.IsValid)
                {
                    _view.UpdateCurrentPos(current.SelectFrameValue(_view.TimeUnit), playbackService.ShouldAutoScroll);
                }
            }
        }

        /// <summary>
        /// グローバルタイムラインの開始フレームが更新されたら
        /// </summary>
        private void UpdateGlobalTimelineBegin()
        {
            var beginTime = GetPlaybackService().Begin;

            if (_viewModel != null && beginTime.Timestamp.IsValid)
            {
                _viewModel.StartTime = beginTime;
            }
        }

        /// <summary>
        /// グローバルタイムラインの最終フレームが更新されたら
        /// </summary>
        private void UpdateGlobalTimelineEnd()
        {
            var endTime = GetPlaybackService().End;

            if (_viewModel != null && endTime.Timestamp.IsValid)
            {
                _viewModel.EndTime = endTime;

                foreach (var item in _viewModel.Items)
                {
                    if (item.IsActive)
                    {
                        item.Update(endTime);
                    }
                }
            }
        }

        /// <summary>
        /// カレントフレームのドラッグを処理します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void HandleDragCurrentFrame(object sender, DragMoveEventArgs e)
        {
            switch (e.Action)
            {
                case DragMoveAction.Start:
                    _grabToken = GetPlaybackService().GetGrabForCurrent();
                    break;

                case DragMoveAction.Update:
                    if (_view != null)
                    {
                        GetPlaybackService().SetCurrentTime(_view.CurrentFrame, _view.TimeUnit, _grabToken);
                    }
                    break;

                case DragMoveAction.End:
                    GetPlaybackService().ReleaseGrabForCurrent(_grabToken);
                    _grabToken = SpyPlaybackService.TokenGrabInvalid;
                    break;
            }
        }

        private void ApplySettings(ApplicationSettings settings)
        {
            if (settings != null)
            {
                _timeUnit = settings.TimeUnit;
            }

            if (_view != null)
            {
                _view.ChangeTimeUnit(_timeUnit, this.FrameSyncModel);
            }
        }

        protected override void OnSettingsApplied(SettingsService service)
        {
            this.ApplySettings(service.GetApplicationSettings());

            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, service.GetApplicationSettings())
                .AddHandlerForAnyProperties(
                    (target, args) => ApplySettings(target));

            base.OnSettingsApplied(service);
        }
    }
}
