﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Shapes;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public partial class TimelineItem : TimelineItemBase
    {
        private readonly List<TimelineItemState> _states = new List<TimelineItemState>();
        private int _verticalPos = -1;
        private static double _canvasHeight = 18;
        private static double _canvasMargin = 2;

        private delegate void ViewOffsetChangedHandler();

        public static double ItemHeight { get { return _canvasHeight + _canvasMargin; } }

        public static string FilterText { get; set; }

        /// <summary>
        /// 時間の表示単位
        /// </summary>
        public SpyTimeUnit TimeUnit
        {
            set
            {
                foreach (var state in _states)
                {
                    state.TimeUnit = value;
                }
            }
        }

        /// <summary>
        /// 開始時刻
        /// </summary>
        public SpyTime StartTime
        {
            get { return (_states.Count > 0) ? _states[0].StartTime : SpyTime.Zero; }
        }

        /// <summary>
        /// 終了時刻
        /// </summary>
        public SpyTime EndTime
        {
            get { return (_states.Count > 0) ? _states[_states.Count - 1].EndTime : SpyTime.Zero; }
        }

        /// <summary>
        /// 開始フレーム
        /// </summary>
        public long StartFrame
        {
            get { return (_states.Count > 0) ? _states[0].StartFrame : 0; }
        }

        /// <summary>
        /// 終了フレーム
        /// </summary>
        public long EndFrame
        {
            get { return (_states.Count > 0) ? _states[_states.Count - 1].EndFrame : 0; }
        }

        public static readonly DependencyProperty PlayerIdProperty =
            DependencyProperty.Register(nameof(PlayerId), typeof(uint), typeof(TimelineItem));

        public uint PlayerId
        {
            get { return (uint)GetValue(PlayerIdProperty); }
            set { SetValue(PlayerIdProperty, value); }
        }

        /// <summary>
        /// ラベルID
        /// </summary>
        public uint SoundId
        {
            get { return (uint)GetValue(SoundIdProperty); }
            set { SetValue(SoundIdProperty, value); }
        }

        // Using a DependencyProperty as the backing store for SoundId.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty SoundIdProperty =
            DependencyProperty.Register(nameof(SoundId), typeof(uint), typeof(TimelineItem));

        /// <summary>
        /// ラベル名
        /// </summary>
        public string Label
        {
            get { return (string)GetValue(LabelProperty); }
            set { SetValue(LabelProperty, value); }
        }

        public static readonly DependencyProperty LabelProperty =
            DependencyProperty.Register(nameof(Label), typeof(string), typeof(TimelineItem), new PropertyMetadata(LabelChanged));

        private static void LabelChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var item = (TimelineItem)d;

            if (!string.IsNullOrEmpty((string)e.NewValue))
            {
                ((TimelineItem)d).ToolTip = (string)e.NewValue;
            }
        }

        /// <summary>
        /// 生きてるかどうか
        /// </summary>
        public bool IsActive
        {
            get { return (bool)GetValue(IsActiveProperty); }
            set { SetValue(IsActiveProperty, value); }
        }

        public static readonly DependencyProperty IsActiveProperty =
            DependencyProperty.Register(nameof(IsActive), typeof(bool), typeof(TimelineItem));

        public int VerticalPos
        {
            get { return _verticalPos; }

            set
            {
                if (_verticalPos == value)
                {
                    return;
                }

                _verticalPos = value;
                UpdateView(true);
            }
        }

        public static double CanvasHeight
        {
            get { return _canvasHeight; }
            set
            {
                _canvasHeight = value;
            }
        }

        public Canvas ParentCanvas { get; set; }

        /// <summary>
        /// 初期化
        /// </summary>
        /// <param name="text"></param>
        public TimelineItem()
        {
            InitializeComponent();

            Canvas.SetZIndex(border, 1);

            Canvas.SetZIndex(Text, 1);

            Label = string.Empty;

            UpdateView();

            DataContextChanged += (sender, e) =>
            {
                SetBinding(PlayerIdProperty, "PlayerId");
                SetBinding(SoundIdProperty, "SoundId");
                SetBinding(LabelProperty, "Name");
            };
        }

        /// <summary>
        /// ステートを追加
        /// </summary>
        /// <param name="state"></param>
        public void AddState(TimelineItemState state)
        {
            if (state != null)
            {
                if (_states.Count > 0)
                {
                    _states.Last().EndFrameChanged -= HandleEndFrameChanged;
                }

                _states.Add(state);
                this.Children.Add(state);

                state.EndFrameChanged += HandleEndFrameChanged;

                UpdateView();
            }
        }

        private void HandleEndFrameChanged(object sender, EventArgs e)
        {
            UpdateView();
        }

        public void UpdateView()
        {
            UpdateView(false);
        }

        public void UpdateView(bool isOnlyPos)
        {
            if (IsItemVisible())
            {
                this.Visibility = Visibility.Visible;

                double width = 0;
                foreach (var state in _states)
                {
                    state.SetPos(width);
                    state.SetHeight(CanvasHeight);
                    width += state.RectWidth;
                }
                this.Width = width;
                this.Height = CanvasHeight;

                Text.Visibility = (this.ActualWidth < 8) ? System.Windows.Visibility.Collapsed : System.Windows.Visibility.Visible;
            }
            else
            {
                this.Visibility = Visibility.Collapsed;
                return;
            }

            {
                double left = (_states.Count == 0) ? 0 : TimelineViewUtil.GetRenderPos(_states[0].StartFrame);
                Canvas.SetLeft(this, left);

                double top = 0;
                top += (_canvasHeight + _canvasMargin) * VerticalPos;
                Canvas.SetTop(this, top);

                if (left < 0)
                {
                    border.Width = this.Width - Math.Abs(left);
                    Canvas.SetLeft(border, Math.Abs(left));
                }
                else
                {
                    border.Width = this.Width;
                    Canvas.SetLeft(border, 0);
                }
            }
        }

        public bool IsItemVisible()
        {
            if (!IsFiltering())
            {
                return false;
            }

            return (StartFrame < TimelineViewUtil.ViewStartFrame && EndFrame > TimelineViewUtil.ViewStartFrame) ||
                (StartFrame >= TimelineViewUtil.ViewStartFrame && EndFrame < TimelineViewUtil.ViewEndFrame) ||
                (StartFrame < TimelineViewUtil.ViewEndFrame && EndFrame >= TimelineViewUtil.ViewEndFrame);
        }

        /// <summary>
        /// フィルタテキストにより表示されなくなる場合はfalseが返る
        /// </summary>
        /// <returns></returns>
        public bool IsFiltering()
        {
            if (string.IsNullOrEmpty(FilterText))
            {
                return true;
            }
            else
            {
                string filter = FilterText.ToLower();
                string label = Text.Text.ToLower();

                return label.Contains(filter);
            }
        }

        protected override void OnMouseDown(MouseButtonEventArgs e)
        {
            base.OnMouseDown(e);

            this.IsSelected = true;

            e.Handled = true;
        }

        private void _contextMenuItemCopyLabel_Click(object sender, RoutedEventArgs e)
        {
            var menuItem = (MenuItem)sender;
            var menu = (ContextMenu)menuItem.Parent;
            TimelineItem timelineItem = (TimelineItem)menu.PlacementTarget;
            ClipboardUtility.SetText(timelineItem.Text.Text);
        }
    }

    /// <summary>
    /// タイムラインアイテム上に並ぶステートイベント
    /// </summary>
    public class TimelineItemState : Image
    {
        /// <summary>
        /// ステートのタイプ
        /// </summary>
        public enum StateType
        {
            Prepare = 0,
            Playing,
            Pause,
        }

        private class TimelineStateStyle
        {
            private Brush _brush = null;

            public Brush Brush { get { return _brush; } }

            public TimelineStateStyle(Brush brush)
            {
                _brush = brush;
            }

            public void SetStyle(ref Rectangle rect)
            {
                rect.Fill = Brush;
                rect.RadiusX = 0;
                rect.RadiusY = 0;
            }
        }

        private static readonly TimelineStateStyle StylePrepare;
        private static readonly TimelineStateStyle StylePlaying;
        private static readonly TimelineStateStyle StylePause;

        static TimelineItemState()
        {
            StylePrepare = new TimelineStateStyle(
                new LinearGradientBrush(
                    new GradientStopCollection()
                    {
                        new GradientStop(Color.FromArgb(0xFF, 0xE9, 0xE9, 0xFF), 0.0),
                        new GradientStop(Color.FromArgb(0xFF, 0x00, 0x00, 0xAA), 1),
                        new GradientStop(Color.FromArgb(0xFF, 0x42, 0x42, 0xFF), 0.832),
                        new GradientStop(Color.FromArgb(0xFF, 0x8E, 0x8E, 0xFF), 0.139)
                    },
                    new Point(0.5, 0), new Point(0.5, 1)));

            StylePlaying = new TimelineStateStyle(
                new LinearGradientBrush(
                    new GradientStopCollection()
                    {
                        new GradientStop(Color.FromArgb(0xFF, 0xFF, 0xFF, 0x2B), 0.817),
                        new GradientStop(Colors.White, 0.0),
                        new GradientStop(Color.FromArgb(0xFF, 0xE0, 0xD7, 0x13), 1)
                    },
                    new Point(0.5, 0), new Point(0.5, 1)));

            StylePause = new TimelineStateStyle(
                new LinearGradientBrush(
                    new GradientStopCollection()
                    {
                        new GradientStop(Colors.White, 0.0),
                        new GradientStop(Color.FromArgb(0xFF, 0xBF, 0x00, 0x00), 1),
                        new GradientStop(Colors.Red, 0.875),
                        new GradientStop(Color.FromArgb(0xFF, 0xFF, 0x87, 0x87), 0.182)
                    },
                    new Point(0.5, 0), new Point(0.5, 1)));
        }

        public StateType Type
        {
            get { return (StateType)GetValue(TypeProperty); }
            set { SetValue(TypeProperty, value); }
        }

        public static readonly DependencyProperty TypeProperty =
            DependencyProperty.Register(nameof(Type), typeof(StateType), typeof(TimelineItemState), new PropertyMetadata(TypeChanged));

        private static void TypeChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            // タイプに合わせて色を変える
            var itemState = (TimelineItemState)d;
            switch (itemState.Type)
            {
                case StateType.Prepare:
                    itemState._geometryDrawing.Brush = StylePrepare.Brush;
                    break;
                case StateType.Playing:
                    itemState._geometryDrawing.Brush = StylePlaying.Brush;
                    break;
                case StateType.Pause:
                    itemState._geometryDrawing.Brush = StylePause.Brush;
                    break;
            }
        }

        /// <summary>
        /// イベント開始時刻
        /// </summary>
        public SpyTime StartTime
        {
            get { return (SpyTime)GetValue(StartTimeProperty); }
            set { SetValue(StartTimeProperty, value); }
        }

        public static readonly DependencyProperty StartTimeProperty =
            DependencyProperty.Register(nameof(StartTime), typeof(SpyTime), typeof(TimelineItemState), new PropertyMetadata(SpyTime.Zero, OnStartTimeChanged));

        private static void OnStartTimeChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var obj = (TimelineItemState)d;
            obj.StartFrame = obj.StartTime.SelectFrameValue(obj.TimeUnit);
        }

        /// <summary>
        /// イベント終了時刻
        /// </summary>
        public SpyTime EndTime
        {
            get { return (SpyTime)GetValue(EndTimeProperty); }
            set { SetValue(EndTimeProperty, value); }
        }

        public static readonly DependencyProperty EndTimeProperty =
            DependencyProperty.Register(nameof(EndTime), typeof(SpyTime), typeof(TimelineItemState), new PropertyMetadata(SpyTime.Zero, OnEndTimeChanged));

        private static void OnEndTimeChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var obj = (TimelineItemState)d;
            obj.EndFrame = obj.EndTime.SelectFrameValue(obj.TimeUnit);
        }

        /// <summary>
        /// 時刻の表示単位。
        /// 変更すると StartFrame, EndFrame の値が切り替わります。
        /// </summary>
        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set
            {
                if (_timeUnit != value)
                {
                    _timeUnit = value;

                    this.StartFrame = this.StartTime.SelectFrameValue(value);
                    this.EndFrame = this.EndTime.SelectFrameValue(value);
                }
            }
        }

        /// <summary>
        /// イベント開始フレーム
        /// </summary>
        public long StartFrame { get; set; }

        /// <summary>
        /// イベント終了フレーム
        /// </summary>
        public long EndFrame
        {
            get { return _endFrame; }
            set
            {
                if (_endFrame != value)
                {
                    _endFrame = value;

                    if (this.EndFrameChanged != null)
                    {
                        this.EndFrameChanged(this, null);
                    }
                }
            }
        }

        /// <summary>
        /// イベント経過フレーム
        /// </summary>
        public long PassFrame
        {
            get { return (StartFrame >= EndFrame) ? 0 : EndFrame - StartFrame; }
        }

        /// <summary>
        /// 矩形の幅
        /// </summary>
        public double RectWidth
        {
            get { return _rect.Width; }
        }

        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private long _endFrame;
        private Rect _rect = new Rect();
        private Size _size = new Size();
        private readonly GeometryDrawing _geometryDrawing;
        private readonly RectangleGeometry _rectangleGeometry;
        private readonly DrawingImage _drawingImage;
        public event EventHandler EndFrameChanged = null;

        public TimelineItemState(SpyTimeUnit timeUnit)
        {
            _timeUnit = timeUnit;
            _rectangleGeometry = new RectangleGeometry(_rect);
            _geometryDrawing = new GeometryDrawing()
            {
                Geometry = _rectangleGeometry,
            };

            _drawingImage = new DrawingImage(_geometryDrawing);
            this.Source = _drawingImage;

            DataContextChanged += (sender, e) =>
            {
                SetBinding(TypeProperty, new Binding("State") { Converter = StateTypeConverter, });
                SetBinding(StartTimeProperty, "StartTime");
                SetBinding(EndTimeProperty, "EndTime");
            };
        }

        public void SetPos(double posX)
        {
            UpdateWidth();

            if (this.Visibility == System.Windows.Visibility.Visible)
            {
                Canvas.SetTop(this, 0);
                Canvas.SetLeft(this, posX);
            }
        }

        private void UpdateWidth()
        {
            {
                var tmpWidth = TimelineViewUtil.GetRenderWidth(PassFrame);
                tmpWidth = tmpWidth == 0 ? 0.5 : tmpWidth;

                if (this.RectWidth != tmpWidth)
                {
                    _size.Width = tmpWidth;
                    _rect.Size = _size;
                }
            }

            if (!IsItemVisible())
            {
                this.Visibility = System.Windows.Visibility.Collapsed;
                return;
            }
            else
            {
                this.Visibility = System.Windows.Visibility.Visible;
            }

            if (this.ActualWidth != _rect.Width)
            //if (rectangleGeometry.Rect.Width != rect.Width)
            {
                _rectangleGeometry.Rect = _rect;
            }
        }

        public void SetHeight(double height)
        {
            _size.Height = height;
            _rect.Size = _size;
        }

        private bool IsItemVisible()
        {
            return (StartFrame < TimelineViewUtil.ViewStartFrame && EndFrame > TimelineViewUtil.ViewStartFrame) ||
                (StartFrame >= TimelineViewUtil.ViewStartFrame && EndFrame < TimelineViewUtil.ViewEndFrame) ||
                (StartFrame < TimelineViewUtil.ViewEndFrame && EndFrame >= TimelineViewUtil.ViewEndFrame);
        }

        /// <summary>
        /// StateTypeのコンバータ
        /// </summary>
        private class TypeConverter : IValueConverter
        {
            public object Convert(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
            {
                var type = (TimelineItemStateViewModel.StateType)value;
                switch (type)
                {
                    case TimelineItemStateViewModel.StateType.Preapre:
                        return TimelineItemState.StateType.Prepare;
                    case TimelineItemStateViewModel.StateType.Pause:
                        return TimelineItemState.StateType.Pause;
                    case TimelineItemStateViewModel.StateType.Playing:
                    default:
                        return TimelineItemState.StateType.Playing;
                }
            }

            public object ConvertBack(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
            {
                throw new NotImplementedException();
            }
        }

        private static TypeConverter StateTypeConverter = new TypeConverter();
    }
}
