﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.ComponentModel;
using System.Linq;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Interactivity;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    public class ListViewBehavior : Behavior<ListView>
    {
        public ListViewBehavior()
        {
            this.SortEnable = true;
        }

        protected override void OnAttached()
        {
            base.OnAttached();
            this.ApplyFilter();
            this.ApplySortEnable();
        }

        protected override void OnDetaching()
        {
            base.OnDetaching();
            this.ApplyFilter(true);
            this.ApplySortEnable(true);
        }

        #region Filter

        public static readonly DependencyProperty FilterProperty =
            DependencyProperty.Register(
            nameof(Filter),
            typeof(Predicate<object>),
            typeof(ListViewBehavior),
            new FrameworkPropertyMetadata((d, e) => ((ListViewBehavior)d).ApplyFilter()));

        /// <summary>
        /// リストアイテムをフィルタリングします。
        /// </summary>
        public Predicate<object> Filter
        {
            get { return (Predicate<object>)GetValue(FilterProperty); }
            set { SetValue(FilterProperty, value); }
        }

        private void ApplyFilter(bool detaching = false)
        {
            if (this.AssociatedObject == null)
            {
                return;
            }

            var listView = this.AssociatedObject;

            if (!detaching)
            {
                if (this.Filter != null)
                {
                    listView.Items.Filter = this.Filter;
                }
                else
                {
                    listView.Items.Filter = null;
                }

                listView.Items.Refresh();
            }
        }

        #endregion
        #region SortEnable

        public static readonly DependencyProperty SortEnableProperty =
            DependencyProperty.Register(
            nameof(SortEnable),
            typeof(bool),
            typeof(ListViewBehavior),
            new FrameworkPropertyMetadata((d, e) => ((ListViewBehavior)d).ApplySortEnable()));

        /// <summary>
        /// 行のソートを行うかを設定・取得します。
        /// </summary>
        public bool SortEnable
        {
            get { return (bool)GetValue(SortEnableProperty); }
            set { SetValue(SortEnableProperty, value); }
        }

        private void ApplySortEnable(bool detaching = false)
        {
            if (this.AssociatedObject == null)
            {
                return;
            }

            var listView = this.AssociatedObject;

            if (detaching || !this.SortEnable)
            {
                listView.RemoveHandler(GridViewColumnHeader.ClickEvent, (RoutedEventHandler)this.OnColumnHeaderClicked);
            }
            else
            {
                listView.AddHandler(GridViewColumnHeader.ClickEvent, (RoutedEventHandler)this.OnColumnHeaderClicked);
            }
        }

        private void OnColumnHeaderClicked(object sender, RoutedEventArgs e)
        {
            var columnHeader = e.OriginalSource as GridViewColumnHeader;
            if (columnHeader != null)
            {
                this.DoSort((ListView)e.Source, columnHeader.Column);
            }
        }

        private void DoSort(ListView listView, GridViewColumn column)
        {
            var sortDescriptions = listView.Items.SortDescriptions;
            var direction = ListSortDirection.Ascending;

            var propertyPath = this.ResolveSortExpression(column);
            if (propertyPath != null)
            {
                if (sortDescriptions.Count > 0)
                {
                    if (sortDescriptions[0].PropertyName == propertyPath)
                    {
                        direction = (GetSortDirection(column) == ListSortDirection.Ascending) ?
                            ListSortDirection.Descending : ListSortDirection.Ascending;
                    }
                    else
                    {
                        var gridView = (GridView)listView.View;
                        foreach (var c in gridView.Columns.Where(c => GetSortDirection(c) != null))
                        {
                            SetSortDirection(c, null);
                        }
                    }

                    sortDescriptions.Clear();
                }

                sortDescriptions.Add(new SortDescription(propertyPath, direction));
                SetSortDirection(column, direction);
            }
        }

        private string ResolveSortExpression(GridViewColumn column)
        {
            var columnBehavior = GridViewColumnBehavior.GetBehavior(column);

            if (columnBehavior != null)
            {
                if (!string.IsNullOrEmpty(columnBehavior.SortExpression))
                {
                    return columnBehavior.SortExpression;
                }
            }

            var binding = column.DisplayMemberBinding as Binding;
            if (binding != null)
            {
                return binding.Path.Path;
            }

            return null;
        }

        /// <summary>
        /// 現在の昇順・降順を記憶するために内部で使用します。
        /// </summary>
        private static readonly DependencyProperty SortDirectionProperty =
            DependencyProperty.RegisterAttached(
            "SortDirection",
            typeof(ListSortDirection?),
            typeof(GridViewColumnBehavior));

        public static ListSortDirection? GetSortDirection(GridViewColumn element)
        {
            return (ListSortDirection?)element.GetValue(SortDirectionProperty);
        }

        public static void SetSortDirection(GridViewColumn element, ListSortDirection? value)
        {
            element.SetValue(SortDirectionProperty, value);
        }

        #endregion
    }
}
