﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.NwSoundSpyPlugin.Models;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Threading;
using System.Windows.Data;
using System.Windows.Markup;

namespace NintendoWare.NwSoundSpyPlugin.Windows
{
    [ContentProperty("Items")]
    public class PlayingSoundPanelViewModel : ObservableObject
    {
        private const int InitialCollectionCount = 30;

        private readonly WeakReference<SoundStatusInfoSpyModel> _refModelStatus = new WeakReference<SoundStatusInfoSpyModel>(null);
        private readonly WeakReference<SoundDataInfoSpyModel> _refModelData = new WeakReference<SoundDataInfoSpyModel>(null);

        /// <summary>
        /// すべてのアイテム
        /// SetModel(SoundDataInfoSpyModel) が呼び出された際にアイテムの内容を更新するのに使います。
        /// </summary>
        private readonly List<PlayingSoundItemViewModel> _allItems = new List<PlayingSoundItemViewModel>();

        /// <summary>
        /// CurrentTimeのアイテム
        /// </summary>
        private readonly ObservableCollection<PlayingSoundItemViewModel> _collection = new ObservableCollection<PlayingSoundItemViewModel>();

        private readonly Manager _manager = new Manager();
        private Manager.Group _currentGroup;
        private SpyTime _currentTime = SpyTime.InvalidValue;

        private readonly StringFilter _nameFilter = new StringFilter();
        private string _nameFilterString = string.Empty;

        private bool _showAudible = true;
        private bool _showMute = true;

        private bool _showColumnPlayerName = true;
        private bool _showColumnVolume = true;
        private bool _showColumnPriority = true;
        private bool _showColumnPlayType = true;

        private int _playingSoundCount;

        public PlayingSoundPanelViewModel()
        {
        }

        protected override void DisposeManagedInstance()
        {
            _allItems.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public SynchronizationContext SyncContext
        {
            get;
            set;
        }

        private SoundStatusInfoSpyModel TargetModelStatus
        {
            get
            {
                return _refModelStatus.GetTarget();
            }
            set
            {
                _refModelStatus.SetTarget(value);
            }
        }

        private SoundDataInfoSpyModel TargetModelData
        {
            get
            {
                return _refModelData.GetTarget();
            }
            set
            {
                _refModelData.SetTarget(value);
            }
        }

        public ObservableCollection<PlayingSoundItemViewModel> Items
        {
            get
            {
                return _collection;
            }
        }

        public ICollectionView ViewItems
        {
            get { return CollectionViewSource.GetDefaultView(this.Items); }
        }

        public int PlayingSoundCount
        {
            get
            {
                return _playingSoundCount;
            }
            set
            {
                this.SetPropertyValue(ref _playingSoundCount, value);
            }
        }

        public string NameFilter
        {
            get
            {
                return _nameFilterString;
            }
            set
            {
                if (this.SetPropertyValue(ref _nameFilterString, value))
                {
                    _nameFilter.Filter = _nameFilterString;
                    this.UpdateItemsFiltered();
                }
            }
        }

        public bool ShowAudible
        {
            get
            {
                return _showAudible;
            }
            set
            {
                if (this.SetPropertyValue(ref _showAudible, value))
                {
                    this.UpdateItemsFiltered();
                }
            }
        }

        public bool ShowMute
        {
            get
            {
                return _showMute;
            }
            set
            {
                if (this.SetPropertyValue(ref _showMute, value))
                {
                    this.UpdateItemsFiltered();
                }
            }
        }

        public bool ShowColumnPlayerName
        {
            get
            {
                return _showColumnPlayerName;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPlayerName, value);
            }
        }

        public bool ShowColumnVolume
        {
            get
            {
                return _showColumnVolume;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnVolume, value);
            }
        }

        public bool ShowColumnPriority
        {
            get
            {
                return _showColumnPriority;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPriority, value);
            }
        }

        public bool ShowColumnPlayType
        {
            get
            {
                return _showColumnPlayType;
            }
            set
            {
                this.SetPropertyValue(ref _showColumnPlayType, value);
            }
        }

        public SpyTime CurrentTime
        {
            get
            {
                return _currentTime;
            }
            set
            {
                if (value == _currentTime)
                {
                    return;
                }
                _currentTime = value;

                UpdateItems();
            }
        }

        ///
        public void SetModel(SoundStatusInfoSpyModel model)
        {
            var lastModel = this.TargetModelStatus;

            // モデルが切り替わるときは一旦 null に設定される必要があります。
            Assertion.Operation.True(model == null || (lastModel == null && model != null));

            if (lastModel == model)
            {
                return;
            }

            this.TargetModelStatus = model;

            this.Clear();

            if (lastModel != null)
            {
                lastModel.SoundInfoChanged -= this.OnSoundInfosChanged;
            }

            if (model != null)
            {
                model.SoundInfoChanged += this.OnSoundInfosChanged;
                model.ReadAllSoundStatusInfos();
            }

            UpdateItems(true);
        }

        ///
        public void SetModel(SoundDataInfoSpyModel model)
        {
            var lastModel = this.TargetModelData;

            // モデルが切り替わるときは一旦 null に設定される必要があります。
            Assertion.Operation.True(model == null || (lastModel == null && model != null));

            this.TargetModelData = model;

            if (model != null && model != lastModel)
            {
                foreach (var item in _allItems)
                {
                    UpdateItemInfo(item);
                }
            }

            UpdateItems(true);
        }

        private void Clear()
        {
            _currentGroup = null;
            this.Items.Clear();
            _allItems.ForEach(it => it.Dispose());
            _allItems.Clear();
            _manager.Clear();
        }

        /// <summary>
        /// CurrentTime に応じて Items を更新します。
        /// </summary>
        private void UpdateItems(bool force = false)
        {
            if (!this.CurrentTime.Timestamp.IsValid)
            {
                return;
            }

            var group = _manager.Find(this.CurrentTime.Timestamp);

            if (force || _currentGroup != group)
            {
                _currentGroup = group;

                this.KillItems();
                if (group != null)
                {
                    foreach (var item in group.Items)
                    {
                        this.SetItem(item);
                    }

                    this.PlayingSoundCount = this.Items.Count(i => i.IsAlive == true);
                }
                else
                {
                    this.PlayingSoundCount = 0;
                }
            }
        }

        /// <summary>
        /// Items のすべての要素を未使用状態にします。
        /// </summary>
        private void KillItems()
        {
            foreach (var item in this.Items)
            {
                item.IsAlive = false;
            }
        }

        ///
        private void SetItem(PlayingSoundItemViewModel item)
        {
            var freeItem = this.Items
                .FirstOrDefault(i => i.IsAlive == false);

            if (freeItem == null)
            {
                freeItem = ExpandCollection();
            }

            freeItem.CopyFrom(item);
            freeItem.IsAlive = true;
            freeItem.IsFiltered = this.ItemFilter(freeItem);
        }

        ///
        private PlayingSoundItemViewModel ExpandCollection()
        {
            var item = new PlayingSoundItemViewModel();
            this.Items.Add(item);
            return item;
        }

        ///
        private void UpdateItemsFiltered()
        {
            foreach (var item in this.Items)
            {
                if (item.IsAlive)
                {
                    item.IsFiltered = this.ItemFilter(item);
                }
            }
        }

        ///
        private bool ItemFilter(object arg)
        {
            var item = arg as PlayingSoundItemViewModel;
            if (item == null)
            {
                return false;
            }

            bool match = true;

            // グループ間はAND, グループ内はOR

            if (match == true)
            {
                match = false;

                if (_nameFilter.Execute(item.SoundName))
                {
                    match = true;
                }
            }

            if (match == true)
            {
                match = false;

                if (_showAudible == true && item.IsAudible == true)
                {
                    match = true;
                }
                else if (_showMute == true && item.IsAudible == false)
                {
                    match = true;
                }
            }

            return match;
        }

        private void UpdateItemInfo(PlayingSoundItemViewModel item)
        {
            var modelData = this.TargetModelData;
            if (modelData != null)
            {
                var soundInfo = modelData.GetSoundInfo(item.SoundID);
                if (soundInfo != null)
                {
                    item.SoundName = soundInfo.Label ?? item.SoundID.ToString();
                    item.Priority = soundInfo.PlayerPriority;
                    item.Volume = soundInfo.Volume;

                    var playerInfo = modelData.GetPlayerInfo(soundInfo.PlayerId);
                    if (playerInfo != null)
                    {
                        item.PlayerName = playerInfo.Label ?? string.Empty;
                    }
                    else
                    {
                        item.PlayerName = soundInfo.PlayerId.ToString();
                    }
                }
            }
            else
            {
                item.SoundName = item.SoundID.ToString();
            }
        }

        private PlayingSoundItemViewModel.StateType GetStateType(SoundStatusInfoSpyModel.SoundInfo info)
        {
            if (info.StatusBitFlag == SoundStatusInfoSpyModel.SoundInfo.StatusPrepared)
            {
                return PlayingSoundItemViewModel.StateType.Preapre; // プリペア中だけ。
            }
            else if ((info.StatusBitFlag & SoundStatusInfoSpyModel.SoundInfo.StatusPause) > 0)
            {
                return PlayingSoundItemViewModel.StateType.Pause; // ポーズがかかってたらこれを優先
            }
            else
            {
                return PlayingSoundItemViewModel.StateType.Playing; // それ以外で生きてたらすべて再生中とみなす
            }
        }

        private PlayingSoundItemViewModel CreateItem(SoundStatusInfoSpyModel.SoundInfo info)
        {
            var item = new PlayingSoundItemViewModel()
            {
                InstanceID = info.InstanceId,
                SoundID = info.SoundId,
                State = GetStateType(info),
            };

            UpdateItemInfo(item);

            return item;
        }

        private bool CheckItemModify(PlayingSoundItemViewModel item, SoundStatusInfoSpyModel.SoundInfo info)
        {
            return item.SoundID != info.SoundId ||
                item.State != GetStateType(info);
        }

        ///
        private void OnSoundInfosChanged(object sender, SoundStatusInfoSpyModel.SoundInfoChangedEventArgs args)
        {
            //// infos は再生中のサウンドのコレクションです。
            //// (変化の通知ではなく、状態の通知)
            //// 追加、削除、変化したアイテムがあった場合だけgroupを追加します。

            var timestamp = args.Timestamp;

            Manager.Group playingGroup = _manager.PlayingGroup;
            if (playingGroup == null)
            {
                foreach (var info in args.SoundInfos)
                {
                    // 再生開始。
                    var item = CreateItem(info);
                    _manager.Add(timestamp, item);
                    _allItems.Add(item);
                }
            }
            else
            {
                List<PlayingSoundItemViewModel> removedItems = null;
                List<PlayingSoundItemViewModel> changedItems = null;
                foreach (var item in playingGroup.Items)
                {
                    var info = args.SoundInfos.FirstOrDefault(i => i.InstanceId == item.InstanceID);
                    if (info == null)
                    {
                        // 再生終了
                        if (removedItems == null)
                        {
                            removedItems = new List<PlayingSoundItemViewModel>();
                        }
                        removedItems.Add(item);
                    }
                    else if (CheckItemModify(item, info))
                    {
                        // パラメータが変化した。
                        if (changedItems == null)
                        {
                            changedItems = new List<PlayingSoundItemViewModel>();
                        }
                        var changedItem = CreateItem(info);
                        changedItems.Add(changedItem);
                    }
                }

                if (removedItems != null)
                {
                    foreach (var item in removedItems)
                    {
                        _manager.Remove(timestamp, item.InstanceID);
                    }
                }

                if (changedItems != null)
                {
                    foreach (var changedItem in changedItems)
                    {
                        _manager.Add(timestamp, changedItem);
                        _allItems.Add(changedItem);
                    }
                }

                foreach (var info in args.SoundInfos)
                {
                    if (!playingGroup.Contains(info.InstanceId))
                    {
                        // 再生開始。
                        var item = CreateItem(info);
                        _manager.Add(timestamp, item);
                        _allItems.Add(item);
                    }
                }
            }
        }

        ///
        public class Manager
        {
            private readonly List<Group> _groups = new List<Group>();
            private Group _playingGroup = null;

            ///
            public Group PlayingGroup
            {
                get { return _playingGroup; }
            }

            ///
            public void Clear()
            {
                _groups.Clear();
                _playingGroup = null;
            }

            ///
            public void Add(SpyGlobalTime position, PlayingSoundItemViewModel item)
            {
                if (_playingGroup != null && _playingGroup.Position == position)
                {
                    _playingGroup.Add(item);
                }
                else
                {
                    var group = new Group(position);

                    if (_playingGroup != null)
                    {
                        group.Items.AddRange(_playingGroup.Items);
                    }

                    group.Add(item);
                    _groups.Add(group);
                    _playingGroup = group;
                }
            }

            ///
            public void Remove(SpyGlobalTime position, uint instanceID)
            {
                if (_playingGroup != null && _playingGroup.Contains(instanceID))
                {
                    if (_playingGroup.Position == position)
                    {
                        _playingGroup.Remove(instanceID);
                    }
                    else
                    {
                        var group = new Group(position);
                        group.Items.AddRange(_playingGroup.Items);
                        group.Remove(instanceID);
                        _groups.Add(group);
                        _playingGroup = group;
                    }
                }
            }

            /// <summary>
            /// 指定したposition以前の最新のグループを返します。
            /// </summary>
            /// <param name="position"></param>
            /// <returns></returns>
            public Group Find(SpyGlobalTime position)
            {
                var index = BinarySearchUtility.BinarySearch(_groups, position, g => g.Position);

                index = Math.Max(index, ~index - 1);
                if (index >= 0)
                {
                    return _groups[index];
                }
                else
                {
                    return null;
                }
            }

            ///
            public class Group
            {
                private readonly List<PlayingSoundItemViewModel> _items = new List<PlayingSoundItemViewModel>();
                private readonly SpyGlobalTime _position;

                public Group(SpyGlobalTime position)
                {
                    _position = position;
                }

                public SpyGlobalTime Position
                {
                    get
                    {
                        return _position;
                    }
                }

                public List<PlayingSoundItemViewModel> Items
                {
                    get
                    {
                        return _items;
                    }
                }

                public bool Contains(uint instanceID)
                {
                    return _items.Any(i => i.InstanceID == instanceID);
                }

                public void Add(PlayingSoundItemViewModel item)
                {
                    var index = _items.FindIndex(i => i.InstanceID == item.InstanceID);
                    if (index >= 0)
                    {
                        _items[index] = item;
                    }
                    else
                    {
                        _items.Add(item);
                    }
                }

                public void Remove(uint instanceID)
                {
                    _items.RemoveAll(i => i.InstanceID == instanceID);
                }
            }
        }
    }
}
