﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Controls;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Markup;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    [ContentProperty("Items")]
    public class WaveformPanelViewModel : ObservableObject, WaveformCache.IWaveProvider
    {
        private const double MaxScaleX = 20; // これより上がると横スクロールできなくなる。

        private readonly WaveformCache _waveCache = new WaveformCache();
        private readonly ChannelCollection _items = new ChannelCollection();
        private FrameSyncSpyModel _frameSyncModel = null;
        private WaveformSpyModel _waveformModel = null;
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private double _currentX = 0.0;
        private double _minimumX = 0.0;
        private double _maximumX = 0.0;
        private double _originX = 0.0;
        private double _scaleX = 0.001;
        private double _scaleY = 1;

        public ObservableCollection<WaveformChannelViewModel> Items => _items;

        public FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSyncModel; }
            set { this.SetPropertyValue(ref _frameSyncModel, value); }
        }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        public double CurrentX
        {
            get { return _currentX; }
            set { this.SetCurrentXImpl(value, from: this); }
        }

        public double MinimumX
        {
            get { return _minimumX; }
            set
            {
                if (this.SetPropertyValue(ref _minimumX, value))
                {
                    this.Items.ForEach(it => it.MinimumX = value);
                }
            }
        }

        public double MaximumX
        {
            get { return _maximumX; }
            set
            {
                if (this.SetPropertyValue(ref _maximumX, value))
                {
                    this.Items.ForEach(it => it.MaximumX = value);
                }
            }
        }

        public double OriginX
        {
            get { return _originX; }
            set { this.SetOriginXImpl(value, from: this); }
        }

        public double ScaleX
        {
            get { return _scaleX; }
            set { this.SetScaleXImpl(value, from: this); }
        }

        public double ScaleY
        {
            get { return _scaleY; }
            set { this.SetScaleYImpl(value, from: this); }
        }

        int WaveformCache.IWaveProvider.WaveformCount
        {
            get
            {
                return _waveformModel?.WaveformCount ?? 0;
            }
        }

        public WaveformPanelViewModel()
        {
            _items.Owner = this;
            _waveCache.WaveProvider = this;
        }

        protected override void DisposeManagedInstance()
        {
            _items.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public void SetCurrentXFromPresenter(double value)
        {
            this.SetCurrentXImpl(value, from: null);
        }

        public void SetCurrentXFromChild(double value, WaveformChannelViewModel from)
        {
            this.SetCurrentXImpl(value, from);
        }

        private void SetCurrentXImpl(double value, object from)
        {
            if (from == this)
            {
                value = MathUtility.Clamp(
                    Math.Floor(value), this.MinimumX, this.MaximumX);
            }

            if (this.SetPropertyValue(ref _currentX, value, nameof(CurrentX)))
            {
                this.Items
                    .Where(it => it != from)
                    .ForEach(it => it.SetCurrentXFromOwner(value));
            }
        }

        public void SetOriginXFromChild(double value, WaveformChannelViewModel from)
        {
            this.SetOriginXImpl(value, from);
        }

        private void SetOriginXImpl(double value, object from)
        {
            if (this.SetPropertyValue(ref _originX, value, nameof(OriginX)))
            {
                this.Items
                    .Where(it => it != from)
                    .ForEach(it => it.SetOriginXFromOwner(value));
            }
        }

        public void SetScaleXFromChild(double value, WaveformChannelViewModel from)
        {
            this.SetScaleXImpl(value, from);
        }

        private void SetScaleXImpl(double value, object from)
        {
            if (from == this)
            {
                value = Math.Min(value, MaxScaleX);
            }

            if (this.SetPropertyValue(ref _scaleX, value, nameof(ScaleX)))
            {
                this.Items
                    .Where(it => it != from)
                    .ForEach(it => it.SetScaleXFromOwner(value));
            }
        }

        public void SetScaleYFromChild(double value, WaveformChannelViewModel from)
        {
            this.SetScaleYImpl(value, from);
        }

        private void SetScaleYImpl(double value, object from)
        {
            if (this.SetPropertyValue(ref _scaleY, value, nameof(ScaleY)))
            {
                this.Items
                    .Where(it => it != from)
                    .ForEach(it => it.SetScaleYFromOwner(value));
            }
        }

        internal void AttachModel(FrameSyncSpyModel model)
        {
            this.FrameSyncModel = model;
        }

        internal void AttachModel(WaveformSpyModel model)
        {
            if (_waveformModel == model)
            {
                return;
            }

            if (_waveformModel != null)
            {
                this.Items.ForEach(it => it.EnumerateLineChartSamples = EnumerateLineChartSamplesDummy);
                this.Items.ForEach(it => it.EvaluateLineChartValue = EvaluateLineChartValueDummy);

                this.Items.ForEach(it => it.Dispose());
                this.Items.Clear();

                _waveCache.ClearCache();

                _waveformModel.UpdateEvent -= this.OnWaveformUpdated;

                _waveCache.WaveProvider = null;
                _waveCache.PeakDataManager = null;
            }

            _waveformModel = model;

            if (_waveformModel != null)
            {
                this.Items.Add(new WaveformChannelViewModel() { Name = "F-L", ChannelNumber = 0, });
                this.Items.Add(new WaveformChannelViewModel() { Name = "F-R", ChannelNumber = 1, });
                this.Items.Add(new WaveformChannelViewModel() { Name = "R-L", ChannelNumber = 2, });
                this.Items.Add(new WaveformChannelViewModel() { Name = "R-R", ChannelNumber = 3, });
                this.Items.Add(new WaveformChannelViewModel() { Name = "F-C", ChannelNumber = 4, });
                this.Items.Add(new WaveformChannelViewModel() { Name = "LFE", ChannelNumber = 5, });

                this.Items.ForEach(it => it.EnumerateLineChartSamples = _waveCache.EnumerateSamples);
                this.Items.ForEach(it => it.EvaluateLineChartValue = _waveCache.Evaluate);

                _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainFrontLeft);
                _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainFrontRight);
                _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainRearLeft);
                _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainRearRight);
                _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainFrontCenter);
                _waveCache.ChannelMapping.Add(WaveformSpyModel.ChannelIndex.MainLfe);

                _waveformModel.UpdateEvent += this.OnWaveformUpdated;

                this.Items.ForEach(it =>
                {
                    it.CurrentX = this.CurrentX;
                    it.MinimumX = this.MinimumX;
                    it.MaximumX = this.MaximumX;
                    it.OriginX = this.OriginX;
                    it.ScaleX = this.ScaleX;
                    it.ScaleY = this.ScaleY;
                });

                _waveCache.SampleRate = _waveformModel.SampleRate;
                _waveCache.WaveProvider = this;
                _waveCache.PeakDataManager = _waveformModel.PeakDataManager;
            }
        }

        private void OnWaveformUpdated(object sender, EventArgs e)
        {
            _waveCache.SourceUpdated();
        }

        int WaveformCache.IWaveProvider.GetWaveformIndex(SpyGlobalTime time)
        {
            return _waveformModel?.GetWaveformIndex(time) ?? ~0;
        }

        WaveformSpyModel.Waveform WaveformCache.IWaveProvider.GetWaveform(int frameIndex)
        {
            return _waveformModel?.GetWaveform(frameIndex);
        }

        private IEnumerable<LineChartSample> EnumerateLineChartSamplesDummy(double start, double end, double scale, int sampleIndex)
        {
            yield break;
        }

        private double EvaluateLineChartValueDummy(double x, int channelIndex)
        {
            return double.NaN;
        }

        private class ChannelCollection : ObservableCollection<WaveformChannelViewModel>
        {
            public WaveformPanelViewModel Owner { get; set; }

            protected override void InsertItem(int index, WaveformChannelViewModel item)
            {
                if (item != null)
                {
                    item.Owner = this.Owner;
                }

                base.InsertItem(index, item);
            }

            protected override void SetItem(int index, WaveformChannelViewModel item)
            {
                if (item != null)
                {
                    item.Owner = this.Owner;
                }

                base.SetItem(index, item);
            }
        }
    }
}
