﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using NintendoWare.Spy;
using NintendoWare.Spy.Framework.Settings;
using NintendoWare.Spy.Plugins;
using NintendoWare.Spy.Settings;
using NintendoWare.Spy.Windows;
using System.Windows.Controls;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    public class WaveformPanelPresenter : SpyPanelPresenter
    {
        private readonly object _observerOwner = new object();
        private readonly object _settingsObserverOwner = new object();
        private WaveformPanelViewModel _viewModel = null;
        private WaveformPanel _view = null;

        public WaveformPanelPresenter()
        {
        }

        protected override void OnInitialize()
        {
            base.OnInitialize();

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Begin,
                    (target, args) => this.UpdateMinimumX());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.End,
                    (target, args) => this.UpdateMaximumX());

            PropertyChangedObservation.GetObserver(_observerOwner, this.GetPlaybackService())
                .AddHandler(
                    target => target.Current,
                    (target, args) => this.UpdateCurrentX());

            this.UpdateMinimumX();
            this.UpdateMaximumX();
            this.UpdateCurrentX();
        }

        protected override void OnUninitialize()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);
            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            Disposer.SafeDispose(ref _viewModel);

            base.OnUninitialize();
        }

        protected override Control CreateContent()
        {
            _viewModel = new WaveformPanelViewModel();

            _view = new WaveformPanel()
            {
                DataContext = _viewModel,
            };

            PropertyChangedObservation.GetObserver(_observerOwner, _viewModel)
                .AddHandler(
                    target => target.CurrentX,
                    (target, args) =>
                    {
                        this.GetPlaybackService().SetCurrentTime(SpyGlobalTime.FromMicroSeconds((long)target.CurrentX));
                    });

            return _view;
        }

        protected override void UpdateSpyModel(string dataName, SpyModel model)
        {
            switch (dataName)
            {
                case FrameSyncSpyModelPlugin.SpyDataName:
                    _viewModel.AttachModel((FrameSyncSpyModel)model);
                    break;

                case WaveformSpyModelPlugin.SpyDataName:
                    _viewModel.AttachModel((WaveformSpyModel)model);
                    break;

                default:
                    break;
            }
        }

        protected override void OnSettingsApplied(SettingsService service)
        {
            base.OnSettingsApplied(service);

            this.ApplySettings(service.GetApplicationSettings());

            PropertyChangedObservation.RemoveObservers(_settingsObserverOwner);

            PropertyChangedObservation.GetObserver(_settingsObserverOwner, service.GetApplicationSettings())
                .AddHandlerForAnyProperties(
                    (target, args) => this.ApplySettings(target));
        }

        private void ApplySettings(ApplicationSettings settings)
        {
            if (_viewModel != null)
            {
                _viewModel.TimeUnit = settings.TimeUnit;
            }
        }

        private void UpdateMinimumX()
        {
            if (_viewModel != null)
            {
                _viewModel.MinimumX = GetPlaybackService().Begin.GetMicroSeconds();
            }
        }

        private void UpdateMaximumX()
        {
            if (_viewModel != null)
            {
                _viewModel.MaximumX = GetPlaybackService().End.GetMicroSeconds();
            }
        }

        private void UpdateCurrentX()
        {
            if (_viewModel != null)
            {
                _viewModel.SetCurrentXFromPresenter(GetPlaybackService().Current.GetMicroSeconds());
            }
        }
    }
}
