﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy.Windows;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Controls.Primitives;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    /// <summary>
    /// XAML のコードビハインドを実装します。
    /// XAML デザイナのプレビューでは XAML 自身のコードビハインドが動作しないので分離して実装します。
    /// </summary>
    public class WaveformPanelControl : ContentControl
    {
        private static readonly Thickness ZeroThickness = new Thickness(0);
        private const string PartHeaderedListBox = "HeaderedListBox";
        private const string TemplatePartScrollViewer = "PART_ScrollViewer";
        private const string TemplatePartVerticalScrollBar = "PART_VerticalScrollBar";

        private static readonly DependencyPropertyKey MarginForPlotCanvasControllerPropertyKey =
            DependencyProperty.RegisterReadOnly(
            nameof(MarginForPlotCanvasController),
            typeof(Thickness),
            typeof(WaveformPanelControl),
            new FrameworkPropertyMetadata(ZeroThickness));

        public static readonly DependencyProperty MarginForPlotCanvasControllerProperty =
            MarginForPlotCanvasControllerPropertyKey.DependencyProperty;

        private static readonly DependencyPropertyKey MarginForHeaderedListBoxPropertyKey =
            DependencyProperty.RegisterReadOnly(
                nameof(MarginForHeaderedListBox),
                typeof(Thickness),
                typeof(WaveformPanelControl),
                new FrameworkPropertyMetadata(ZeroThickness));

        public static readonly DependencyProperty MarginForHeaderedListBoxProperty =
            MarginForHeaderedListBoxPropertyKey.DependencyProperty;

        private HeaderedListBox _headeredListBox;
        private ScrollViewer _scrollViewer;
        private ScrollBar _verticalScrollBar;

        public Thickness MarginForPlotCanvasController
        {
            get { return (Thickness)GetValue(MarginForPlotCanvasControllerProperty); }
            private set { SetValue(MarginForPlotCanvasControllerPropertyKey, value); }
        }

        public Thickness MarginForHeaderedListBox
        {
            get { return (Thickness)GetValue(MarginForHeaderedListBoxProperty); }
            private set { SetValue(MarginForHeaderedListBoxPropertyKey, value); }
        }

        public WaveformPanelControl()
        {
            this.Loaded += (o, e) => Initialize();
        }

        private void Initialize()
        {
            _headeredListBox = this.FindName(PartHeaderedListBox) as HeaderedListBox;

            this.ResolveTemplateElement(
                ref _scrollViewer,
                _headeredListBox,
                _headeredListBox?.Template,
                TemplatePartScrollViewer);

            this.ResolveTemplateElement(
                ref _verticalScrollBar,
                _scrollViewer,
                _scrollViewer?.Template,
                TemplatePartVerticalScrollBar);

            if (_headeredListBox != null && _verticalScrollBar != null)
            {
                _headeredListBox.ItemHeaderWidthChanged += (o, e) => UpdateMargins();
                _verticalScrollBar.SizeChanged += (o, e) => UpdateMargins();

                UpdateMargins();
            }
        }

        private void UpdateMargins()
        {
            var scrollBarWidth = _verticalScrollBar.ActualWidth;
            var headerWidth = _headeredListBox.ItemHeaderWidth;

            if (double.IsNaN(scrollBarWidth) || double.IsNaN(headerWidth))
            {
                return;
            }

            this.MarginForPlotCanvasController = new Thickness(headerWidth, 0.0, scrollBarWidth, 0.0);
            this.MarginForHeaderedListBox = new Thickness(-headerWidth, 18.0, -scrollBarWidth, 0.0);
        }

        private void ResolveTemplateElement<ElementType>(
            ref ElementType field,
            FrameworkElement templatedParent,
            FrameworkTemplate template,
            string name)
            where ElementType : class
        {
            Ensure.Argument.StringIsNotNullOrEmpty(name);

            if (field == null)
            {
                if (templatedParent == null || template == null)
                {
                    return;
                }

                field = template.FindName(name, templatedParent) as ElementType;
            }
        }
    }
}
