﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Controls;
using NintendoWare.Spy;
using NintendoWare.Spy.Extensions;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Media;
using System.Windows.Threading;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    internal class LoudnessMeterPanelViewModel : ObservableObject
    {
        public const double DisplayRangeMaximumY = 0;
        public const double DisplayRangeCenterY = -30;
        public const double DisplayRangeMinimumY = -60;
        internal const double MaxScaleX = 20; // これより上がると横スクロールできなくなる。
        internal const double FramesPerSecond = 1000000.0; // 1 [frame] : 1 [usec]
        internal const double SecondsPerFrame = 1 / FramesPerSecond;

        private const float FillChartTranslucency = 0.5f;

        private class ItemCollection<T> : ObservableCollection<T>
            where T : IDisposable
        {
            public Action<T> Setup { get; set; }

            public void Dispose()
            {
                this.ForEach(it => it.Dispose());
            }

            protected override void InsertItem(int index, T item)
            {
                if (item != null)
                {
                    Setup?.Invoke(item);
                }

                base.InsertItem(index, item);
            }

            protected override void SetItem(int index, T item)
            {
                if (item != null)
                {
                    Setup?.Invoke(item);
                }

                base.SetItem(index, item);
            }
        }

        private readonly object _observerOwner = new object();
        private FrameSyncSpyModel _frameSyncModel = null;
        private ILoudnessProvider _loudnessProvider = null;
        private readonly List<WaveformSpyModel.LoudnessInfo> _loudnessInfos = new List<WaveformSpyModel.LoudnessInfo>();
        private readonly MergedRequestDispatcher _requestUpdateLoudness = new MergedRequestDispatcher();
        private SpyTimeUnit _timeUnit = SpyTimeUnit.Timestamp;
        private double _currentX = 0.0;
        private double _minimumX = 0.0;
        private double _maximumX = 0.0;
        private double _originX = 0.0;
        private double _scaleX = LineChartScale.FromValue(160 / (FramesPerSecond * 2.0), doRound: true).ToValue(); // グリッド間隔が2秒程度になるように調整
        private double _scaleY = 1;
        private double _scaleThresholdForSampleDotDrawing = double.NaN;
        private bool _isLevel2chVisible = true;
        private bool _isLevel6chVisible = false;
        private bool _isLoudnessUnavailable = false;
        private bool _isLevelPeakSelected = true;
        private bool _isLevelRmsSelected = false;
        private bool _isTruePeakEnabled = true;

        internal delegate double ValueSelectFunc(WaveformSpyModel.LoudnessInfo loudnessInfo);

        public FrameSyncSpyModel FrameSyncModel
        {
            get { return _frameSyncModel; }
            set { this.SetPropertyValue(ref _frameSyncModel, value); }
        }

        public ObservableCollection<ILoudnessMeterChartItemViewModel> ChartItems { get; }

        /// <summary>
        /// 通常のチャート（ラウドネス、ピーク等）です。
        /// </summary>
        private ObservableCollection<LoudnessMeterChartItemLinearViewModel> LinearChartItems { get; }

        /// <summary>
        /// チャートが１つだけ表示されているときに、付加情報を背景として表示する補助チャートです。
        /// </summary>
        private ObservableCollection<LoudnessMeterChartItemFillViewModel> FillChartItems { get; }

        /// <summary>
        /// ラウドネスのチャートが１つだけ表示されているときに、付加情報を背景として表示する補助チャートです。
        /// </summary>
        private IList<LoudnessMeterChartItemFillViewModel> LoudnessFillChartItems { get; } =
            new List<LoudnessMeterChartItemFillViewModel>();

        /// <summary>
        /// TruePeak のチャートだけが表示されているときに、付加情報を背景として表示する補助チャートです。
        /// </summary>
        private IList<LoudnessMeterChartItemFillViewModel> TruePeakFillChartItems { get; } =
            new List<LoudnessMeterChartItemFillViewModel>();

        /// <summary>
        /// Peak のチャートだけが表示されているときに、付加情報を背景として表示する補助チャートです。
        /// </summary>
        private IList<LoudnessMeterChartItemFillViewModel> PeakFillChartItems { get; } =
            new List<LoudnessMeterChartItemFillViewModel>();

        public ObservableCollection<LoudnessMeterLevelItemViewModel> LevelItems { get; }

        private ObservableCollection<LoudnessMeterLevelItemViewModel> ChannelLevelItems { get; }

        public ObservableCollection<LoudnessMeterValueItemViewModel> ValueItems { get; }

        public LoudnessMeterChartItemLinearViewModel ChartMomentary { get; }

        public LoudnessMeterChartItemLinearViewModel ChartShortTerm { get; }

        public LoudnessMeterChartItemLinearViewModel ChartIntegrated { get; }

        public LoudnessMeterChartItemLinearViewModel ChartPeak { get; }

        public LoudnessMeterChartItemLinearViewModel ChartRms { get; }

        public LoudnessMeterLevelItemViewModel LevelMomentary { get; }

        public LoudnessMeterLevelItemViewModel LevelShortTerm { get; }

        public LoudnessMeterLevelItemViewModel LevelIntegrated { get; }

        public LoudnessMeterLevelItemViewModel LevelFrontLeft { get; }

        public LoudnessMeterLevelItemViewModel LevelFrontRight { get; }

        public LoudnessMeterLevelItemViewModel LevelRearLeft { get; }

        public LoudnessMeterLevelItemViewModel LevelRearRight { get; }

        public LoudnessMeterLevelItemViewModel LevelFrontCenter { get; }

        public LoudnessMeterLevelItemViewModel LevelLfe { get; }

        public SpyTimeUnit TimeUnit
        {
            get { return _timeUnit; }
            set { this.SetPropertyValue(ref _timeUnit, value); }
        }

        public double CurrentX
        {
            get { return _currentX; }
            set { this.SetCurrentXImpl(value, from: this); }
        }

        public double MinimumX
        {
            get { return _minimumX; }
            set
            {
                if (this.SetPropertyValue(ref _minimumX, value))
                {
                    this.ChartItems.ForEach(it => it.MinimumX = value);
                }
            }
        }

        public double MaximumX
        {
            get { return _maximumX; }
            set
            {
                if (this.SetPropertyValue(ref _maximumX, value))
                {
                    this.ChartItems.ForEach(it => it.MaximumX = value);
                }
            }
        }

        public double OriginX
        {
            get { return _originX; }
            set { this.SetPropertyValue(ref _originX, value); }
        }

        public double ScaleX
        {
            get { return _scaleX; }
            set
            {
                value = Math.Min(value, MaxScaleX);
                this.SetPropertyValue(ref _scaleX, value);
            }
        }

        public double ScaleY
        {
            get { return _scaleY; }
            set { this.SetPropertyValue(ref _scaleY, value); }
        }

        /// <summary>
        /// サンプル点を表す丸を描画するスケールの閾値です。
        /// (ラウドネスの計算されるフレーム区間 / 画面のピクセル数)
        /// </summary>
        public double ScaleThresholdForSampleDotDrawing
        {
            get { return _scaleThresholdForSampleDotDrawing; }
            set { this.SetPropertyValue(ref _scaleThresholdForSampleDotDrawing, value); }
        }

        public bool IsLevel2chVisible
        {
            get { return _isLevel2chVisible; }
            set
            {
                if (this.SetPropertyValue(ref _isLevel2chVisible, value))
                {
                    if (value)
                    {
                        this.IsLevel6chVisible = false;
                    }

                    this.NotifyPropertyChanged(nameof(IsLevelChannelOff));
                    this.UpdateLevelVisibility();
                }
            }
        }

        public bool IsLevel6chVisible
        {
            get { return _isLevel6chVisible; }
            set
            {
                if (this.SetPropertyValue(ref _isLevel6chVisible, value))
                {
                    if (value)
                    {
                        this.IsLevel2chVisible = false;
                    }

                    this.NotifyPropertyChanged(nameof(IsLevelChannelOff));
                    this.UpdateLevelVisibility();
                }
            }
        }

        public bool IsLevelChannelOff
        {
            get { return !_isLevel2chVisible && !_isLevel6chVisible; }
            set
            {
                if (value != this.IsLevelChannelOff)
                {
                    if (value)
                    {
                        this.IsLevel2chVisible = false;
                        this.IsLevel6chVisible = false;
                    }

                    this.NotifyPropertyChanged();
                    this.UpdateLevelVisibility();
                }
            }
        }

        public bool IsLevelPeakSelected
        {
            get { return _isLevelPeakSelected; }
            set
            {
                if (this.SetPropertyValue(ref _isLevelPeakSelected, value))
                {
                    this.IsLevelRmsSelected = !value;
                    this.UpdateChannelThresholds();
                    this.UpdateCurrentValue();
                    this.UpdateChannelLevelItemsLabelColor();
                }
            }
        }

        public bool IsLevelRmsSelected
        {
            get { return _isLevelRmsSelected; }
            set
            {
                if (this.SetPropertyValue(ref _isLevelRmsSelected, value))
                {
                    this.IsLevelPeakSelected = !value;
                    this.UpdateChannelThresholds();
                    this.UpdateCurrentValue();
                    this.UpdateChannelLevelItemsLabelColor();
                }
            }
        }

        public bool IsLoudnessUnavailable
        {
            get { return _isLoudnessUnavailable; }
            set { this.SetPropertyValue(ref _isLoudnessUnavailable, value); }
        }

        public bool IsTruePeakEnabled
        {
            get { return _isTruePeakEnabled; }
            set
            {
                if (this.SetPropertyValue(ref _isTruePeakEnabled, value))
                {
                    this.UpdateChannelThresholds();
                    this.UpdateCurrentValue();
                    this.ChartItems.ForEach(it => it.NotifySamplesChanged());
                }
            }
        }

        internal ILoudnessProvider LoudnessProvider => _loudnessProvider;

        internal IList<WaveformSpyModel.LoudnessInfo> LoudnessInfos => _loudnessInfos;

        private LoudnessMeterThreshold[] ChannelThresholdsTruePeak { get; }

        private LoudnessMeterThreshold[] ChannelThresholdsPeak { get; }

        private LoudnessMeterThreshold[] ChannelThresholdsRms { get; }

        public LoudnessMeterPanelViewModel()
        {
            this.ChartItems = new ItemCollection<ILoudnessMeterChartItemViewModel>()
            {
                Setup = (item) =>
                {
                    item.Owner = this;
                    item.MaximumX = this.MaximumX;
                    item.MinimumX = this.MinimumX;
                },
            };

            this.LinearChartItems = new ItemCollection<LoudnessMeterChartItemLinearViewModel>()
            {
                Setup = (item) =>
                {
                    item.IsVisibleChanged += (sender, args) => this.UpdateFillChart();
                },
            };

            this.FillChartItems = new ItemCollection<LoudnessMeterChartItemFillViewModel>()
            {
                Setup = (item) =>
                {
                    item.IsVisible = false;
                },
            };

            this.LevelItems = new ItemCollection<LoudnessMeterLevelItemViewModel>()
            {
                Setup = (item) =>
                {
                    item.Owner = this;
                },
            };

            this.ChannelLevelItems = new ItemCollection<LoudnessMeterLevelItemViewModel>();

            this.ValueItems = new ItemCollection<LoudnessMeterValueItemViewModel>()
            {
                Setup = (item) =>
                {
                    item.Owner = this;
                },
            };

            ValueSelectFunc valueMomentary = info => info.MomentaryLoudnessValue;
            ValueSelectFunc valueShortTerm = info => info.ShortTermLoudnessValue;
            ValueSelectFunc valueIntegrated = info => info.RelGatedLoudnessValue;
            ValueSelectFunc valuePeak = info => info.Channels.Max(c => this.IsTruePeakEnabled ? c.TruePeakValue : c.PeakValue);
            ValueSelectFunc valueRms = info => info.Channels.Max(c => c.RmsValue);

            var redBrush = new SolidColorBrush(Color.FromRgb(0xfb, 0x77, 0x77));
            var yellowBrush = new SolidColorBrush(Color.FromRgb(0xff, 0xc1, 0x00));
            var greenBrush = new SolidColorBrush(Color.FromRgb(0x91, 0xcf, 0x4f));
            var blueBrush = new SolidColorBrush(Color.FromRgb(0x5b, 0x9b, 0xd5));

            redBrush.Freeze();
            yellowBrush.Freeze();
            greenBrush.Freeze();
            blueBrush.Freeze();

            var loudnessThresholds = new[]
            {
                new LoudnessMeterThreshold()
                {
                    Threshold = DisplayRangeMaximumY,
                    Fill = redBrush,
                },
                new LoudnessMeterThreshold()
                {
                    Threshold = -22,
                    Fill = yellowBrush,
                },
                new LoudnessMeterThreshold()
                {
                    Threshold = -24,
                    Fill = greenBrush,
                },
                new LoudnessMeterThreshold()
                {
                    Threshold = -28,
                    Fill = blueBrush,
                },
            };

            this.ChannelThresholdsTruePeak = new[]
            {
                new LoudnessMeterThreshold()
                {
                    Threshold = DisplayRangeMaximumY,
                    Fill = redBrush,
                },
                new LoudnessMeterThreshold()
                {
                    Threshold = -1,
                    Fill = blueBrush,
                },
            };

            this.ChannelThresholdsPeak = new[]
            {
                new LoudnessMeterThreshold()
                {
                    Threshold = DisplayRangeMaximumY,
                    Fill = redBrush,
                },
                new LoudnessMeterThreshold()
                {
                    Threshold = -3,
                    Fill = blueBrush,
                },
            };

            this.ChannelThresholdsRms = new[]
            {
                new LoudnessMeterThreshold()
                {
                    Threshold = DisplayRangeMaximumY,
                    Fill = blueBrush,
                },
            };

            // チャート・アイテムを登録します。
            this.ChartMomentary = new LoudnessMeterChartItemLinearViewModel()
            {
                Name = "MMT",
                Color = Colors.Tomato,
                ValueSelectFunc = valueMomentary,
                IsFillEnabled = true,
                FillChartItems = this.LoudnessFillChartItems,
            };

            this.ChartShortTerm = new LoudnessMeterChartItemLinearViewModel()
            {
                Name = "SHT",
                Color = Colors.DeepSkyBlue,
                ValueSelectFunc = valueShortTerm,
                IsFillEnabled = true,
                FillChartItems = this.LoudnessFillChartItems,
            };

            this.ChartIntegrated = new LoudnessMeterChartItemLinearViewModel()
            {
                Name = "INT",
                Color = Colors.Green,
                ValueSelectFunc = valueIntegrated,
                IsFillEnabled = true,
                FillChartItems = this.LoudnessFillChartItems,
            };

            this.ChartPeak = new LoudnessMeterChartItemLinearViewModel()
            {
                Name = "Peak",
                Color = Colors.Yellow,
                ValueSelectFunc = valuePeak,
                IsFillEnabled = true,
                FillChartItems = this.TruePeakFillChartItems, // 動的に切り替わります
            };

            this.ChartRms = new LoudnessMeterChartItemLinearViewModel()
            {
                Name = "RMS",
                Color = Colors.GreenYellow,
                ValueSelectFunc = valueRms,
                IsFillEnabled = false,
            };

            this.LinearChartItems.Add(this.ChartMomentary);
            this.LinearChartItems.Add(this.ChartShortTerm);
            this.LinearChartItems.Add(this.ChartIntegrated);
            this.LinearChartItems.Add(this.ChartPeak);
            this.LinearChartItems.Add(this.ChartRms);

            Func<SolidColorBrush, Color> toTranslucentColor = (brush) =>
            {
                var color = brush.Color;
                return Color.FromScRgb(color.ScA * FillChartTranslucency, color.ScR, color.ScG, color.ScB);
            };

            loudnessThresholds
                .Concat(new[] { new LoudnessMeterThreshold() { Threshold = DisplayRangeMinimumY } })
                .SuccessingPair()
                .ForEach((it, i) =>
                {
                    var fillChart = new LoudnessMeterChartItemFillViewModel()
                    {
                        Name = $"LoudnessFillChart{i}",
                        Color = toTranslucentColor(it.Item1.Fill),
                        LowerThreshold = it.Item2.Threshold,
                        UpperThreshold = it.Item1.Threshold,
                    };

                    this.LoudnessFillChartItems.Add(fillChart);
                    this.FillChartItems.Add(fillChart);
                });

            this.ChannelThresholdsTruePeak
                .Concat(new[] { new LoudnessMeterThreshold() { Threshold = DisplayRangeMinimumY } })
                .SuccessingPair()
                .ForEach((it, i) =>
                {
                    var fillChart = new LoudnessMeterChartItemFillViewModel()
                    {
                        Name = $"TruePeakFillChart{i}",
                        Color = toTranslucentColor(it.Item1.Fill),
                        LowerThreshold = it.Item2.Threshold,
                        UpperThreshold = it.Item1.Threshold,
                    };

                    this.TruePeakFillChartItems.Add(fillChart);
                    this.FillChartItems.Add(fillChart);
                });

            this.ChannelThresholdsPeak
                .Concat(new[] { new LoudnessMeterThreshold() { Threshold = DisplayRangeMinimumY } })
                .SuccessingPair()
                .ForEach((it, i) =>
                {
                    var fillChart = new LoudnessMeterChartItemFillViewModel()
                    {
                        Name = $"PeakFillChart{i}",
                        Color = toTranslucentColor(it.Item1.Fill),
                        LowerThreshold = it.Item2.Threshold,
                        UpperThreshold = it.Item1.Threshold,
                    };

                    this.PeakFillChartItems.Add(fillChart);
                    this.FillChartItems.Add(fillChart);
                });

            // FillChartItems が LinearChartItems の後ろに見えるようにするため、
            // ChartItems には FillChartItems を先に登録します。
            this.FillChartItems.ForEach(item => this.ChartItems.Add(item));
            this.LinearChartItems.ForEach(item => this.ChartItems.Add(item));

            // レベル・アイテムを登録します。
            this.LevelMomentary = new LoudnessMeterLevelItemViewModel()
            {
                Name = "MMT",
                ValueSelectFunc = valueMomentary,
                Thresholds = loudnessThresholds,
                ChartColor = this.ChartMomentary.Color,
            };

            this.LevelShortTerm = new LoudnessMeterLevelItemViewModel()
            {
                Name = "SHT",
                ValueSelectFunc = valueShortTerm,
                Thresholds = loudnessThresholds,
                ChartColor = this.ChartShortTerm.Color,
            };

            this.LevelIntegrated = new LoudnessMeterLevelItemViewModel()
            {
                Name = "INT",
                ValueSelectFunc = valueIntegrated,
                Thresholds = loudnessThresholds,
                ChartColor = this.ChartIntegrated.Color,
            };

            this.LevelFrontLeft = new LoudnessMeterLevelItemViewModel()
            {
                Name = "L",
                ValueSelectFunc = info => this.SelectChannelLevelValue(info.Channels[(int)WaveformSpyModel.ChannelIndex.MainFrontLeft]),
                Thresholds = this.ChannelThresholdsTruePeak, // 動的に切り替わります
            };

            this.LevelFrontRight = new LoudnessMeterLevelItemViewModel()
            {
                Name = "R",
                ValueSelectFunc = info => this.SelectChannelLevelValue(info.Channels[(int)WaveformSpyModel.ChannelIndex.MainFrontRight]),
                Thresholds = this.ChannelThresholdsTruePeak, // 動動的に切り替わります
            };

            this.LevelRearLeft = new LoudnessMeterLevelItemViewModel()
            {
                Name = "RL",
                IsVisible = false,
                ValueSelectFunc = info => this.SelectChannelLevelValue(info.Channels[(int)WaveformSpyModel.ChannelIndex.MainRearLeft]),
                Thresholds = this.ChannelThresholdsTruePeak, // 動的に切り替わります
            };

            this.LevelRearRight = new LoudnessMeterLevelItemViewModel()
            {
                Name = "RR",
                IsVisible = false,
                ValueSelectFunc = info => this.SelectChannelLevelValue(info.Channels[(int)WaveformSpyModel.ChannelIndex.MainRearRight]),
                Thresholds = this.ChannelThresholdsTruePeak, // 動的に切り替わります
            };

            this.LevelFrontCenter = new LoudnessMeterLevelItemViewModel()
            {
                Name = "FC",
                IsVisible = false,
                ValueSelectFunc = info => this.SelectChannelLevelValue(info.Channels[(int)WaveformSpyModel.ChannelIndex.MainFrontCenter]),
                Thresholds = this.ChannelThresholdsTruePeak, // 動的に切り替わります
            };

            this.LevelLfe = new LoudnessMeterLevelItemViewModel()
            {
                Name = "LFE",
                IsVisible = false,
                ValueSelectFunc = info => this.SelectChannelLevelValue(info.Channels[(int)WaveformSpyModel.ChannelIndex.MainLfe]),
                Thresholds = this.ChannelThresholdsTruePeak, // 動的に切り替わります
            };

            this.LevelItems.Add(this.LevelMomentary);
            this.LevelItems.Add(this.LevelShortTerm);
            this.LevelItems.Add(this.LevelIntegrated);

            this.ChannelLevelItems.Add(this.LevelFrontLeft);
            this.ChannelLevelItems.Add(this.LevelFrontRight);
            this.ChannelLevelItems.Add(this.LevelRearLeft);
            this.ChannelLevelItems.Add(this.LevelRearRight);
            this.ChannelLevelItems.Add(this.LevelFrontCenter);
            this.ChannelLevelItems.Add(this.LevelLfe);

            this.ChannelLevelItems.ForEach(it => this.LevelItems.Add(it));

            // バリュー・アイテムを登録します。
            this.ValueItems.Add(new LoudnessMeterValueItemViewModel()
            {
                Name = "Momentary(LKFS)",
                ValueSelectFunc = valueMomentary,
                Color = this.ChartMomentary.Color,
                Thresholds = loudnessThresholds,
            });

            this.ValueItems.Add(new LoudnessMeterValueItemViewModel()
            {
                Name = "ShortTerm(LKFS)",
                ValueSelectFunc = valueShortTerm,
                Color = this.ChartShortTerm.Color,
                Thresholds = loudnessThresholds,
            });

            this.ValueItems.Add(new LoudnessMeterValueItemViewModel()
            {
                Name = "Integrated(LKFS)",
                ValueSelectFunc = valueIntegrated,
                Color = this.ChartIntegrated.Color,
                Thresholds = loudnessThresholds,
            });

            this.InvalidateCurrentValue();
            this.UpdateLevelVisibility();
            this.UpdateChannelLevelItemsLabelColor();
        }

        protected override void DisposeManagedInstance()
        {
            PropertyChangedObservation.RemoveObservers(_observerOwner);

            this.ChartItems.ForEach(it => it.Dispose());
            this.LevelItems.ForEach(it => it.Dispose());
            this.ValueItems.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public void SetCurrentXFromPresenter(double value)
        {
            this.SetCurrentXImpl(value, from: null);
        }

        private void SetCurrentXImpl(double value, object from)
        {
            if (from == this)
            {
                value = Spy.MathUtility.Clamp(
                    Math.Floor(value), this.MinimumX, this.MaximumX);
            }

            if (this.SetPropertyValue(ref _currentX, value, nameof(CurrentX)))
            {
                this.UpdateCurrentValue();
            }
        }

        private float SelectChannelLevelValue(WaveformSpyModel.ChannelLoudnessInfo cli)
        {
            if (this.IsLevelPeakSelected)
            {
                return this.IsTruePeakEnabled ? cli.TruePeakValue : cli.PeakValue;
            }
            else if (this.IsLevelRmsSelected)
            {
                return cli.RmsValue;
            }
            else
            {
                throw new NotImplementedException();
            }
        }

        internal void AttachModel(FrameSyncSpyModel model)
        {
            this.FrameSyncModel = model;
        }

        internal void AttachModel(ILoudnessProvider model)
        {
            if (_loudnessProvider == model)
            {
                return;
            }

            if (_loudnessProvider != null)
            {
                _loudnessInfos.Clear();

                PropertyChangedObservation.RemoveObservers(_observerOwner);

                _loudnessProvider.LoudnessUpdateEvent -= this.OnLoudnessUpdated;

                this.InvalidateCurrentValue();

                this.IsLoudnessUnavailable = false;
            }

            _loudnessProvider = model;

            if (_loudnessProvider != null)
            {
                _loudnessProvider.LoudnessUpdateEvent += this.OnLoudnessUpdated;

                {
                    var observer = PropertyChangedObservation.GetObserver(_observerOwner, _loudnessProvider);

                    observer.AddHandler(
                        target => target.IsWaveformMetadataReady,
                        (target, args) =>
                        {
                            this.UpdateLoudnessUnavailable();
                        });

                    this.UpdateLoudnessUnavailable();
                }

                this.UpdateLoudness();

                this.ChartItems.ForEach(it =>
                {
                    it.MinimumX = this.MinimumX;
                    it.MaximumX = this.MaximumX;
                });
            }
        }

        private void UpdateLoudnessUnavailable()
        {
            if (_loudnessProvider.IsWaveformMetadataReady)
            {
                if (_loudnessProvider.IsLoudnessAvailable)
                {
                    this.IsLoudnessUnavailable = false;
                    this.ScaleThresholdForSampleDotDrawing = (FramesPerSecond * _loudnessProvider.LoudnessStepSampleCount / _loudnessProvider.SampleRate) / 10;
                }
                else
                {
                    this.IsLoudnessUnavailable = true;
                    this.ScaleThresholdForSampleDotDrawing = double.NaN;
                }
            }
        }

        private void OnLoudnessUpdated(object sender, EventArgs e)
        {
            this.RequestUpdateLoudness();
        }

        private void RequestUpdateLoudness()
        {
            _requestUpdateLoudness.Request(this.UpdateLoudness, DispatcherPriority.Background);
        }

        private void UpdateLoudness()
        {
            if (_loudnessProvider == null)
            {
                return;
            }

            _loudnessProvider.UpdateLoudness();

            if (_loudnessProvider.LoudnessCount > _loudnessInfos.Count)
            {
                var currentSampleIndex = (this.CurrentX * SecondsPerFrame - _loudnessProvider.BeginTime.Timestamp.Seconds) * _loudnessProvider.SampleRate;

                while (_loudnessProvider.LoudnessCount > _loudnessInfos.Count)
                {
                    var loudnessInfo = _loudnessProvider.GetLoudness(_loudnessInfos.Count);
                    _loudnessInfos.Add(loudnessInfo);

                    // カレントフレームのラウドネス情報が更新された場合。
                    if (loudnessInfo.SampleIndex <= currentSampleIndex && currentSampleIndex <= loudnessInfo.SampleIndex + _loudnessProvider.LoudnessStepSampleCount)
                    {
                        this.UpdateCurrentValue();
                    }
                }

                this.ChartItems.ForEach(it => it.NotifySamplesChanged());
            }
        }

        private void UpdateLevelVisibility()
        {
            if (this.IsLevel2chVisible)
            {
                this.LevelFrontLeft.IsVisible = true;
                this.LevelFrontRight.IsVisible = true;
                this.LevelRearLeft.IsVisible = false;
                this.LevelRearRight.IsVisible = false;
                this.LevelFrontCenter.IsVisible = false;
                this.LevelLfe.IsVisible = false;
            }
            else if (this.IsLevel6chVisible)
            {
                this.LevelFrontLeft.IsVisible = true;
                this.LevelFrontRight.IsVisible = true;
                this.LevelRearLeft.IsVisible = true;
                this.LevelRearRight.IsVisible = true;
                this.LevelFrontCenter.IsVisible = true;
                this.LevelLfe.IsVisible = true;
            }
            else
            {
                this.LevelFrontLeft.IsVisible = false;
                this.LevelFrontRight.IsVisible = false;
                this.LevelRearLeft.IsVisible = false;
                this.LevelRearRight.IsVisible = false;
                this.LevelFrontCenter.IsVisible = false;
                this.LevelLfe.IsVisible = false;
            }
        }

        private void UpdateChannelLevelItemsLabelColor()
        {
            if (this.IsLevelPeakSelected)
            {
                this.ChannelLevelItems.ForEach(it => it.ChartColor = this.ChartPeak.Color);
            }
            else if (this.IsLevelRmsSelected)
            {
                this.ChannelLevelItems.ForEach(it => it.ChartColor = this.ChartRms.Color);
            }
        }

        private void UpdateChannelThresholds()
        {
            // レベル表示の閾値を切り替えます。
            if (this.IsLevelPeakSelected)
            {
                if (this.IsTruePeakEnabled)
                {
                    this.ChannelLevelItems.ForEach(it => it.Thresholds = this.ChannelThresholdsTruePeak);
                }
                else
                {
                    this.ChannelLevelItems.ForEach(it => it.Thresholds = this.ChannelThresholdsPeak);
                }
            }
            else if (this.IsLevelRmsSelected)
            {
                this.ChannelLevelItems.ForEach(it => it.Thresholds = this.ChannelThresholdsRms);
            }

            // チャートの塗りつぶし背景を切り替えます。
            {
                var fillChartItems = this.IsTruePeakEnabled ? this.TruePeakFillChartItems : this.PeakFillChartItems;
                if (this.ChartPeak.FillChartItems != fillChartItems)
                {
                    this.ChartPeak.FillChartItems.ForEach(it => it.IsVisible = false);
                    this.ChartPeak.FillChartItems = fillChartItems;
                    this.UpdateFillChart();
                }
            }
        }

        private void InvalidateCurrentValue()
        {
            foreach (var item in this.LinearChartItems)
            {
                item.InvalidateCurrent();
            }

            foreach (var item in this.LevelItems)
            {
                item.Value = double.NegativeInfinity;
                item.Color = LoudnessMeterLevelItemViewModel.DefaultColor;
            }

            foreach (var item in this.ValueItems)
            {
                item.Value = double.NegativeInfinity;
                item.ValueBrush = LoudnessMeterValueItemViewModel.DefaultValueBrush;
            }
        }

        private void UpdateCurrentValue()
        {
            if (_loudnessProvider == null ||
                !_loudnessProvider.IsWaveformMetadataReady ||
                _loudnessProvider.LoudnessCount == 0)
            {
                return;
            }

            double seconds = SecondsPerFrame * this.CurrentX;

            if (seconds < _loudnessProvider.BeginTime.Timestamp.Seconds)
            {
                this.InvalidateCurrentValue();
                return;
            }

            long sampleIndex = (long)((seconds - _loudnessProvider.BeginTime.Timestamp.Seconds) * _loudnessProvider.SampleRate);

            var index = BinarySearchUtility.BinarySearch(_loudnessInfos, sampleIndex, i => i.SampleIndex);
            if (index < 0)
            {
                index = ~index - 1;
            }

            if (index < 0)
            {
                this.InvalidateCurrentValue();
                return;
            }

            var info = _loudnessInfos[index];
            foreach (var item in this.LinearChartItems)
            {
                item.UpdateCurrent(info);
            }

            foreach (var item in this.LevelItems)
            {
                item.Value = item.ValueSelectFunc(info);
                item.Color = this.ChooseThresholdValue(
                    Spy.MathUtility.Clamp(item.Value, item.Minimum, item.Maximum),
                    item.Thresholds,
                    t => t.Fill,
                    LoudnessMeterLevelItemViewModel.DefaultColor);
            }

            foreach (var item in this.ValueItems)
            {
                item.Value = item.ValueSelectFunc(info);
                item.ValueBrush = this.ChooseThresholdValue(
                        Spy.MathUtility.Clamp(item.Value, DisplayRangeMinimumY, DisplayRangeMaximumY),
                        item.Thresholds,
                        t => t.Fill,
                        LoudnessMeterValueItemViewModel.DefaultValueBrush);
            }
        }

        private T ChooseThresholdValue<T>(
            double value,
            IEnumerable<LoudnessMeterThreshold> thresholds,
            Func<LoudnessMeterThreshold, T> valueSelector,
            T defaultValue)
        {
            var thr = thresholds
                ?.Where(it => it.Threshold >= value)
                .MinBy(it => it.Threshold)
                .FirstOrDefault();

            return (thr != null) ? valueSelector(thr) : defaultValue;
        }

        private void UpdateFillChart()
        {
            var visibleChartCount = this.LinearChartItems.Count(it => it.IsVisible);
            if (visibleChartCount == 1)
            {
                var chartItem = this.LinearChartItems.First(it => it.IsVisible);

                if (chartItem.IsFillEnabled && chartItem.FillChartItems != null)
                {
                    chartItem.FillChartItems
                        .ForEach(it =>
                        {
                            it.ValueSelectFunc = chartItem.ValueSelectFunc;
                            it.IsVisible = true;
                            it.NotifySamplesChanged();
                        });

                    return;
                }
            }

            this.FillChartItems
                .ForEach(it =>
                {
                    it.IsVisible = false;
                });
        }
    }
}
