﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Windows.Controls;
using Nintendo.ToolFoundation.Windows.Primitives;
using NintendoWare.Spy;
using System;
using System.Collections.Generic;
using System.Windows.Media;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    /// <summary>
    /// 線状のチャートアイテムが１つだけ表示されているときに、チャートの背景に閾値に応じた色を表示するためのチャート・アイテムです。
    /// </summary>
    internal class LoudnessMeterChartItemFillViewModel : ObservableObject, ILoudnessMeterChartItemViewModel, ILineChartItemData
    {
        private double _minimumX = 0.0;
        private double _maximumX = 0.0;
        private bool _visible = true;
        private Color _color = Colors.WhiteSmoke;

        public event EventHandler SamplesChanged;

        public LoudnessMeterPanelViewModel Owner { get; set; }

        public string Name { get; set; } = string.Empty;

        public bool IsVisible
        {
            get { return _visible; }
            set { this.SetPropertyValue(ref _visible, value); }
        }

        LineChartDrawType ILineChartItemData.DrawType => LineChartDrawType.Line;

        public Color Color
        {
            get { return _color; }
            set { this.SetPropertyValue(ref _color, value); }
        }

        LineChartInterpolation ILineChartItemData.Interpolation => LineChartInterpolation.Linear;

        public double MaximumX
        {
            get { return _maximumX; }
            set
            {
                if (this.SetPropertyValue(ref _maximumX, value))
                {
                    this.NotifySamplesChanged();
                }
            }
        }

        public double MinimumX
        {
            get { return _minimumX; }
            set
            {
                if (this.SetPropertyValue(ref _minimumX, value))
                {
                    this.NotifySamplesChanged();
                }
            }
        }

        public double MinimumY => LoudnessMeterPanelViewModel.DisplayRangeMinimumY;

        public double MaximumY => LoudnessMeterPanelViewModel.DisplayRangeMaximumY;

        public double CenterY => LoudnessMeterPanelViewModel.DisplayRangeCenterY;

        public LoudnessMeterPanelViewModel.ValueSelectFunc ValueSelectFunc { get; set; }

        public double UpperThreshold { get; set; }

        public double LowerThreshold { get; set; }

        public LoudnessMeterChartItemFillViewModel()
        {
        }

        public void NotifySamplesChanged()
        {
            this.SamplesChanged?.Invoke(this, EventArgs.Empty);
            // LineChart が SampleChanged イベントを使っていないので、その代わり。
            this.NotifyPropertyChanged(nameof(MaximumX));
        }

        IEnumerable<LineChartSample> ILineChartItemData.EnumerateSamples(double beginX, double endX, double scale)
        {
            return this.EnumerateLineChartSamplesWithThreshold(beginX, endX, scale, this.LowerThreshold, this.UpperThreshold);
        }

        double ILineChartItemData.Evaluate(double x)
        {
            return double.NaN;
        }

        private IEnumerable<LineChartSample> EnumerateLineChartSamplesWithThreshold(
            double start,
            double end,
            double scale,
            double lowerThreshold,
            double upperThreshold)
        {
            var loudnessProvider = this.Owner.LoudnessProvider;
            if (loudnessProvider == null)
            {
                yield break;
            }

            if (!loudnessProvider.IsWaveformMetadataReady)
            {
                yield break;
            }

            if (loudnessProvider.LoudnessCount == 0)
            {
                yield break;
            }

            double secStart = start * LoudnessMeterPanelViewModel.SecondsPerFrame;
            double secEnd = end * LoudnessMeterPanelViewModel.SecondsPerFrame;
            double secScale = scale * LoudnessMeterPanelViewModel.SecondsPerFrame;
            double secBase = loudnessProvider.BeginTime.Timestamp.Seconds;
            double rcpSampleRate = 1.0 / loudnessProvider.SampleRate;
            var loudnessInfos = this.Owner.LoudnessInfos;

            // SampleIndex を秒に変換します。
            Func<long, double> toSeconds =
                (sampleIndex) => secBase + sampleIndex * rcpSampleRate;

            // 無限大を実数にクランプします。
            double minValue = this.CenterY + (this.MinimumY - this.CenterY) * 100;
            double maxValue = this.CenterY + (this.MaximumY - this.CenterY) * 100;

            // LoudnessInfo から値に変換します。
            var valueSelector = this.ValueSelectFunc;
            Func<WaveformSpyModel.LoudnessInfo, double> getValue =
                (info) => Spy.MathUtility.Clamp(valueSelector(info), minValue, maxValue);

            // LoudnessInfo １つあたりのサンプル数が１ピクセルあたりのサンプル数より多ければ
            // LoudnessInfo をそのままプロットし、そうでなければ MinMax 値をプロットします。
            if (loudnessProvider.LoudnessStepSampleCount > secScale * loudnessProvider.SampleRate)
            {
                var index = BinarySearchUtility.BinarySearch(loudnessInfos, (int)((secStart - secBase) * loudnessProvider.SampleRate), it => it.SampleIndex);
                if (index < 0)
                {
                    index = Math.Max(0, ~index - 1);
                }

                // start ~ end までを scale の間隔で表示データを作ると LineChart の表示で隙間が入る問題があります。
                // scale の間隔を微妙に減らして、表示データの個数をピクセル数より多く詰め込むようにすると回避できます。
                secScale *= 0.9999999;

                var info1 = this.Owner.LoudnessInfos[index];
                var info0 = info1;

                secStart = GetMultipleOf(Math.Max(secStart, toSeconds(info1.SampleIndex)), secScale);

                for (var t = secStart; t <= secEnd; t += secScale)
                {
                    while (toSeconds(info1.SampleIndex) < t)
                    {
                        if (index >= loudnessInfos.Count)
                        {
                            yield break;
                        }

                        info0 = info1;
                        info1 = loudnessInfos[index++];
                    }

                    var t0 = toSeconds(info0.SampleIndex);
                    var v0 = getValue(info0);
                    var t1 = toSeconds(info1.SampleIndex);
                    var v1 = getValue(info1);

                    double y0 = t0 < t1
                        ? (v0 * (t1 - t) + v1 * (t - t0)) / (t1 - t0)
                        : v1;

                    if (y0 >= lowerThreshold)
                    {
                        y0 = Math.Min(y0, upperThreshold);

                        // rangeX.Length >= 1 にすることで LineChart を MinMax モードで描画させます。
                        // rangeX = new Range1(x, x + 1) だと、計算誤差により rangeX.Length < 1 になってしまう場合があります。
                        var x = t * LoudnessMeterPanelViewModel.FramesPerSecond;
                        var rangeX = new Range1(x, x + 2);
                        var rangeY = new Range1(lowerThreshold, y0);
                        yield return new LineChartSample(rangeX, rangeY);
                    }
                }
            }
            else
            {
                var index = BinarySearchUtility.BinarySearch(loudnessInfos, (int)((secStart - secBase) * loudnessProvider.SampleRate), it => it.SampleIndex);
                if (index < 0)
                {
                    index = Math.Max(0, ~index - 1);
                }

                // start ~ end までを scale の間隔で表示データを作ると LineChart の表示で隙間が入る問題があります。
                // scale の間隔を微妙に減らして、表示データの個数をピクセル数より多く詰め込むようにすると回避できます。
                secScale *= 0.9999999;

                var info = loudnessInfos[index];

                secStart = GetMultipleOf(Math.Max(secStart, toSeconds(info.SampleIndex)), secScale);
                var y = getValue(info);
                var maxY = y;
                var rangeY = new Range1(lowerThreshold, Math.Min(upperThreshold, maxY));

                for (var t = secStart; t <= secEnd; t += secScale)
                {
                    if (index >= loudnessInfos.Count)
                    {
                        break;
                    }

                    info = loudnessInfos[index];
                    var secInfo = toSeconds(info.SampleIndex);
                    if (secInfo < t + secScale)
                    {
                        y = getValue(info);
                        maxY = Math.Max(maxY, y);
                        index++;

                        while (index < loudnessInfos.Count)
                        {
                            info = loudnessInfos[index];
                            secInfo = toSeconds(info.SampleIndex);
                            if (secInfo >= t + secScale)
                            {
                                break;
                            }

                            y = getValue(info);
                            maxY = Math.Max(maxY, y);
                            index++;
                        }

                        rangeY = new Range1(minValue, Math.Min(upperThreshold, maxY));

                        maxY = y;
                    }

                    if (rangeY.Max >= lowerThreshold)
                    {
                        // rangeX.Length >= 1 にすることで LineChart を MinMax モードで描画させます。
                        // rangeX = new Range1(x, x + 1) だと、計算誤差により rangeX.Length < 1 になってしまう場合があります。
                        var x = t * LoudnessMeterPanelViewModel.FramesPerSecond;
                        var rangeX = new Range1(x, x + 2);
                        yield return new LineChartSample(rangeX, rangeY);
                    }
                }
            }
        }

        /// <summary>
        /// baseValue の倍数になるように value を切り捨てます。
        /// </summary>
        /// <param name="value"></param>
        /// <param name="baseValue"></param>
        /// <returns></returns>
        private static double GetMultipleOf(double value, double baseValue)
        {
            return Math.Floor(value / baseValue) * baseValue;
        }
    }
}
