﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using Nintendo.ToolFoundation.Collections;
using Nintendo.ToolFoundation.ComponentModel;
using Nintendo.ToolFoundation.Contracts;
using NintendoWare.Spy;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Markup;
using System.Windows.Media;

namespace NintendoWare.NnAtkSpyPlugin.Windows
{
    /// <summary>
    /// Atk パフォーマンスステートのビューモデルです。
    /// </summary>
    [ContentProperty("Values")]
    public class AtkPerformanceStateViewModel : ObservableObject
    {
        private readonly object _observerOwner = new object();
        private ObservableList<AtkPerformanceStateValueViewModel> _values = new ObservableList<AtkPerformanceStateValueViewModel>();
        private IList<PlotSpyModel.PlotStateValue> _valuesSource;
        private Color _currentColor = Colors.Black;
        private string _currentValue = string.Empty;
        private AtkPerformanceStateValueViewModel _lastValue = null;
        private bool _isVisible = true;
        private double _currentFrame;

        protected AtkPerformanceStateViewModel()
        {
        }

        public AtkPerformanceStateViewModel(PlotSpyModel.PlotState model, string name)
            : this()
        {
            Assertion.Argument.NotNull(model);

            this.Name = name;
            this.Color = model.Color;
            this.ValuesSource = model.Values;
        }

        protected override void DisposeManagedInstance()
        {
            CollectionChangedObservation.RemoveObservers(_observerOwner);

            _values.ForEach(it => it.Dispose());

            base.DisposeManagedInstance();
        }

        public double CurrentFrame
        {
            get { return _currentFrame; }
            set
            {
                if (this.SetPropertyValue(ref _currentFrame, value))
                {
                    this.UpdateCurrent();
                }
            }
        }

        public string Name { get; set; }

        public Color Color { get; set; }

        public ObservableCollection<AtkPerformanceStateValueViewModel> Values { get { return _values; } }

        /// <summary>
        /// Values のソースです。
        /// ValuesSource の内容は Values に追加されます。
        /// </summary>
        public IList<PlotSpyModel.PlotStateValue> ValuesSource
        {
            get { return _valuesSource; }
            set
            {
                if (this.SetPropertyValue(ref _valuesSource, value))
                {
                    CollectionChangedObservation.RemoveObservers(_observerOwner);

                    _values.ForEach(it => it.Dispose());
                    _values.Clear();
                    _lastValue = null;

                    CollectionChangedObservation.GetObserver(_observerOwner, value).AddHandlerForAddItems(
                        (sender, e) =>
                        {
                            e.Items.OfType<PlotSpyModel.PlotStateValue>().ForEach(it => this.AddNewValue(it));
                        });

                    value.ForEach(it => this.AddNewValue(it));
                }
            }
        }

        public Color CurrentColor
        {
            get { return _currentColor; }
            private set { this.SetPropertyValue(ref _currentColor, value); }
        }

        public string CurrentValue
        {
            get { return _currentValue; }
            private set { this.SetPropertyValue(ref _currentValue, value); }
        }

        public bool IsVisible
        {
            get { return _isVisible; }
            set { this.SetPropertyValue(ref _isVisible, value); }
        }

        private void AddNewValue(PlotSpyModel.PlotStateValue model)
        {
            if (_lastValue != null)
            {
                if (model.Time.Timestamp < _lastValue.BeginTime.Timestamp)
                {
                    Assertion.Operation.Fail();
                    return;
                }

                _lastValue.SetEndTime(model.Time);
                _lastValue = null;
            }

            if (!string.IsNullOrEmpty(model.Value))
            {
                var value = new AtkPerformanceStateValueViewModel(model);
                this.Values.Add(value);
                _lastValue = value;
            }
        }

        private void UpdateCurrent()
        {
            var currentFrame = this.CurrentFrame;
            var index = BinarySearchUtility.BinarySearch(_values, currentFrame, it => it.LeftX);
            if (index < 0)
            {
                index = ~index - 1;
            }

            if (index >= 0)
            {
                var currentData = _values[index];

                if (currentData.LeftX <= currentFrame && currentFrame < currentData.RightX)
                {
                    this.CurrentColor = currentData.Color;
                    this.CurrentValue = currentData.Value;
                    return;
                }
            }

            this.CurrentColor = Colors.Black;
            this.CurrentValue = string.Empty;
        }
    }
}
