﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace Nintendo.AudioToolkit.Windows.Controls
{
    /// <summary>
    /// WaveView.xaml の相互作用ロジック
    /// </summary>
    public partial class WaveView : UserControl
    {
        public WaveView()
        {
            InitializeComponent();

            var invokeSizeChanged = new Action<Size>((size) =>
                {
                    BindableRenderSize = size;
                });

            SizeChanged += (s, e) => invokeSizeChanged(e.NewSize);
            DataContextChanged += (s, e) => invokeSizeChanged(RenderSize);
        }

        /// <summary>
        /// サムネイル画像
        /// </summary>
        public Image Thumbnail
        {
            get { return (Image)GetValue(ThumbnailProperty); }
            set { SetValue(ThumbnailProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Thumbnail.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ThumbnailProperty =
            DependencyProperty.Register("Thumbnail", typeof(Image), typeof(WaveView), new FrameworkPropertyMetadata(null));

        public double LoopStartPos
        {
            get { return (double)GetValue(LoopStartPosProperty); }
            set { SetValue(LoopStartPosProperty, value); }
        }

        // Using a DependencyProperty as the backing store for LoopStartPos.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty LoopStartPosProperty =
            DependencyProperty.Register("LoopStartPos", typeof(double), typeof(WaveView), new FrameworkPropertyMetadata(0D, LoopStartPosChanged));


        private static void LoopStartPosChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var view = d as WaveView;
            view.loopArea.Margin = new Thickness(double.IsNaN(view.LoopStartPos) ? 0 : view.LoopStartPos, 0, 0, 0);
        }

        public double LoopEndPos
        {
            get { return (double)GetValue(LoopEndPosProperty); }
            set { SetValue(LoopEndPosProperty, value); }
        }

        // Using a DependencyProperty as the backing store for LoopEndPos.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty LoopEndPosProperty =
            DependencyProperty.Register("LoopEndPos", typeof(double), typeof(WaveView), new FrameworkPropertyMetadata(0D, LoopEndPosChanged));


        private static void LoopEndPosChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
        {
            var view = d as WaveView;
            view.loopArea.Width = view.LoopEndPos - view.LoopStartPos;
            if (view.loopArea.Width > 0)
            {
                view.loopArea.Visibility = Visibility.Visible;
            }
        }

        /// <summary>
        /// バインド可能な描画サイズ。
        /// ただし、バインド時の設定で、
        /// Mode=OneWayToSource
        /// UpdateSourceTrigger=PropertyChanged
        /// にする必要あり。
        /// </summary>
        public Size BindableRenderSize
        {
            get { return (Size)GetValue(BindableRenderSizeProperty); }
            set { SetValue(BindableRenderSizeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for BindableRenderSize.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty BindableRenderSizeProperty =
            DependencyProperty.Register("BindableRenderSize", typeof(Size), typeof(WaveView), new FrameworkPropertyMetadata(new Size(120, 32)));
    }
}
