﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Windows.Threading;
using System.Collections.Specialized;
using System.Reflection;
using System.Runtime.CompilerServices;

namespace Nintendo.AudioToolkit.Views
{
    using Nintendo.Foundation.ComponentModel;
    using Nintendo.Foundation.Windows.Input;
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.DomainModels;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.Windows.Controls;

    public class WaveSoundPresenter : IPresenter
    {
        private WaveSoundView view = null;
        private WaveSoundViewModel viewModel = null;

        private WaveSoundResource waveSoundResource = null;

        private ConditionalWeakTable<WaveSoundResource, ViewInformation> viewInformations = new ConditionalWeakTable<WaveSoundResource, ViewInformation>();

        public event EventHandler BeginTransaction;
        public event EventHandler EndTransaction;
        public event EventHandler CancelTransaction;
        public event DropFileEventHandler DropFiles;
        public event RequestDuplicateEventHandler RequestDuplicate;
        public event EventHandler SelectionChanged;

        public delegate void DropFileEventHandler(object sender, DropFileEventArgs e);
        public delegate void RequestDuplicateEventHandler(object sender, RequestDuplicateEventArgs e);

        public void Initialize()
        {
            this.view = new WaveSoundView();
        }

        /// <summary>
        /// モデルを設定します。
        /// </summary>
        /// <param name="waveSoundDM"></param>
        public void SetModel(WaveSoundResource waveSoundDM)
        {
            if (this.waveSoundResource != null)
            {
                this.PreserveViewInformation(this.waveSoundResource, this.viewModel);
            }

            if (waveSoundDM != null)
            {
                this.waveSoundResource = waveSoundDM;

                this.viewModel = new WaveSoundViewModel(waveSoundDM);
                this.viewModel.SelectionChanged += OnSelectionChanged;

                this.viewModel.UpdateScale();
                this.viewModel.UpdateSnap();

                this.viewModel.ResizeStartedCommand = new DelegateCommand((o) => this.OnBeginTransaction());
                this.viewModel.ResizeCompletedCommand = new DelegateCommand((o) => this.OnEndTransaction());
                this.viewModel.ResizeCanceledCommand = new DelegateCommand((o) => this.OnCancelTransaction());
                this.viewModel.DropStartedCommand = new DelegateCommand((o) => this.OnBeginTransaction());
                this.viewModel.DropCompletedCommand = new DelegateCommand((o) => this.OnEndTransaction());
                this.viewModel.DropCanceledCommand = new DelegateCommand((o) => this.OnCancelTransaction());

                this.viewModel.DropFileCommand = new DelegateCommand<TimelineControl.DropFileEventArgs>(OnDropFile);
                this.viewModel.RequestDuplicateCommand = new DelegateCommand<TimelineControl.RequestDuplicateEventArgs>(OnRequestDuplicate);
                this.viewModel.RequestMoveCommand = new DelegateCommand<TimelineControl.RequestMoveEventArgs>(OnRequestMove);
                this.viewModel.CheckDragClipCommand = new DelegateCommand<TimelineControl.CheckDragClipEventArgs>(OnCheckDragClip);
                this.view.DataContext = this.viewModel;
                this.view.IsEnabled = true;

                this.RestoreViewInformation(this.waveSoundResource, this.viewModel);
            }
            else
            {
                this.waveSoundResource = null;
                this.viewModel = null;
                this.view.DataContext = null;
                this.view.IsEnabled = false;
            }
        }

        private void PreserveViewInformation(WaveSoundResource waveSoundResource, WaveSoundViewModel viewModel)
        {
            if (this.waveSoundResource == null)
            {
                return;
            }

            ViewInformation viewInformation = null;
            if (this.viewInformations.TryGetValue(waveSoundResource, out viewInformation) == false)
            {
                viewInformation = new ViewInformation();
                this.viewInformations.Add(waveSoundResource, viewInformation);
            }

            viewInformation.HorizontalScale = viewModel.HorizontalScale;
            viewInformation.HorizontalOffset = viewModel.HorizontalOffset;
        }

        private void RestoreViewInformation(WaveSoundResource waveSoundResource, WaveSoundViewModel viewModel)
        {
            ViewInformation viewInformation = null;
            if (this.viewInformations.TryGetValue(waveSoundResource, out viewInformation) == false)
            {
                return;
            }

            viewModel.HorizontalScale = viewInformation.HorizontalScale;
            viewModel.HorizontalOffset = viewInformation.HorizontalOffset;
        }


        private void OnSelectionChanged(object sender, EventArgs e)
        {
            Dispatcher.CurrentDispatcher.BeginInvoke(DispatcherPriority.Background, new Action(() =>
                {
                    if (this.SelectionChanged != null)
                    {
                        this.SelectionChanged(sender, EventArgs.Empty);
                    }
                }));
        }

        public Control View
        {
            get
            {
                return this.view;
            }
        }

        public IEnumerable<WaveSoundTrackResource> SelectedTracks
        {
            get
            {
                return this.viewModel.Tracks.Where(t => t.IsSelected == true).Select(c => c.Model);
            }
        }

        public IEnumerable<WaveSoundClipResource> SelectedClips
        {
            get
            {
                return this.viewModel.Tracks.SelectMany(t => t.Clips).Where(c => c.IsSelected == true).Select(c => c.Model);
            }
        }

        public bool GetCenterLogicalPoint(ref int trackNo, ref double position, ref int row)
        {
            return this.view.GetCenterLogicalPoint(ref trackNo, ref position, ref row);
        }

        public bool GetLogicalPoint(ref int trackNo, ref double position, ref int row, int screenX, int screenY)
        {
            return this.view.GetLogicalPoint(ref trackNo, ref position, ref row, screenX, screenY);
        }

        /// <summary>
        /// クリップをすべて選択します。
        /// </summary>
        public void SelectClipAll()
        {
            this.viewModel.Tracks
                .SelectMany(t => t.Clips)
                .ForEach(c => c.IsSelected = true);
        }

        /// <summary>
        /// クリップをすべて非選択にします。
        /// </summary>
        public void UnselectClipAll()
        {
            this.viewModel.Tracks
                .SelectMany(t => t.Clips)
                .ForEach(c => c.IsSelected = false);
        }

        /// <summary>
        /// クリップを選択します。
        /// </summary>
        public void SelectClip(WaveSoundClipResource clipDM)
        {
            var clipVM = this.viewModel.Tracks
                .SelectMany(t => t.Clips)
                .Where(c => c.Model == clipDM)
                .FirstOrDefault();

            if (clipVM != null)
            {
                clipVM.IsSelected = true;
            }
        }

        /// <summary>
        /// 指定トラックにクリップを追加します。
        /// </summary>
        /// <param name="trackDM"></param>
        /// <param name="clipDM"></param>
        public void AddClip(WaveSoundTrackResource trackDM, WaveSoundClipResource clipDM)
        {
            if (trackDM == null)
            {
                return;
            }
            this.waveSoundResource.Add(trackDM, clipDM);
        }

        /// <summary>
        /// 選択されているトラックにクリップを追加します。
        /// </summary>
        /// <param name="data"></param>
        public void AddClipToSelectedTrack(WaveSoundClipResource clipDM)
        {
            this.AddClip(this.SelectedTracks.FirstOrDefault(), clipDM);
        }

        /// <summary>
        /// 選択されているクリップを削除します。
        /// </summary>
        public void RemoveSelectedClips()
        {
            this.waveSoundResource.Remove(this.SelectedClips);
        }

        /// <summary>
        /// トラックを追加します。
        /// </summary>
        /// <param name="trackDM"></param>
        public void AddTrack(WaveSoundTrackResource trackDM)
        {
            this.waveSoundResource.Add(trackDM);
        }

        /// <summary>
        /// 選択されているトラックを削除します。
        /// </summary>
        public void RemoveSelectedTracks()
        {
            this.waveSoundResource.Remove(this.SelectedTracks);
        }

        private void OnBeginTransaction()
        {
            if (this.BeginTransaction != null)
            {
                this.BeginTransaction(this, EventArgs.Empty);
            }
        }

        private void OnEndTransaction()
        {
            if (this.EndTransaction != null)
            {
                this.EndTransaction(this, EventArgs.Empty);
            }
        }

        private void OnCancelTransaction()
        {
            if (this.CancelTransaction != null)
            {
                this.CancelTransaction(this, EventArgs.Empty);
            }
        }

        private void OnDropFile(TimelineControl.DropFileEventArgs e)
        {
            if (this.DropFiles != null)
            {
                var trackVM = (e.Sender as TimelineControl).DataContext as TrackViewModel;
                var trackDM = trackVM.Model;
                this.DropFiles(this, new DropFileEventArgs(trackDM, e.FilePaths, e.Position));
            }
        }

        private void OnRequestDuplicate(TimelineControl.RequestDuplicateEventArgs e)
        {
            if (this.RequestDuplicate != null)
            {
                var trackVM = (e.Sender as TimelineControl).DataContext as TrackViewModel;
                var trackDM = trackVM.Model;

                var clipVM = e.Target as ClipViewModel;
                var clipDM = clipVM.Model;
                var args = new RequestDuplicateEventArgs(clipDM);
                this.RequestDuplicate(this, args);

                if (args.NewInstance is WaveSoundClipResource)
                {
                    var newClipDM = args.NewInstance as WaveSoundClipResource;
                    this.AddClip(trackDM, newClipDM);

                    newClipDM.Position = (int)e.X;
                    newClipDM.Row = (int)(e.Y / this.viewModel.VerticalSnap);
                }
            }
        }

        private void OnRequestMove(TimelineControl.RequestMoveEventArgs e)
        {
            var horizontalValue = (this.viewModel.HorizontalViewRange / 100.0) * this.ProtrusionToMagnification(e.ProtrusionValue);
            var verticalValue = 20.0 * this.ProtrusionToMagnification(e.ProtrusionValue);
            switch (e.MoveDirection)
            {
                case TimelineControl.RequestMoveEventArgs.MoveDirections.Left:
                    var value = this.viewModel.HorizontalOffset - horizontalValue;
                    if (value < 0.0)
                    {
                        value = 0.0;
                    }
                    this.viewModel.HorizontalOffset = value;
                    break;

                case TimelineControl.RequestMoveEventArgs.MoveDirections.Right:
                    this.viewModel.HorizontalOffset += horizontalValue;
                    break;

                case TimelineControl.RequestMoveEventArgs.MoveDirections.Up:
                    this.view.ScrollVerticalListBox(-verticalValue);
                    break;

                case TimelineControl.RequestMoveEventArgs.MoveDirections.Down:
                    this.view.ScrollVerticalListBox(verticalValue);
                    break;
            }
        }

        private double ProtrusionToMagnification(double value)
        {
            var magnification = (double)(int)(value / 30.0) + 1.0;
            return Math.Min(magnification, 5.0);
        }

        private void OnCheckDragClip(TimelineControl.CheckDragClipEventArgs e)
        {
            var trackVM = (e.Sender as TimelineControl).DataContext as TrackViewModel;
            var clipDM = (e.Target as ClipViewModel).Model;
            var row = (int)(e.Y / this.viewModel.VerticalSnap);
            var bounds = new Rect(e.X, row, clipDM.Duration - 1.0, 0.0);
            var isHit = trackVM.Clips
                .Where(c => e.ExcludeTargets.Contains(c) == false)
                .Select(c => new Rect(c.Model.Position, c.Model.Row, c.Model.Duration - 1.0, 0.0))
                .Any(r => r.IntersectsWith(bounds));
            e.Result = isHit == true ? false : true;
        }

        public class DropFileEventArgs : EventArgs
        {
            private WaveSoundTrackResource trackDataModel = null;
            private string[] filePaths = null;
            private int trackNo;
            private Point position;

            public DropFileEventArgs(WaveSoundTrackResource trackDataModel, string[] filePaths, Point position)
            {
                this.trackDataModel = trackDataModel;
                this.filePaths = filePaths;
                this.position = position;
                this.trackNo = trackDataModel.Owner.Tracks.IndexOf(trackDataModel);
            }

            public WaveSoundTrackResource TrackDataModel
            {
                get
                {
                    return this.trackDataModel;
                }
            }

            public Point Position
            {
                get
                {
                    return this.position;
                }
            }

            public string[] FilePaths
            {
                get
                {
                    return this.filePaths;
                }
            }

            public int TrackNo
            {
                get
                {
                    return this.trackNo;
                }
            }
        }

        public class RequestDuplicateEventArgs : EventArgs
        {
            private object instance = null;

            public RequestDuplicateEventArgs(object instance)
            {
                this.instance = instance;
            }

            public object Instance
            {
                get
                {
                    return this.instance;
                }
            }

            public object NewInstance
            {
                get;
                set;
            }
        }

        public class ViewInformation
        {
            public double HorizontalOffset { get; set; }
            public double HorizontalScale { get; set; }
        }
    }


#if false
    /// <summary>
    ///
    /// </summary>
    public class WaveSoundClipDataModel : ObservableObject
    {
        private readonly WaveSoundClipData data = null;
        private IOperationIssuer issuer = null;

        public WaveSoundClipDataModel(WaveSoundClipData data, IOperationIssuer issuer)
        {
            this.data = data;
            this.issuer = issuer;
        }

        public WaveSoundTrackDataModel Owner
        {
            get;
            set;
        }

        public WaveSoundClipData Data
        {
            get { return this.data; }
        }

        public bool IsEnabled
        {
            get { return this.data.IsEnabled; }
            set
            {
                if (this.data.IsEnabled != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, bool>(this, this.data.IsEnabled, value,
                        (v) =>
                        {
                            this.data.IsEnabled = v;
                            this.NotifyPropertyChanged<bool>(() => IsEnabled);
                        });
                }
            }
        }

        public int Position
        {
            get { return this.data.Position; }
            set
            {
                if (this.data.Position != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, int>(this, this.data.Position, value,
                        (v) => {
                            this.data.Position = v;
                            this.NotifyPropertyChanged<int>(() => Position);
                        });
                }
            }
        }

        public int StartOffset
        {
            get { return this.data.StartOffset; }
            set
            {
                if (this.data.StartOffset != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, int>(this, this.data.StartOffset, value,
                        (v) =>
                        {
                            this.data.StartOffset = v;
                            this.NotifyPropertyChanged<int>(() => StartOffset);
                        });
                }
            }
        }

        public int Duration
        {
            get { return this.data.Duration; }
            set
            {
                if (this.data.Duration != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, int>(this, this.data.Duration, value,
                        (v) =>
                        {
                            this.data.Duration = v;
                            this.NotifyPropertyChanged<int>(() => Duration);
                        });
                }
            }
        }

        public float Volume
        {
            get { return this.data.Volume; }
            set
            {
                if (this.data.Volume != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, float>(this, this.data.Volume, value,
                        (v) =>
                        {
                            this.data.Volume = v;
                            this.NotifyPropertyChanged<float>(() => Volume);
                        });
                }
            }
        }

        public float Pitch
        {
            get { return this.data.Pitch; }
            set
            {
                if (this.data.Pitch != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, float>(this, this.data.Pitch, value,
                        (v) =>
                        {
                            this.data.Pitch = v;
                            this.NotifyPropertyChanged<float>(() => Pitch);
                        });
                }
            }
        }

        public float Pan
        {
            get { return this.data.Pan; }
            set
            {
                if (this.data.Pan != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, float>(this, this.data.Pan, value,
                        (v) =>
                        {
                            this.data.Pan = v;
                            this.NotifyPropertyChanged<float>(() => Pan);
                        });
                }
            }
        }

        public string Path
        {
            get { return this.data.Path; }
            set
            {
                if (this.data.Path != value)
                {
                    this.issuer.SetParameter<WaveSoundClipDataModel, string>(this, this.data.Path, value,
                        (v) =>
                        {
                            this.data.Path = v;
                            this.NotifyPropertyChanged<string>(() => Path);
                        });
                }
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class WaveSoundTrackDataModel : ObservableObject
    {
        private readonly WaveSoundTrackData data = null;
        private readonly ObservableCollection<WaveSoundClipDataModel> clips = new ObservableCollection<WaveSoundClipDataModel>();
        private IOperationIssuer issuer = null;

        public WaveSoundTrackDataModel(WaveSoundTrackData data, IOperationIssuer issuer)
        {
            this.data = data;
            this.issuer = issuer;
            this.BuildDataModel(this.data, issuer);

            this.clips.CollectionChanged += OnCollectionChangedClips;
        }

        public IList<WaveSoundClipDataModel> Clips
        {
            get { return this.clips; }
        }

        public WaveSoundDataModel Owner
        {
            get;
            set;
        }

        public bool IsEnabled
        {
            get { return this.data.IsEnabled; }
            set
            {
                this.data.IsEnabled = value;
                this.NotifyPropertyChanged();
            }
        }

        internal WaveSoundTrackData Data
        {
            get { return this.data; }
        }

        private void BuildDataModel(WaveSoundTrackData track, IOperationIssuer issuer)
        {
            foreach (var clip in track.Clips)
            {
                var clipDM = new WaveSoundClipDataModel(clip, issuer);
                clipDM.Owner = this;
                this.Clips.Add(clipDM);
            }
        }

        public void AddClip(WaveSoundClipDataModel clipDM)
        {
            if (this.Contains(clipDM) == true)
            {
                return;
            }

            this.issuer.AddItem(this.Clips, clipDM);
        }

        public void RemoveClip(WaveSoundClipDataModel clipDM)
        {

            if (this.Contains(clipDM) == false)
            {
                return;
            }

            this.issuer.RemoveItem(this.Clips, clipDM);
        }

        private bool Contains(WaveSoundClipDataModel clipDM)
        {
            return this.Clips.Where(c => c == clipDM).Count() > 0 ? true : false;
        }

        private void OnCollectionChangedClips(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (var newItem in e.NewItems)
                    {
                        var clipDM = newItem as WaveSoundClipDataModel;
                        clipDM.Owner = this;

                        this.data.Clips.Add(clipDM.Data);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var oldItem in e.OldItems)
                    {
                        var clipDM = oldItem as WaveSoundClipDataModel;
                        clipDM.Owner = null;

                        this.data.Clips.Remove(clipDM.Data);
                    }
                    break;
            }
        }
    }

    /// <summary>
    ///
    /// </summary>
    public class WaveSoundDataModel : ObservableObject
    {
        private readonly WaveSoundData data = null;
        private readonly ObservableCollection<WaveSoundTrackDataModel> tracks = new ObservableCollection<WaveSoundTrackDataModel>();
        private IOperationIssuer issuer = null;

        public WaveSoundDataModel(WaveSoundData data, IOperationIssuer issuer)
        {
            this.data = data;
            this.issuer = issuer;
            this.BuildDataModel(this.data, issuer);

            this.tracks.CollectionChanged += OnCollectionChangedTracks;
        }

        public IList<WaveSoundTrackDataModel> Tracks
        {
            get { return this.tracks; }
        }

        public void Add(WaveSoundTrackDataModel trackDM, WaveSoundClipDataModel clipDM)
        {
            this.issuer.AddItem(trackDM.Clips, clipDM);
        }

        public void Remove(IEnumerable<WaveSoundClipDataModel> clips)
        {
            foreach (var clip in clips.ToList())
            {
                this.issuer.RemoveItem(clip.Owner.Clips, clip);
            }
        }

        public void Add(WaveSoundTrackDataModel trackDM)
        {
            this.issuer.AddItem(this.Tracks, trackDM);
        }

        public void Remove(IEnumerable<WaveSoundTrackDataModel> tracks)
        {
            foreach (var track in tracks.ToList())
            {
                this.issuer.RemoveItem(track.Owner.Tracks, track);
            }
        }

        private void BuildDataModel(WaveSoundData waveSound, IOperationIssuer issuer)
        {
            foreach (var track in waveSound.Tracks)
            {
                var trackDM = new WaveSoundTrackDataModel(track, issuer);
                trackDM.Owner = this;
                this.Tracks.Add(trackDM);
            }
        }

        private void OnCollectionChangedTracks(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (var newItem in e.NewItems)
                    {
                        var trackDM = newItem as WaveSoundTrackDataModel;
                        trackDM.Owner = this;

                        this.data.Tracks.Add(trackDM.Data);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var oldItem in e.OldItems)
                    {
                        var trackDM = oldItem as WaveSoundTrackDataModel;
                        trackDM.Owner = null;

                        this.data.Tracks.Remove(trackDM.Data);
                    }
                    break;
            }
        }
    }

    public interface IOperationIssuer
    {
        void SetParameter<TItem, T>(TItem item, T oldValue, T newValue, Action<T> action) where TItem : class;
        //void SetParameter<TItem, T>(TItem item, T oldValue, T newValue, System.Linq.Expressions.Expression<Func<T>> propertyName) where TItem : class;
        void AddItem<TItem>(IList<TItem> collection, TItem item) where TItem : class;
        void RemoveItem<TItem>(IList<TItem> collection, TItem item) where TItem : class;

        void BeginTransaction();
        void EndTransaction();
        void CancelTransaction();
    }

    /// <summary>
    ///
    /// </summary>
    public class SoundMakerOperationIssuer : IOperationIssuer
    {
        private readonly OperationHistory operationHistory = null;

        public SoundMakerOperationIssuer(OperationHistory operationHistory)
        {
            this.operationHistory = operationHistory;
        }

        void IOperationIssuer.SetParameter<TItem, T>(TItem item, T oldValue, T newValue, Action<T> action)
        {
            var operation = new SetParameterOpearation<TItem, T>(item, oldValue, newValue, action);
            operation.Execute();
            this.operationHistory.AddOperation(operation);
        }

        //void IOperationIssuer.SetParameter<TItem, T>(TItem item, T oldValue, T newValue, System.Linq.Expressions.Expression<Func<T>> propertyName)
        //{
        //    var operation = new SetParameterOpearation<TItem, T>(item, oldValue, newValue, propertyName);
        //    operation.Execute();
        //    this.operationHistory.AddOperation(operation);
        //}

        void IOperationIssuer.AddItem<TItem>(IList<TItem> collection, TItem item)
        {
            var operation = new AddItemOperation<TItem>(collection, item);
            operation.Execute();
            this.operationHistory.AddOperation(operation);
        }

        void IOperationIssuer.RemoveItem<TItem>(IList<TItem> collection, TItem item)
        {
            var operation = new RemoveItemOperation<TItem>(collection, item);
            operation.Execute();
            this.operationHistory.AddOperation(operation);
        }

        void IOperationIssuer.BeginTransaction()
        {
            this.operationHistory.BeginTransaction();
        }

        void IOperationIssuer.EndTransaction()
        {
            this.operationHistory.EndTransaction();
        }

        void IOperationIssuer.CancelTransaction()
        {
            this.operationHistory.CancelTransaction();
        }
    }

    /// <summary>
    ///
    /// </summary>
    /// <typeparam name="TItem"></typeparam>
    public class AddItemOperation<TItem> : OperationImpl where TItem : class
    {
        private IList<TItem> targetCollection = null;
        private TItem target = null;

        public AddItemOperation(IList<TItem> targetCollection, TItem target)
        {
            this.targetCollection = targetCollection;
            this.target = target;
        }

        protected override bool ExecuteInternal()
        {
            this.targetCollection.Add(this.target);
            return true;
        }

        protected override bool RollbackInternal()
        {
            this.targetCollection.Remove(this.target);
            return true;
        }
    }

    /// <summary>
    ///
    /// </summary>
    /// <typeparam name="TItem"></typeparam>
    public class RemoveItemOperation<TItem> : OperationImpl where TItem : class
    {
        private IList<TItem> targetCollection = null;
        private TItem target = null;
        private int index;

        public RemoveItemOperation(IList<TItem> targetCollection, TItem target)
        {
            this.targetCollection = targetCollection;
            this.target = target;
            this.index = this.targetCollection.IndexOf(this.target);
        }

        protected override bool ExecuteInternal()
        {
            this.targetCollection.RemoveAt(this.index);
            return true;
        }

        protected override bool RollbackInternal()
        {
            this.targetCollection.Insert(this.index, this.target);
            return true;
        }
    }

    /// <summary>
    ///
    /// </summary>
    /// <typeparam name="TItem"></typeparam>
    /// <typeparam name="T"></typeparam>
    //public class SetParameterOpearation<TItem, T> : OperationImpl where TItem : class
    //{
    //    private TItem target = null;
    //    private T oldValue;
    //    private T newValue;
    //    private string name = null;

    //    public SetParameterOpearation(TItem target, T oldValue, T newValue, System.Linq.Expressions.Expression<Func<T>> propertyName)
    //    {
    //        this.target = target;
    //        this.oldValue = oldValue;
    //        this.newValue = newValue;

    //        var member = propertyName.Body as System.Linq.Expressions.MemberExpression;
    //        if (member == null)
    //        {
    //            throw new ArgumentException();
    //        }
    //        this.name = member.Member.Name;
    //    }

    //    protected override bool ExecuteInternal()
    //    {
    //        Type type = typeof(TItem);
    //        PropertyInfo info = type.GetProperty(name);
    //        info.SetValue(this.target, this.newValue, null);
    //        return true;
    //    }

    //    protected override bool RollbackInternal()
    //    {
    //        Type type = typeof(TItem);
    //        PropertyInfo info = type.GetProperty(name);
    //        info.SetValue(this.target, this.oldValue, null);
    //        return true;
    //    }
    //}

    public class SetParameterOpearation<TItem, T> : OperationImpl where TItem : class
    {
        private TItem target = null;
        private T oldValue;
        private T newValue;
        private Action<T> action = null;

        public SetParameterOpearation(TItem target, T oldValue, T newValue, Action<T> action)
        {
            this.target = target;
            this.oldValue = oldValue;
            this.newValue = newValue;
            this.action = action;
        }

        protected override bool ExecuteInternal()
        {
            this.action(this.newValue);
            return true;
        }

        protected override bool RollbackInternal()
        {
            this.action(this.oldValue);
            return true;
        }
    }
#endif
}
