﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Collections.Specialized;
using System.Reflection;

namespace Nintendo.AudioToolkit.DomainModels
{
    using Nintendo.Foundation.ComponentModel;
    using Nintendo.Foundation.Windows.Input;
    using Nintendo.AudioToolkit.DataModels;
    using Nintendo.AudioToolkit.Operations;
    using Nintendo.AudioToolkit.Presenters;
    using Nintendo.AudioToolkit.Windows.Controls;
    using Nintendo.AudioToolkit.Extensions;

    public class WaveSoundTrackResource : ObservableObject
    {
        private readonly WaveSoundTrackResourceData data = null;
        private readonly ObservableCollection<WaveSoundClipResource> clips = new ObservableCollection<WaveSoundClipResource>();
        private IOperationExecutor operationExecutor = null;

        public WaveSoundTrackResource(WaveSoundTrackResourceData data)
        {
            this.data = data;
            this.BuildDataModel(this.data);

            this.clips.CollectionChanged += OnCollectionChangedClips;
        }

        public IList<WaveSoundClipResource> Clips
        {
            get { return this.clips; }
        }

        public bool IsEnabled
        {
            get { return this.data.IsEnabled; }
            set
            {
                this.data.IsEnabled = value;
                this.NotifyPropertyChanged();
            }
        }

        internal WaveSoundTrackResourceData Data
        {
            get { return this.data; }
        }

        private void BuildDataModel(WaveSoundTrackResourceData track)
        {
            foreach (var clip in track.ClipResources)
            {
                var clipDM = new WaveSoundClipResource(clip);
                clipDM.Owner = this;
                this.Clips.Add(clipDM);
            }
        }

        public void AddClip(WaveSoundClipResource clipDM)
        {
            if (this.Contains(clipDM) == true)
            {
                return;
            }

            this.OperationExecutor.AddItem(this.Clips, clipDM);
        }

        public void RemoveClip(WaveSoundClipResource clipDM)
        {

            if (this.Contains(clipDM) == false)
            {
                return;
            }

            this.OperationExecutor.RemoveItem(this.Clips, clipDM);
        }

        private bool Contains(WaveSoundClipResource clipDM)
        {
            return this.Clips.Where(c => c == clipDM).Count() > 0 ? true : false;
        }

        private void OnCollectionChangedClips(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                    foreach (var newItem in e.NewItems)
                    {
                        var clipDM = newItem as WaveSoundClipResource;
                        clipDM.Owner = this;
                        clipDM.OperationExecutor = this.OperationExecutor;

                        this.data.ClipResources.Add(clipDM.Data);
                    }
                    break;

                case NotifyCollectionChangedAction.Remove:
                    foreach (var oldItem in e.OldItems)
                    {
                        var clipDM = oldItem as WaveSoundClipResource;
                        clipDM.Owner = null;
                        clipDM.OperationExecutor = null;

                        this.data.ClipResources.Remove(clipDM.Data);
                    }
                    break;
            }
        }

        internal WaveSoundResource Owner
        {
            get;
            set;
        }

        internal IOperationExecutor OperationExecutor
        {
            get { return this.operationExecutor; }
            set
            {
                this.operationExecutor = value;
                this.Clips.ToList().ForEach(c => c.OperationExecutor = value);
            }
        }
    }
}
