﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"

#include <panels/cafe/EffectCafePanel.h>
#include <resources/GlobalResources.h>

namespace {
    const u32 DEVICE_TYPE_MAIN = 0;
    const u32 DEVICE_TYPE_DRC0 = 1;

    static nw::snd::PreviewEffectType s_DefaultPreviewEffectTypes[] =
    {
        nw::snd::EFFECT_NONE,
        nw::snd::EFFECT_REVERB_HI,
        nw::snd::EFFECT_REVERB_STD,
        nw::snd::EFFECT_CHORUS,
        nw::snd::EFFECT_DELAY,
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
        nw::snd::EFFECT_MULTICH_REVERB,
        nw::snd::EFFECT_MULTICH_CHORUS,
        nw::snd::EFFECT_MULTICH_DELAY
#endif
    };
}

namespace nw {
namespace snd {

EffectCafePanel::EffectCafePanel() :
m_pEffectManager(NULL),
m_pEffectManagerDrc(NULL),
m_EffectAppliedEventHandler(this, &EffectCafePanel::OnEffectApplied),
m_EffectDrcAppliedEventHandler(this, &EffectCafePanel::OnEffectApplied)
{
}

EffectCafePanel::~EffectCafePanel()
{
}

void EffectCafePanel::Initialize(IResourceProvider& resourceProvider)
{
    PreviewEffectManager* pPreviewEffectManager = reinterpret_cast<PreviewEffectManager*>(resourceProvider.GetResource(GlobalResources::PreviewEffectManagerPath));
    PreviewEffectManager* pPreviewEffectManagerDrc = reinterpret_cast<PreviewEffectManager*>(resourceProvider.GetResource(GlobalResources::PreviewEffectManagerDrcPath));
    NW_ASSERT_NOT_NULL(pPreviewEffectManager);
    NW_ASSERT_NOT_NULL(pPreviewEffectManagerDrc);

    InitializeContent();

    m_pEffectManager = pPreviewEffectManager;
    m_pEffectManager->GetAppliedEvent() += m_EffectAppliedEventHandler;
    m_pEffectManagerDrc = pPreviewEffectManagerDrc;
    m_pEffectManagerDrc->GetAppliedEvent() += m_EffectDrcAppliedEventHandler;

    InitializeModel();
}

void EffectCafePanel::Finalize()
{
    if(m_pEffectManager != NULL)
    {
        m_pEffectManager->GetAppliedEvent() -= m_EffectAppliedEventHandler;
        m_pEffectManager = NULL;
    }

    if(m_pEffectManagerDrc != NULL)
    {
        m_pEffectManagerDrc->GetAppliedEvent() -= m_EffectDrcAppliedEventHandler;
        m_pEffectManagerDrc = NULL;
    }
}

void EffectCafePanel::SetDevice(u32 deviceType)
{
    if (deviceType == DEVICE_TYPE_MAIN)
    {
        m_MainLamp.SetIsLamped(true);
        m_DrcLamp.SetIsLamped(false);
    }
    else if(deviceType == DEVICE_TYPE_DRC0)
    {
        m_MainLamp.SetIsLamped(false);
        m_DrcLamp.SetIsLamped(true);
    }
    else
    {
        NW_ASSERT(false);
    }

    InitializeModel();
}

PreviewEffectManager* EffectCafePanel::CurrentPreviewEffectManager()
{
    if (IsMainOutput())
    {
        return m_pEffectManager;
    }
    else
    {
        return m_pEffectManagerDrc;
    }
}

nw::snd::PreviewEffectType* EffectCafePanel::GetDefaultPreviewEffectTypes()
{
    return s_DefaultPreviewEffectTypes;
}

u32 EffectCafePanel::GetPreviewEffectCount()
{
    return sizeof(s_DefaultPreviewEffectTypes) / sizeof(PreviewEffectType);
}

bool EffectCafePanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if ( pad.IsHold(nw::dev::Pad::MASK_L) )
    {
        if ( pad.IsTrig(nw::dev::Pad::MASK_LEFT) || pad.IsTrig(nw::dev::Pad::MASK_RIGHT) )
        {
            SetDevice( IsMainOutput() ? DEVICE_TYPE_DRC0 : DEVICE_TYPE_MAIN );
        }

        return false;
    }
    else
    {
        return EffectPanel::OnUpdateFocusedInput(inputs);
    }
}

bool EffectCafePanel::IsMainOutput()
{
    return m_MainLamp.GetIsLamped();
}

void EffectCafePanel::InitializeLampBar(nw::internal::dw::StackPanel& lampBar)
{
    m_LampBarContents.AddItem(&m_MainLamp);
    m_LampBarContents.AddItem(&m_DrcLamp);
    lampBar.SetContents(m_LampBarContents);

    m_MainLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_MainLamp.SetLabelText("Main");
    m_MainLamp.SetIsLamped(true);

    m_DrcLamp.SetMargin(nw::internal::dw::Thickness(1.f));
    m_DrcLamp.SetLabelText("DRC");
    m_DrcLamp.SetIsLamped(false);
}

void EffectCafePanel::OnEffectApplied()
{
    if(m_pEffectManager == NULL && m_pEffectManagerDrc == NULL)
    {
        SetAssignedMemory("0 KB");
    }
    else
    {
        s32 assignedSize = 0;
        if (m_pEffectManager != NULL)
        {
            assignedSize += m_pEffectManager->GetAssinedSize();
        }
        if (m_pEffectManagerDrc != NULL)
        {
            assignedSize += m_pEffectManagerDrc->GetAssinedSize();
        }

        char text[16];
        ut::snprintf(text, 16, 16 - 1, "%d KB", assignedSize / 1024);

        SetAssignedMemory(text);
    }
}

} // snd
} // nw
