﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <panels/EffectPanel.h>
#include <models/PreviewEffectSet.h>

#include <nw/dw/system/dw_NwTypeUtility.h>

namespace nw {
namespace snd {

EffectPanel::EffectPanel() :
m_SelectedAuxBusIndex(INVALID_INDEX),
m_SelectedEffectIndex(INVALID_INDEX),
m_CommandState(CommandState_Idle)
{
}

EffectPanel::~EffectPanel()
{
}

void EffectPanel::InitializeContent()
{
    nw::snd::PreviewEffectType* defaultPreviewEffectTypes = GetDefaultPreviewEffectTypes();
    m_pPreviewEffectTypes = defaultPreviewEffectTypes;

    CalculateEffectCount();

    m_Contents.AddItem(&m_Container);
    SetContents(m_Contents);

    m_Container.SetMargin(nw::internal::dw::Thickness(0.f));
    m_Container.SetPadding(nw::internal::dw::Thickness(0.f));

    m_Container.GetContents().AddItem(&m_AuxBusPanel);
    m_Container.GetContents().AddItem(&m_AssignedMemoryPanel);
    m_Container.GetContents().AddItem(&m_Separator);
    m_Container.GetContents().AddItem(&m_EffectParametersView);
    m_Container.GetContents().AddItem(&m_LampBar);

    m_AuxBusPanel.SetDock(nw::internal::dw::DOCK_TOP);
    m_AuxBusPanel.SetMargin(nw::internal::dw::Thickness(0.f));
    m_AuxBusPanel.SetPadding(nw::internal::dw::Thickness(0.f, 2.f));
    m_AuxBusPanel.SetOrientation(nw::internal::dw::VERTICAL);

    m_AssignedMemoryPanel.SetDock(nw::internal::dw::DOCK_TOP);
    m_AssignedMemoryPanel.SetMargin(nw::internal::dw::Thickness(0.f));
    m_AssignedMemoryPanel.SetHorizontalAlignment(nw::internal::dw::HORIZONTAL_RIGHT);
    m_AssignedMemoryPanel.SetOrientation(nw::internal::dw::HORIZONTAL);
    m_AssignedMemoryPanel.GetContents().AddItem(&m_AssignedMemoryLabel);
    m_AssignedMemoryPanel.GetContents().AddItem(&m_AssignedMemoryValueLabel);

    m_AssignedMemoryLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f, 0.f, 1.f));
    m_AssignedMemoryLabel.SetTextScale(0.8f);
    m_AssignedMemoryLabel.SetText("Memory :");

    m_AssignedMemoryValueLabel.SetMargin(nw::internal::dw::Thickness(2.f, 0.f, 2.f, 1.f));
    m_AssignedMemoryValueLabel.SetTextScale(0.8f);
    m_AssignedMemoryValueLabel.SetTextAlignment(nw::internal::dw::TEXT_RIGHT);
    m_AssignedMemoryValueLabel.SetMinimumWidth(52.f);

    m_Separator.SetDock(nw::internal::dw::DOCK_TOP);
    m_Separator.SetMargin(nw::internal::dw::Thickness(2.f, 4.f));
    m_Separator.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_Separator.SetHeight(0.5f);
    m_Separator.SetOrientation(nw::internal::dw::HORIZONTAL);
    m_Separator.SetBorderColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.455f, 0.486f, 0.560f, 1.f)));

    m_EffectParametersView.SetMargin(nw::internal::dw::Thickness(0.f, 2.f));

    char busID = 'A';

    for(s32 i = 0; i < MAX_EFFECT_BUS_COUNT; ++i)
    {
        AuxBusControl& auxBus = m_AuxBusControls[i];
        auxBus.SetMargin(nw::internal::dw::Thickness(2.f, 0.f));

        char busName[8];
        ut::snprintf(busName, 8,  8 - 1, "Aux%c", busID);
        auxBus.SetText(busName);

        busID++;
    }

    SetAuxBusCount(PreviewEffectManager::EFFECT_COUNT);

    m_LampBar.SetOrientation(nw::internal::dw::HORIZONTAL);
    m_LampBar.SetDock(nw::internal::dw::DOCK_BOTTOM);
    m_LampBar.SetMargin(nw::internal::dw::Thickness(0.f));
    m_LampBar.SetBackgroundColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.3f, 0.3f, 0.3f, 1.f)));
    InitializeLampBar(m_LampBar);
}

void EffectPanel::InitializeModel()
{
    NW_ASSERT_NOT_NULL( CurrentPreviewEffectManager() );
    for(s32 i = 0; i < GetAuxBusCount(); ++i)
    {
        m_AuxBusControls[i].Initialize(*CurrentPreviewEffectManager()->GetAuxBus(i));
    }

    if(m_SelectedAuxBusIndex == INVALID_INDEX || m_SelectedEffectIndex == INVALID_INDEX)
    {
        return;
    }

    CalculateEffectCount();

    NW_ASSERT(PreviewEffectManager::AUX_BUS_COUNT >= GetAuxBusCount());

    m_EffectParametersView.SetModel(
        GetSelectedEffectSet(),
        m_SelectedAuxBusIndex,
        m_SelectedEffectIndex);

    OnEffectApplied();
}

s32 EffectPanel::GetSelectedAuxBusIndex()
{
    return m_SelectedAuxBusIndex;
}

s32 EffectPanel::GetSelectedEffectIndex()
{
    return m_SelectedEffectIndex;
}

void EffectPanel::SetSelectedEffectIndex(s32 auxBusIndex, s32 effectIndex)
{
    if(m_SelectedAuxBusIndex == auxBusIndex && m_SelectedEffectIndex == effectIndex)
    {
        return;
    }

    for(s32 i = 0; i < GetAuxBusCount(); ++i)
    {
        m_AuxBusControls[i].SetSelectedEffectIndex(INVALID_INDEX);
    }

    m_SelectedAuxBusIndex = auxBusIndex;
    m_SelectedEffectIndex = effectIndex;

    if(m_SelectedAuxBusIndex == INVALID_INDEX)
    {
        return;
    }

    m_AuxBusControls[auxBusIndex].SetSelectedEffectIndex(effectIndex);

    if(effectIndex == INVALID_INDEX)
    {
        return;
    }

    if(CurrentPreviewEffectManager() == NULL)
    {
        return;
    }

    m_EffectParametersView.SetModel(
        GetSelectedEffectSet(),
        m_SelectedAuxBusIndex,
        m_SelectedEffectIndex);
}

void EffectPanel::SelectEffect(s32 auxBusIndex, s32 effectIndex)
{
    SetSelectedEffectIndex(auxBusIndex, effectIndex);

    if(auxBusIndex == INVALID_INDEX || effectIndex == INVALID_INDEX)
    {
        return;
    }

    m_AuxBusControls[auxBusIndex].SelectEffect(effectIndex);
}

bool EffectPanel::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_LEFT))
    {
        if(m_SelectedAuxBusIndex != INVALID_INDEX &&
            m_SelectedEffectIndex > 0)
        {
            SelectEffect(m_SelectedAuxBusIndex, m_SelectedEffectIndex - 1);
        }
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_RIGHT))
    {
        if(m_SelectedAuxBusIndex != INVALID_INDEX &&
            m_SelectedEffectIndex < m_AuxBusControls[m_SelectedAuxBusIndex].GetEffectCount() - 1)
        {
            SelectEffect(m_SelectedAuxBusIndex, m_SelectedEffectIndex + 1);
        }
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_UP))
    {
        if(m_SelectedAuxBusIndex > 0)
        {
            SelectEffect(m_SelectedAuxBusIndex - 1, m_SelectedEffectIndex);
        }
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_DOWN))
    {
        if(m_SelectedAuxBusIndex < GetAuxBusCount() - 1)
        {
            SelectEffect(m_SelectedAuxBusIndex + 1, m_SelectedEffectIndex);
        }
    }
    if(pad.IsTrig(nw::dev::Pad::MASK_X))
    {
        s32 previewEffectTypeIndex = GetSelectedPreviewEffectTypeIndex();

        if(m_PreviewEffectCount <= 0)
        {
            return true;
        }

        previewEffectTypeIndex++;

        if(previewEffectTypeIndex == m_PreviewEffectCount)
        {
            previewEffectTypeIndex = 0;
        }

        SetSelectedPreviewEffectTypeIndex(previewEffectTypeIndex);
        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_Y))
    {
        if(m_AuxBusPanel.GetIsContainsFocus())
        {
            m_EffectParametersView.SetFocus();
        }
        else
        {
            m_AuxBusPanel.SetFocus();
        }
    }

    return false;
}

void EffectPanel::SetAssignedMemory(const char* text)
{
    m_AssignedMemoryValueLabel.SetText(text);
}

s32 EffectPanel::GetAuxBusCount() const
{
    return m_AuxBusPanel.GetContents().GetCount();
}

void EffectPanel::SetAuxBusCount(s32 value)
{
    NW_ASSERT(value < MAX_EFFECT_BUS_COUNT);

    m_AuxBusPanel.GetContents().ClearItems();

    for(s32 i = 0; i < value; ++i)
    {
        m_AuxBusPanel.GetContents().AddItem(&m_AuxBusControls[i]);
    }

    if(value > 0)
    {
        SetSelectedEffectIndex(0, 0);
    }
    else
    {
        SetSelectedEffectIndex(INVALID_INDEX, INVALID_INDEX);
    }
}

PreviewEffectSet* EffectPanel::GetSelectedEffectSet()
{
    NW_ASSERT_NOT_NULL( CurrentPreviewEffectManager() );
    return CurrentPreviewEffectManager()->GetAuxBus(m_SelectedAuxBusIndex)->GetEffect(m_SelectedEffectIndex);
}

s32 EffectPanel::GetSelectedPreviewEffectTypeIndex()
{
    PreviewEffectSet* pEffectSet = GetSelectedEffectSet();

    if(pEffectSet == NULL)
    {
        return INVALID_INDEX;
    }

    PreviewEffectType previewEffectType = pEffectSet->GetCurrent().GetEffectType();

    for(s32 i = 0; i < m_PreviewEffectCount; ++i)
    {
        if(GetPreviewEffectType(i) == previewEffectType)
        {
            return i;
        }
    }

    return INVALID_INDEX;
}

void EffectPanel::SetSelectedPreviewEffectTypeIndex(s32 index)
{
    PreviewEffectSet* pEffectSet = GetSelectedEffectSet();

    if(pEffectSet == NULL)
    {
        return;
    }

    pEffectSet->SetCurrent(GetPreviewEffectType(index));
}

snd::PreviewEffectType EffectPanel::GetPreviewEffectType(s32 index)
{
    if(index < 0 || m_PreviewEffectCount <= index)
    {
        NW_ASSERT(m_PreviewEffectCount > 0);
        return m_pPreviewEffectTypes[0];
    }

    return m_pPreviewEffectTypes[index];
}

void EffectPanel::CalculateEffectCount()
{
    m_PreviewEffectCount = GetPreviewEffectCount();
}

} // snd
} // nw
