﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <models/PreviewSound.h>
#include <models/PreviewSoundArchive.h>

#if defined(NW_ENABLE_SNDCTRL)
#include <nw/snd/ctrl/sndctrl_SequenceVariable.h>
#endif

namespace nw {
namespace snd {

#if defined(NW_ENABLE_SNDCTRL)

PreviewSound::PreviewSound() :
m_pPreviewSoundArchive(NULL),
m_SoundController(NULL)
{
}

void PreviewSound::Initialize(
    PreviewSoundArchive* previewSoundArchive,
    ctrl::SoundController* soundController)
{
    NW_ASSERT_NOT_NULL(previewSoundArchive);
    NW_ASSERT_NOT_NULL(soundController);

    m_pPreviewSoundArchive = previewSoundArchive;
    m_SoundController =
        reinterpret_cast<ctrl::internal::SoundControllerImpl*>(soundController);
}

void PreviewSound::Finalize()
{
    // todo:
    // Initialize で、再度初期化する前にリセットが必要な処理（メモリ確保など）が追加されたら
    // こちらに後始末の処理を実装してください。
}

const nw::snd::SoundHandle& PreviewSound::GetSoundHandle() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return *m_SoundController->GetSoundHandle();
}

nw::snd::SoundHandle& PreviewSound::GetSoundHandle()
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return *m_SoundController->GetSoundHandle();
}

nw::snd::SoundArchive::ItemId PreviewSound::GetSoundID() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetSoundID();
}

void PreviewSound::SetSoundID(nw::snd::SoundArchive::ItemId soundID)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetSoundID(soundID);
}

u32 PreviewSound::GetSoundIndex() const
{
    // HACK : インデックスを抽出する正式な方法が見つかれば置き換える
    return nw::snd::internal::Util::GetItemIndex(GetSoundID());
}

void PreviewSound::SetSoundIndex(u32 index)
{
    NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewSound is not initialized.");
    SetSoundID(m_pPreviewSoundArchive->GetSoundID(index));
}

void PreviewSound::SetLabel(const char* label)
{
    NW_ASSERTMSG(m_pPreviewSoundArchive != NULL, "PreviewSound is not initialized.");
    SetSoundID(m_pPreviewSoundArchive->GetSoundID(label));
}

const char* PreviewSound::GetLabel() const
{
    if (m_pPreviewSoundArchive == NULL)
    {
        return "";
    }

    if(GetSoundID() == nw::snd::SoundArchive::INVALID_ID)
    {
        return "";
    }
    else
    {
        return m_pPreviewSoundArchive->GetItemLabel(GetSoundID());
    }
}

PreviewSound::State PreviewSound::GetState() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(m_SoundController->GetState())
    {
    case ctrl::SoundController::STATE_DISABLED:
    case ctrl::SoundController::STATE_STOPPED:
        return STATE_STOPPED;

    case ctrl::SoundController::STATE_PLAYING:
        return STATE_PLAYING;

    case ctrl::SoundController::STATE_PAUSED:
        return STATE_PAUSED;
    }

    return STATE_STOPPED;
}

void PreviewSound::Update()
{
    // m_SoundIDにマッチするデータがあればそれで設定する
    SetSoundIndex(ValidateSoundID() ? GetSoundIndex() : 0);
}

bool PreviewSound::Play()
{
    bool result = false;
    NW_ASSERT_NOT_NULL(m_SoundController);

    NW_ASSERT_NOT_NULL(m_pPreviewSoundArchive);
    if (!m_pPreviewSoundArchive->IsOpened())
    {
        return false;
    }
#ifdef NW_PLATFORM_CAFE
    if(GetStartOffset() > 0)
#else
    const SoundArchive& soundArchive = m_pPreviewSoundArchive->GetSoundArchive();
    if(soundArchive.GetSoundType(m_SoundController->GetSoundID()) != nw::snd::SoundArchive::SOUND_TYPE_STRM &&
       GetStartOffset() > 0)
#endif
    {
        nw::snd::SoundStartable::StartInfo startInfo;
        startInfo.enableFlag = nw::snd::SoundStartable::StartInfo::ENABLE_START_OFFSET;
        startInfo.startOffsetType = nw::snd::SoundStartable::StartInfo::START_OFFSET_TYPE_MILLISEC;
        startInfo.startOffset = GetStartOffset();

        result = m_SoundController->Play(&startInfo).IsSucceeded();
    }
    else
    {
        result = m_SoundController->Play().IsSucceeded();
    }

    return result;
}

bool PreviewSound::Stop()
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->Stop().IsSucceeded();
}

bool PreviewSound::Pause()
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->Pause().IsSucceeded();
}

void PreviewSound::Next(s32 offset)
{
    if (m_pPreviewSoundArchive == NULL)
    {
        NW_WARNING(false, "PreviewSound is not initialized.\n");
        return;
    }

    s32 soundCount = static_cast<s32>(m_pPreviewSoundArchive->GetSoundCount());

    if(soundCount == 0)
    {
        return;
    }

    s32 soundIndex = static_cast<s32>(GetSoundIndex()) + offset;

    if(soundIndex >= soundCount)
    {
        soundIndex = 0;
    }
    else if(soundIndex < 0)
    {
        soundIndex = soundCount - 1;
    }

    SetSoundIndex(soundIndex);
}

u32 PreviewSound::GetStartOffset() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().startOffset;
}

void PreviewSound::SetStartOffset(u32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().startOffset != value);
    m_SoundController->GetParameters().startOffset = value;
}

f32 PreviewSound::GetPitch() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().pitch;
}

void PreviewSound::SetPitch(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().pitch != value);
    m_SoundController->GetParameters().pitch = value;
}

f32 PreviewSound::GetLpfFrequency() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().lpfFrequency;
}

void PreviewSound::SetLpfFrequency(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().lpfFrequency != value);
    m_SoundController->GetParameters().lpfFrequency = value;
}

BiquadFilterType PreviewSound::GetBiquadFilterType() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return static_cast<BiquadFilterType>(m_SoundController->GetParameters().biquadFilterType);
}

void PreviewSound::SetBiquadFilterType(BiquadFilterType value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().biquadFilterType != value);
    m_SoundController->GetParameters().biquadFilterType = value;
}

f32 PreviewSound::GetBiquadFilterValue() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().biquadFilterValue;
}

void PreviewSound::SetBiquadFilterValue(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().biquadFilterValue != value);
    m_SoundController->GetParameters().biquadFilterValue = value;
}

f32 PreviewSound::GetVolume() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().volume;
}

void PreviewSound::SetVolume(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().volume != value);
    m_SoundController->GetParameters().volume = value;
}

f32 PreviewSound::GetPan() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().pan;
}

void PreviewSound::SetPan(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().pan != value);
    m_SoundController->GetParameters().pan = value;
}

f32 PreviewSound::GetSurroundPan() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().surroundPan;
}

void PreviewSound::SetSurroundPan(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().surroundPan != value);
    m_SoundController->GetParameters().surroundPan = value;
}

f32 PreviewSound::GetMainSend() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().mainSend;
}

void PreviewSound::SetMainSend(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainSend != value);
    m_SoundController->GetParameters().mainSend = value;
}

f32 PreviewSound::GetFxSend(AuxBus auxBus) const
{
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(auxBus)
    {
    case AUX_BUS_A:
        return m_SoundController->GetParameters().fxSend[0];

    case AUX_BUS_B:
        return m_SoundController->GetParameters().fxSend[1];

    case AUX_BUS_C:
        return m_SoundController->GetParameters().fxSend[2];

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }

    return 0.f;
}

void PreviewSound::SetFxSend(AuxBus auxBus, f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(auxBus)
    {
    case AUX_BUS_A:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().fxSend[0] != value);
        m_SoundController->GetParameters().fxSend[0] = value;
        break;

    case AUX_BUS_B:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().fxSend[1] != value);
        m_SoundController->GetParameters().fxSend[1] = value;
        break;

    case AUX_BUS_C:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().fxSend[2] != value);
        m_SoundController->GetParameters().fxSend[2] = value;
        break;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }
}

f32 PreviewSound::GetMainOutVolume() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().mainOutVolume;
}

void PreviewSound::SetMainOutVolume(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainOutVolume != value);
    m_SoundController->GetParameters().mainOutVolume = value;
}

f32 PreviewSound::GetMainPan() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().mainPan;
}

void PreviewSound::SetMainPan(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainPan != value);
    m_SoundController->GetParameters().mainPan = value;
}

f32 PreviewSound::GetMainSurroundPan() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().mainSurroundPan;
}

void PreviewSound::SetMainSurroundPan(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainSurroundPan != value);
    m_SoundController->GetParameters().mainSurroundPan = value;
}

f32 PreviewSound::GetMainMainSend() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().mainMainSend;
}

void PreviewSound::SetMainMainSend(f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainMainSend != value);
    m_SoundController->GetParameters().mainMainSend = value;
}

f32 PreviewSound::GetMainFxSend(AuxBus auxBus) const
{
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(auxBus)
    {
    case AUX_BUS_A:
        return m_SoundController->GetParameters().mainFxSend[0];

    case AUX_BUS_B:
        return m_SoundController->GetParameters().mainFxSend[1];

    case AUX_BUS_C:
        return m_SoundController->GetParameters().mainFxSend[2];

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }

    return 0.f;
}

void PreviewSound::SetMainFxSend(AuxBus auxBus, f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(auxBus)
    {
    case AUX_BUS_A:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainFxSend[0] != value);
        m_SoundController->GetParameters().mainFxSend[0] = value;
        break;

    case AUX_BUS_B:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainFxSend[1] != value);
        m_SoundController->GetParameters().mainFxSend[1] = value;
        break;

    case AUX_BUS_C:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().mainFxSend[2] != value);
        m_SoundController->GetParameters().mainFxSend[2] = value;
        break;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }
}

f32 PreviewSound::GetDrcOutVolume() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().drcOutVolume[0];
#else
    return 0.f;
#endif
}

void PreviewSound::SetDrcOutVolume(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcOutVolume[0] != value);
    m_SoundController->GetParameters().drcOutVolume[0] = value;
    m_SoundController->GetParameters().drcOutVolume[1] = value;
#endif
}

f32 PreviewSound::GetDrcPan() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().drcPan[0];
#else
    return 0.f;
#endif
}

void PreviewSound::SetDrcPan(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcPan[0] != value);
    m_SoundController->GetParameters().drcPan[0] = value;
    m_SoundController->GetParameters().drcPan[1] = value;
#endif
}

f32 PreviewSound::GetDrcSurroundPan() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().drcSurroundPan[0];
#else
    return 0.f;
#endif
}

void PreviewSound::SetDrcSurroundPan(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcSurroundPan[0] != value);
    m_SoundController->GetParameters().drcSurroundPan[0] = value;
    m_SoundController->GetParameters().drcSurroundPan[1] = value;
#endif
}

f32 PreviewSound::GetDrcMainSend() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().drcMainSend[0];
#else
    return 0.f;
#endif
}

void PreviewSound::SetDrcMainSend(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcMainSend[0] != value);
    m_SoundController->GetParameters().drcMainSend[0] = value;
    m_SoundController->GetParameters().drcMainSend[1] = value;
#endif
}

f32 PreviewSound::GetDrcFxSend(AuxBus auxBus) const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(auxBus)
    {
    case AUX_BUS_A:
        return m_SoundController->GetParameters().drcFxSend[0][0];

    case AUX_BUS_B:
        return m_SoundController->GetParameters().drcFxSend[0][1];

    case AUX_BUS_C:
        return m_SoundController->GetParameters().drcFxSend[0][2];

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }

    return 0.f;
#else
    return 0.f;
#endif
}

void PreviewSound::SetDrcFxSend(AuxBus auxBus, f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);

    switch(auxBus)
    {
    case AUX_BUS_A:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcFxSend[0][0] != value);
        m_SoundController->GetParameters().drcFxSend[0][0] = value;
        m_SoundController->GetParameters().drcFxSend[1][0] = value;
        break;

    case AUX_BUS_B:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcFxSend[0][1] != value);
        m_SoundController->GetParameters().drcFxSend[0][1] = value;
        m_SoundController->GetParameters().drcFxSend[1][1] = value;
        break;

    case AUX_BUS_C:
        m_SoundController->SetParameterDirty(m_SoundController->GetParameters().drcFxSend[0][2] != value);
        m_SoundController->GetParameters().drcFxSend[0][2] = value;
        m_SoundController->GetParameters().drcFxSend[1][2] = value;
        break;

    default:
        NW_FATAL_ERROR("invalid AuxBus.");
    }
#endif
}

f32 PreviewSound::GetRemote0OutVolume() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().remoteOutVolume[0];
#else
    return 0.f;
#endif
}

void PreviewSound::SetRemote0OutVolume(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().remoteOutVolume[0] != value);
    m_SoundController->GetParameters().remoteOutVolume[0] = value;
#endif
}

f32 PreviewSound::GetRemote1OutVolume() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().remoteOutVolume[1];
#else
    return 0.f;
#endif
}

void PreviewSound::SetRemote1OutVolume(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().remoteOutVolume[1] != value);
    m_SoundController->GetParameters().remoteOutVolume[1] = value;
#endif
}

f32 PreviewSound::GetRemote2OutVolume() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().remoteOutVolume[2];
#else
    return 0.f;
#endif
}

void PreviewSound::SetRemote2OutVolume(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().remoteOutVolume[2] != value);
    m_SoundController->GetParameters().remoteOutVolume[2] = value;
#endif
}

f32 PreviewSound::GetRemote3OutVolume() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().remoteOutVolume[3];
#else
    return 0.f;
#endif
}

void PreviewSound::SetRemote3OutVolume(f32 value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().remoteOutVolume[3] != value);
    m_SoundController->GetParameters().remoteOutVolume[3] = value;
#endif
}

bool PreviewSound::GetIsMainOutEnabled() const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().isMainOutEnabled;
}

void PreviewSound::SetIsMainOutEnabled(bool value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().isMainOutEnabled != value);
    m_SoundController->GetParameters().isMainOutEnabled = value;
}

bool PreviewSound::GetIsDrcOutEnabled() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().isDrcOutEnabled[0];
#else
    return false;
#endif
}

void PreviewSound::SetIsDrcOutEnabled(bool value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().isDrcOutEnabled[0] != value);
    m_SoundController->GetParameters().isDrcOutEnabled[0] = value;
    m_SoundController->GetParameters().isDrcOutEnabled[1] = value;
#endif
}

bool PreviewSound::GetIsRemote0OutEnabled() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().isRemoteOutEnabled[0];
#else
    return false;
#endif
}

void PreviewSound::SetIsRemote0OutEnabled(bool value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().isRemoteOutEnabled[0] != value);
    m_SoundController->GetParameters().isRemoteOutEnabled[0] = value;
#endif
}

bool PreviewSound::GetIsRemote1OutEnabled() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().isRemoteOutEnabled[1];
#else
    return false;
#endif
}

void PreviewSound::SetIsRemote1OutEnabled(bool value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().isRemoteOutEnabled[1] != value);
    m_SoundController->GetParameters().isRemoteOutEnabled[1] = value;
#endif
}

bool PreviewSound::GetIsRemote2OutEnabled() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().isRemoteOutEnabled[2];
#else
    return false;
#endif
}

void PreviewSound::SetIsRemote2OutEnabled(bool value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().isRemoteOutEnabled[2] != value);
    m_SoundController->GetParameters().isRemoteOutEnabled[2] = value;
#endif
}

bool PreviewSound::GetIsRemote3OutEnabled() const
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().isRemoteOutEnabled[3];
#else
    return false;
#endif
}

void PreviewSound::SetIsRemote3OutEnabled(bool value)
{
#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().isRemoteOutEnabled[3] != value);
    m_SoundController->GetParameters().isRemoteOutEnabled[3] = value;
#endif
}

bool PreviewSound::GetIsTrackMuteEnabled(u32 index) const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().tracksMute[index];
}

void PreviewSound::SetIsTrackMuteEnabled(u32 index, bool value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().tracksMute[index] != value);
    m_SoundController->GetParameters().tracksMute[index] = value;
}

f32 PreviewSound::GetTrackVolume(u32 index) const
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetParameters().tracksVolume[index];
}

void PreviewSound::SetTrackVolume(u32 index, f32 value)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    m_SoundController->SetParameterDirty(m_SoundController->GetParameters().tracksVolume[index] != value);
    m_SoundController->GetParameters().tracksVolume[index] = value;
}

bool PreviewSound::GetVariables(ctrl::internal::ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->GetVariables(pVariables);
}

bool PreviewSound::SetVariables(const ctrl::internal::ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL(m_SoundController);
    return m_SoundController->SetVariables(pVariables);
}

bool PreviewSound::ValidateSoundID()
{
    if (GetSoundID() == nw::snd::SoundArchive::INVALID_ID)
    {
        return false;
    }

    // 登録されているアーカイブに、m_SoundIDに一致するデータが含まれるかどうかを調べます。
    if (m_pPreviewSoundArchive == NULL ||
        !m_pPreviewSoundArchive->IsOpened() )
    {
        return false;
    }

    const char* itemLabel = m_pPreviewSoundArchive->GetItemLabel(GetSoundID());
    if (itemLabel != NULL)
    {
        return true;
    }

    // 見つからなかった
    return false;
}

#endif

} // nw::snd
} // nw
