﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <nw/ut.h>
#include <io/WaveRenderer.h>

#if defined( NW_USE_NINTENDO_SDK)
#include <nn/nn_Windows.h>
#elif defined( NW_PLATFORM_WIN32)
#include <Windows.h>
#endif

//#define NW_OUTPUT_WAVE_LOG

#ifdef NW_OUTPUT_WAVE_LOG
#include <nw/snd/fnd/io/sndfnd_File.h>
#include <nw/snd/fnd/io/sndfnd_FileStream.h>

#if defined(NW_PLATFORM_CAFE)
#include <nn/save.h>
#endif
#endif

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
using namespace nw::internal::winext;
#endif

namespace {

#ifdef NW_OUTPUT_WAVE_LOG
static nw::snd::internal::fnd::FileStream s_LogFile;

#if defined(NW_PLATFORM_CAFE)
static FSClient s_LogFSClient;
static FSCmdBlock s_LogFSCmdBlock;
#endif
#endif

template<typename TValue>
void CopySample(void* pDestination, const void* pSource)
{
    *static_cast<TValue*>(pDestination) = *static_cast<const TValue*>(pSource);
}

}

namespace nw {
namespace snd {

WaveRenderer::WaveRenderer() :
m_IsOpened(false),
m_VoiceCount(0),
m_BytesPerSample(0),
m_DataSize(0),
m_CurrentWaveBufferIndex(INVALID_INDEX),
m_CurrentWaveBufferValidSize(0),
m_TotalWrittenSize(0)
{
    for(u32 voiceIndex = 0; voiceIndex < MAX_VOICES; ++voiceIndex)
    {
        for(u32 bufferIndex = 0; bufferIndex < WAVE_BUFFER_COUNT; ++bufferIndex)
    {
            WaveBuffer& waveBufferContainer = m_WaveBufferContainers[voiceIndex][bufferIndex];

            waveBufferContainer.bufferAddress = m_WaveBuffer[voiceIndex][bufferIndex];
            waveBufferContainer.sampleLength = 0;
            waveBufferContainer.loopFlag = false;
        }
    }

#if defined(NW_OUTPUT_WAVE_LOG)
#if defined(NW_PLATFORM_CAFE)
    SAVEInit();

    FSAddClient(&s_LogFSClient, FS_RET_NO_ERROR);
    FSInitCmdBlock(&s_LogFSCmdBlock);
#endif
#endif
}

WaveRenderer::~WaveRenderer()
{
}

bool WaveRenderer::Open(const internal::fnd::WaveFormat& waveFormat, u32 dataSize)
{
    NW_ASSERT(!IsOpened());

    SampleFormat sampleFormat = SAMPLE_FORMAT_PCM_S16;

    switch(waveFormat.bitsPerSample)
    {
    case 8:
        sampleFormat = SAMPLE_FORMAT_PCM_S8;
        break;

    case 16:
        sampleFormat = SAMPLE_FORMAT_PCM_S16;
        break;

    case 32:
        sampleFormat = SAMPLE_FORMAT_PCM_S32;
        break;

    default:
        NW_LOG("[WaveRenderer] invalid bits per sample.\n");
        return false;
    }

    if(waveFormat.channels == 0 || MAX_VOICES < waveFormat.channels)
    {
        NW_LOG("[WaveRenderer] invalid channel count.\n");
        return false;
    }

    SoundSystem::SoundThreadScopedLock lock;

    for(u32 i = 0; i < waveFormat.channels; ++i)
    {
        nw::snd::internal::Voice& voice = m_Voices[i];

        if ( ! voice.AllocVoice(internal::Voice::PRIORITY_NODROP) )
        {
            break;
        }

        voice.SetVolume(1.f);
        voice.SetSampleRate(waveFormat.samplesPerSec);
        voice.SetSampleFormat(sampleFormat);

        SetOutMix(voice, waveFormat, i);

        voice.UpdateParam();

        m_VoiceCount++;
    }

    m_BytesPerSample = waveFormat.bitsPerSample / 8;
    m_DataSize = dataSize;
    m_CurrentWaveBufferIndex = 0;
    m_CurrentWaveBufferValidSize = 0;
    m_TotalWrittenSize = 0;
    m_IsOpened = true;

#if defined(NW_OUTPUT_WAVE_LOG)
#if defined(NW_PLATFORM_CAFE)
    if(s_LogFile.IsOpened())
    {
        s_LogFile.Close();
    }

    internal::fnd::File::Open(
        &s_LogFile,
        &s_LogFSClient,
        &s_LogFSCmdBlock,
        "/vol/save/log.dat",
        internal::fnd::File::ACCESS_MODE_WRITE);
#endif
#endif

    return true;
}

void WaveRenderer::Close()
{
    if(!IsOpened())
    {
        return;
    }

    // Write() 処理途中でも中止可能にするために、ロック前にフラグを設定します。
    m_IsOpened = false;

    // Write() と Close() が同時に走らないようにロックします。
    nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_CloseLock);

    {
        SoundSystem::SoundThreadScopedLock lock;

        for(u32 i = 0; i < m_VoiceCount; ++i)
        {
            m_Voices[i].SetState(internal::VOICE_STATE_STOP);
            m_Voices[i].Free();
        }
    }

    for(u32 bufferIndex = 0; bufferIndex < WAVE_BUFFER_COUNT; ++bufferIndex)
    {
        for(u32 voiceIndex = 0; voiceIndex < MAX_VOICES; ++voiceIndex)
        {
            WaveBuffer& waveBufferContainer = m_WaveBufferContainers[voiceIndex][bufferIndex];
            waveBufferContainer.sampleLength = 0;
        }
    }

    m_VoiceCount = 0;
    m_BytesPerSample = 0;
    m_DataSize = 0;
    m_CurrentWaveBufferIndex = INVALID_INDEX;
    m_CurrentWaveBufferValidSize = 0;
    m_TotalWrittenSize = 0;

#if defined(NW_OUTPUT_WAVE_LOG)
#if defined(NW_PLATFORM_CAFE)
    if(s_LogFile.IsOpened())
    {
        s_LogFile.Close();
    }
#endif
#endif
}

u32 WaveRenderer::Write(const void* pBuffer, u32 size)
{
    // Write() と Close() が同時に走らないようにロックします。
    nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_CloseLock);

    if(!IsOpened())
    {
        return size;
    }

    u32 frameSize = m_BytesPerSample * m_VoiceCount;
    u32 sizePerVoice = size / m_VoiceCount;

    if(sizePerVoice == 0)
    {
        return size;
    }

    for(u32 i = 0; i < size; i += frameSize)
    {
        if(!IsOpened())
        {
            break;
        }

        if(m_TotalWrittenSize >= m_DataSize)
        {
            break;
        }

        if(size - i < frameSize)
        {
            NW_LOG(
                "[WaveRenderer] detect a partial wave frame. : size=%d, frameSize=%d, restSize=%d writtenSize=%d/%d\n",
                size, frameSize, size - i, m_TotalWrittenSize, m_DataSize);
            break;
        }

        WriteFrame(nw::ut::AddOffsetToPtr(pBuffer, i));
    }

#if defined(NW_OUTPUT_WAVE_LOG)
#if defined(NW_PLATFORM_CAFE)
    if(m_TotalWrittenSize >= m_DataSize)
    {
        s_LogFile.Close();
    }
#endif
#endif

    return size;
}

void WaveRenderer::EndOfStream()
{
    StartPreparedVoices(true);
}

bool WaveRenderer::IsOpened() const
{
    return m_IsOpened;
}

bool WaveRenderer::IsFinished() const
{
    if(!IsOpened())
    {
        return true;
    }

    for(u32 i = 0; i < WAVE_BUFFER_COUNT; ++i)
    {
        switch(m_WaveBufferContainers[0][i].status)
        {
        case WaveBuffer::STATUS_WAIT:
        case WaveBuffer::STATUS_PLAY:
            return false;
        }
    }

    return true;
}

void WaveRenderer::WriteFrame(const void* pBuffer)
{
    NW_ASSERT(m_CurrentWaveBufferIndex != INVALID_INDEX);
    NW_ASSERT(m_CurrentWaveBufferValidSize < WAVE_BUFFER_SIZE);
    NW_ASSERT(m_BytesPerSample > 0);
    NW_ASSERT(m_TotalWrittenSize < m_DataSize);

    COPY_SAMPLE_HANDLER copySample = GetCopySampleHandler();

    if(copySample == NULL)
    {
        return;
    }

    const u8* pCurrent = static_cast<const u8*>(pBuffer);

    for(u32 voiceIndex = 0; voiceIndex < m_VoiceCount; ++voiceIndex)
    {
        copySample(m_WaveBuffer[voiceIndex][m_CurrentWaveBufferIndex] + m_CurrentWaveBufferValidSize, pCurrent);
        pCurrent += m_BytesPerSample;
    }

    m_CurrentWaveBufferValidSize += m_BytesPerSample;
    m_TotalWrittenSize += m_BytesPerSample * m_VoiceCount;

    NW_ASSERT(m_TotalWrittenSize <= m_DataSize);

    // 以下のいずれかの条件を満たしたら、Voice に追加します。
    // 1. データ終端に達した
    // 2. バッファの有効サイズが WAVE_BUFFER_DATA_SIZE を超えた
    //    ただし残りデータサイズが WAVE_BUFFER_SPARE_SIZE を下回ったら、
    //    WAVE_BUFFER_DATA_SIZE を超えて１バッファにまとめます。
    {
        if(m_CurrentWaveBufferValidSize == 0)
        {
            return;
        }

        if(m_TotalWrittenSize >= m_DataSize)
        {
            AppendWaveBuffer();
        }
        else if(
            m_CurrentWaveBufferValidSize >= WAVE_BUFFER_DATA_SIZE &&
            m_DataSize - m_TotalWrittenSize > WAVE_BUFFER_SPARE_SIZE)
        {
            AppendWaveBuffer();
        }
    }
}

void WaveRenderer::AppendWaveBuffer()
{
    NW_ASSERT(m_TotalWrittenSize <= m_DataSize);

    {
        SoundSystem::SoundThreadScopedLock lock;

        for(u32 voiceIndex = 0; voiceIndex < m_VoiceCount; ++voiceIndex)
        {
            WaveBuffer& waveBufferContainer = m_WaveBufferContainers[voiceIndex][m_CurrentWaveBufferIndex];
            waveBufferContainer.sampleLength = m_CurrentWaveBufferValidSize / m_BytesPerSample;

#ifdef NW_PRINT_LOG
            OSReport("Append [idx] %d [addr] %08x [len] %d [FreeBuffer] %d / %d\n",
                m_CurrentWaveBufferIndex,
                waveBufferContainer.bufferAddress,
                waveBufferContainer.sampleLength,
                GetFreeWaveBufferCount(),
                WAVE_BUFFER_COUNT);
#endif

#if defined(NW_OUTPUT_WAVE_LOG)
#if defined(NW_PLATFORM_CAFE)
            if(s_LogFile.IsOpened())
            {
                s_LogFile.Write(waveBufferContainer.bufferAddress, m_CurrentWaveBufferValidSize);
            }
#endif
#endif

            AXPrepareEfxData(const_cast<void*>(waveBufferContainer.bufferAddress), m_CurrentWaveBufferValidSize);
            m_Voices[voiceIndex].AppendWaveBuffer(&waveBufferContainer);
        }

        StartPreparedVoices(false);
    }

    m_CurrentWaveBufferIndex = INVALID_INDEX;
    m_CurrentWaveBufferValidSize = 0;

    // メインスレッド以外から呼び出されることを想定しているため、
    // 空きバッファができるまでブロックします。
    while(IsOpened())
    {
        u32 newBufferIndex = GetFreeWaveBufferIndex();

        if(newBufferIndex == INVALID_INDEX)
        {
#if defined( NW_PLATFORM_CAFE )
            OSSleepMilliseconds(100);
#elif defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
            Sleep(100);
#endif
            continue;
        }

        m_CurrentWaveBufferIndex = newBufferIndex;
        break;
    }
}

u32 WaveRenderer::GetFreeWaveBufferIndex()
{
    for(u32 i = 0; i < WAVE_BUFFER_COUNT; ++i)
    {
        WaveBuffer& waveBufferContainer = m_WaveBufferContainers[0][i];

        switch(waveBufferContainer.status)
        {
        case WaveBuffer::STATUS_FREE:
        case WaveBuffer::STATUS_DONE:
            return i;
        }
    }

    return INVALID_INDEX;
}

u32 WaveRenderer::GetFreeWaveBufferCount()
{
    u32 result = 0;

    for(u32 i = 0; i < WAVE_BUFFER_COUNT; ++i)
    {
        WaveBuffer& waveBufferContainer = m_WaveBufferContainers[0][i];

        switch(waveBufferContainer.status)
        {
        case WaveBuffer::STATUS_FREE:
        case WaveBuffer::STATUS_DONE:
            result++;
            break;
        }
    }

    return result;
}

void WaveRenderer::StartPreparedVoices(bool isImmediate)
{
    SoundSystem::SoundThreadScopedLock lock;

    if(m_Voices[0].GetState() == internal::VOICE_STATE_PLAY)
    {
        return;
    }

    if(GetFreeWaveBufferCount() == WAVE_BUFFER_COUNT)
    {
        return;
    }

    if(!isImmediate &&
        GetFreeWaveBufferCount() > WAVE_BUFFER_COUNT - PREPARE_BUFFER_COUNT)
    {
        return;
    }

    for(u32 voiceIndex = 0; voiceIndex < m_VoiceCount; ++voiceIndex)
    {
        if(m_Voices[voiceIndex].GetState() != internal::VOICE_STATE_PLAY)
        {
            m_Voices[voiceIndex].SetState(internal::VOICE_STATE_PLAY);
        }
    }
}

WaveRenderer::COPY_SAMPLE_HANDLER WaveRenderer::GetCopySampleHandler() const
{
    switch(m_BytesPerSample)
    {
    case 1:
        return CopySample<s8>;

    case 2:
        return CopySample<s16>;

    case 4:
        return CopySample<s32>;
    }

    NW_ASSERT(false);
    return NULL;
}

} // snd
} // nw
