﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <nw/types.h>

#include <core/TaskThread.h>

//#define TASKTHREAD_DEBUG

namespace nw {
namespace snd {

//---------------------------------------------------------------------------

void TaskThread::MainHandler::ThreadHandlerProc()
{
#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] Begin MainHandlerProc.\n");
#endif

    {
        nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_Owner.m_stateLock);
        m_IsRunning = true;
    }

    while(!m_IsExiting)
    {
        Task* pTask = m_Owner.PopTask(true);

        if(pTask == NULL)
        {
            break;
        }

        pTask->Invoke();
    }

    {
        nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_Owner.m_stateLock);
        m_IsRunning = false;
        m_Owner.OnExitThread();
    }

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] End MainHandlerProc.\n");
#endif
}

//---------------------------------------------------------------------------

TaskThread::TaskThread() :
m_MainHandler(*this),
m_TaskThread(&m_MainHandler),
m_StopTask(*this),
m_IsStopping(false)
{
}

//---------------------------------------------------------------------------

void TaskThread::Start(s32 priority)
{
    if(m_MainHandler.IsRunning() ||
        !m_MainHandler.IsExiting())
    {
        return;
    }

    if(m_IsStopping)
    {
        WaitStop();
    }
    NW_ASSERT(!m_IsStopping);

    // キューの中身を空にするために再初期化します。
    m_MessageQueue.Finalize();
    m_MessageQueue.Initialize(m_OSMessages, sizeof(m_OSMessages) / sizeof(m_OSMessages[0]));

    m_MainHandler.Reset();

    nw::ut::Thread::CreateArg args;
    args.priority = priority;

#ifdef NW_PLATFORM_CAFE
    args.stackBase = m_TaskThreadStack;
    args.stackSize = sizeof(m_TaskThreadStack);
#endif

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] Start.\n");
#endif

    bool result = m_TaskThread.Create(args);

    if(!result)
    {
#ifdef TASKTHREAD_DEBUG
        NW_LOG("[TaskThread] failed to Start.\n");
#endif
        return;
    }

    m_TaskThread.Resume();
}

void TaskThread::Stop()
{
    nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_stateLock);

    if(!m_MainHandler.IsRunning() || m_IsStopping)
    {
        return;
    }

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] Register StopTask.\n");
#endif

    // 残りタスクを処理するために、StopTaskを追加登録します。
    m_IsStopping = true;
    RegisterTask(m_StopTask);

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] Registered StopTask.\n");
#endif
}

void TaskThread::StopImmediately()
{
    {
        nw::ut::ScopedLock<nw::ut::CriticalSection> lock(m_stateLock);

        if(!m_MainHandler.IsRunning())
        {
            return;
        }

#ifdef TASKTHREAD_DEBUG
        NW_LOG("[TaskThread] StopImmediately.\n");
#endif

        // 残りタスクをキャンセルするために、StopTask::Invoke() します。
        m_IsStopping = true;
        m_StopTask.Invoke();
        m_MessageQueue.Send(NULL, true);
    }

    m_TaskThread.Join();

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] Succeeded to StopImmediately.\n");
#endif
}

void TaskThread::WaitStop()
{
    if(!m_MainHandler.IsRunning())
    {
        return;
    }

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] WaitStop.\n");
#endif

    m_TaskThread.Join();

#ifdef TASKTHREAD_DEBUG
    NW_LOG("[TaskThread] Succeeded to WaitStop.\n");
#endif
}

void TaskThread::RegisterTask(Task& task)
{
    if(m_MainHandler.IsExiting())
    {
        return;
    }

    m_MessageQueue.Send(&task, false);
}

TaskThread::Task* TaskThread::PopTask(bool isBlocking)
{
    Task* pTask = NULL;

    if(!m_MessageQueue.Recv(reinterpret_cast<nw::ut::MessageQueue::MessageType*>(&pTask), isBlocking))
    {
        return NULL;
    }

    return pTask;
}

void TaskThread::OnExitThread()
{
    m_IsStopping = false;
    m_MessageQueue.Finalize();
}

} // namespace snd
} // namespace nw
