﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <nw/g3d.h>
#include <nw/dev.h>
#include "anim/ModelUtility.h"
#include <cafe/gfd.h>

namespace nw {
namespace snd {

//------------------------------------------------------------------------------
// ※ グラフィックスに関する処理は、g3dのデモを参考にしてください。
//------------------------------------------------------------------------------

// 使用する頂点属性の名前を列挙します。
// ここでは配列中のインデックスを explicit attribute location と一致させています。
const char* s_AttribName[] = { "_p0", "_n0", "_u0", "_i0", "_w0" };


// Uniform Block の ID と Uniform の ID を列挙します。
//const char* s_BlockId[] = { "Material", "Shape", "Skeleton", "View" };

const int s_NumAttrib   = sizeof(s_AttribName) / sizeof(char*);


// 使用するサンプラーの名前と ID を列挙します。
// 事前にシェーダを関連付けたマテリアルは ResSamplerAssign に名前と ID の関連付けを持っています。
const char* s_SamplerName[] = { "_a0" };
const char* s_SamplerId[]   = { "albedoTex" };

// Uniform Block の ID と Uniform の ID を列挙します。
const char* s_BlockId[] = { "Material", "Shape", "Skeleton", "View" };

const int s_NumSampler      = sizeof(s_SamplerName) / sizeof(char*);
const int s_NumBlock        = sizeof(s_BlockId) / sizeof(char*);

// シェーダリソース
#if NW_G3D_GFXSHADER_SETUPARG_CONSTRUCTOR
nw::g3d::GfxShader::SetupArg gShaderSetupArg;
#else
nw::g3d::GfxShader::SetupArg gShaderSetupArg = { NULL, NULL, NULL };
#endif
void* gG3dShaderSrc[] = {NULL, NULL};

#ifdef NW_PLATFORM_CAFE
void* gShaderResource = NULL;
#endif

//------------------------------------------------------------------------------
void InitializeG3dShaderResource( nw::ut::IAllocator* allocator )
{
    // 決めうちシェーダーのセットアップ
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。

    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path        = "g3d/demo/shader/simple_vertex.glsl";
        loadArg.allocator   = allocator;
        loadArg.alignment   = 128;
        void* ptr = fileSystem->Load( loadArg );
        NW_ASSERT( loadArg.readSize > 0 );

        gG3dShaderSrc[0] = allocator->Alloc( loadArg.readSize + 1 );
        memset( gG3dShaderSrc[0], 0, loadArg.readSize + 1 );
        memcpy( gG3dShaderSrc[0], ptr, loadArg.readSize );

        fileSystem->Unload( loadArg, (u8 *)ptr );
    }

    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path        = "g3d/demo/shader/simple_fragment.glsl";
        loadArg.allocator   = allocator;
        loadArg.alignment   = 128;
        void* ptr = fileSystem->Load( loadArg );
        NW_ASSERT( loadArg.readSize > 0 );

        gG3dShaderSrc[1] = allocator->Alloc( loadArg.readSize + 1 );
        memset( gG3dShaderSrc[1], 0, loadArg.readSize + 1 );
        memcpy( gG3dShaderSrc[1], ptr, loadArg.readSize );

        fileSystem->Unload( loadArg, (u8 *)ptr );
    }

    gShaderSetupArg.pVertexShader   = static_cast<char*>(gG3dShaderSrc[0]);
    gShaderSetupArg.pGeometryShader = NULL;
    gShaderSetupArg.pFragmentShader = static_cast<char*>(gG3dShaderSrc[1]);

#else

    nw::dev::FileDevice::LoadArg loadArg;
    loadArg.path        = "g3d/demo/shader/simple.gsh";
    loadArg.allocator   = allocator;
    loadArg.alignment   = GX2_SHADER_ALIGNMENT;
    gShaderResource = fileSystem->Load( loadArg );
    NW_ASSERT( loadArg.readSize > 0 );

    NW_ASSERTMSG(GFDGetAlignMode(gShaderResource) == GFD_ALIGN_MODE_ENABLE,
        ".gsh must be compiled with '-align'.\n    Path: %s\n", loadArg.path);

    // "-align" オプションをつけてコンパイルしている前提。
    gShaderSetupArg.pVertexShader   = GFDGetVertexShaderPointer(0, gShaderResource);
    gShaderSetupArg.pGeometryShader = GFDGetGeometryShaderPointer(0, gShaderResource);
    gShaderSetupArg.pFragmentShader = GFDGetPixelShaderPointer(0, gShaderResource);
#endif
}

//------------------------------------------------------------------------------
void GetG3dShaderResource( nw::g3d::GfxShader::SetupArg* setupArg )
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    setupArg->pVertexShader     = gShaderSetupArg.pVertexShader;
    setupArg->pGeometryShader   = NULL;
    setupArg->pFragmentShader   = gShaderSetupArg.pFragmentShader;
#else
    setupArg->pVertexShader     = gShaderSetupArg.pVertexShader;
    setupArg->pGeometryShader   = NULL;
    setupArg->pFragmentShader   = gShaderSetupArg.pFragmentShader;
#endif
}

//------------------------------------------------------------------------------
void FinalizeG3dShaderResource( nw::ut::IAllocator* allocator )
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    allocator->Free( gG3dShaderSrc[0] );
    allocator->Free( gG3dShaderSrc[1] );
#else
    allocator->Free( gShaderResource );
#endif
}

//------------------------------------------------------------------------------
void SetupSimpleShader( SimpleShader& shader, nw::ut::IAllocator* allocator )
{
    // シェーダリソースを取得
#if NW_G3D_GFXSHADER_SETUPARG_CONSTRUCTOR
    nw::g3d::GfxShader::SetupArg arg;
#else
    nw::g3d::GfxShader::SetupArg arg = { NULL, NULL, NULL };
#endif
    GetG3dShaderResource( &arg );

    shader.Setup(arg);
    shader.InitSamplerLocations( s_NumSampler, s_SamplerId );
    shader.InitBlockLocations( s_NumBlock, s_BlockId );
    shader.InitViewBlock( allocator );
}

//------------------------------------------------------------------------------
nw::g3d::ModelObj* BuildModel( nw::ut::IAllocator* allocator, nw::g3d::res::ResModel* rMdl )
{
    NW_ASSERT_NOT_NULL(rMdl);

    nw::g3d::ModelObj* mdl =
        new(allocator->Alloc(sizeof(nw::g3d::ModelObj))) nw::g3d::ModelObj();
    NW_ASSERT_NOT_NULL(mdl);

    nw::g3d::ModelObj::InitArg mdlArg(rMdl);
    size_t bufferSize = nw::g3d::ModelObj::CalcBufferSize(mdlArg);
    bool result = mdl->Init(mdlArg, allocator->Alloc(bufferSize, nw::g3d::ModelObj::BUFFER_ALIGNMENT), bufferSize);
    NW_ASSERT(result);

    return mdl;
}

//------------------------------------------------------------------------------
void DestroyModel( nw::ut::IAllocator* allocator, nw::g3d::ModelObj* mdl )
{
    NW_ASSERT_NOT_NULL(mdl);

    void* blockBuffer = mdl->GetBlockBufferPtr();
    if (blockBuffer != NULL)
    {
        allocator->Free(blockBuffer);
    }

    allocator->Free(mdl->GetBufferPtr());
    allocator->Free(mdl);
}

//------------------------------------------------------------------------------
nw::g3d::SkeletalAnimObj* BuildSklAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResSkeletalAnim* rSklAnim, int maxSklCount )
{
    NW_ASSERT_NOT_NULL(rSklAnim);

    nw::g3d::SkeletalAnimObj* anim =
        new(allocator->Alloc(sizeof(nw::g3d::SkeletalAnimObj))) nw::g3d::SkeletalAnimObj();
    NW_ASSERT_NOT_NULL(rSklAnim);

    nw::g3d::SkeletalAnimObj::InitArg animArg;
    animArg.SetMaxBoneCount(maxSklCount);
    animArg.SetMaxBoneAnimCount(rSklAnim->GetBoneAnimCount());
    animArg.SetMaxCurveCount(0);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::SkeletalAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, allocator->Alloc(size, nw::g3d::SkeletalAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rSklAnim);

    return anim;
}

//------------------------------------------------------------------------------
nw::g3d::ShaderParamAnimObj* BuildParamAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResShaderParamAnim* rParamAnim, int maxMatCount )
{
    NW_ASSERT_NOT_NULL(rParamAnim);

    nw::g3d::ShaderParamAnimObj* anim =
        new(allocator->Alloc(sizeof(nw::g3d::ShaderParamAnimObj))) nw::g3d::ShaderParamAnimObj();
    NW_ASSERT_NOT_NULL(anim);

    nw::g3d::ShaderParamAnimObj::InitArg animArg;
    animArg.SetMaxMatCount(maxMatCount);
    animArg.SetMaxMatAnimCount(rParamAnim->GetMatAnimCount());
    animArg.SetMaxParamAnimCount(rParamAnim->GetParamAnimCount());
    animArg.SetMaxCurveCount(rParamAnim->GetCurveCount());

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::ShaderParamAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, allocator->Alloc(size, nw::g3d::ShaderParamAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rParamAnim);

    return anim;
}

//------------------------------------------------------------------------------
nw::g3d::TexPatternAnimObj* BuildTexAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResTexPatternAnim* rTexAnim, int maxMatCount )
{
    NW_ASSERT_NOT_NULL(rTexAnim);

    nw::g3d::TexPatternAnimObj* anim =
        new(allocator->Alloc(sizeof(nw::g3d::TexPatternAnimObj))) nw::g3d::TexPatternAnimObj();
    NW_ASSERT_NOT_NULL(anim);

    nw::g3d::TexPatternAnimObj::InitArg animArg;
    animArg.SetMaxMatCount(maxMatCount);
    animArg.SetMaxMatAnimCount(rTexAnim->GetMatAnimCount());
    animArg.SetMaxPatAnimCount(rTexAnim->GetPatAnimCount());
    animArg.SetMaxCurveCount(0);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::TexPatternAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, allocator->Alloc(size, nw::g3d::TexPatternAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rTexAnim);

    return anim;
}

//------------------------------------------------------------------------------
nw::g3d::VisibilityAnimObj* BuildVisAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResVisibilityAnim* rVisAnim, int maxCount )
{
    NW_ASSERT_NOT_NULL(rVisAnim);

    nw::g3d::VisibilityAnimObj* anim =
        new(allocator->Alloc(sizeof(nw::g3d::VisibilityAnimObj))) nw::g3d::VisibilityAnimObj();
    NW_ASSERT_NOT_NULL(anim);

    nw::g3d::VisibilityAnimObj::InitArg animArg;
    animArg.SetMaxBoneCount(maxCount);
    animArg.SetMaxMatCount(maxCount);
    animArg.SetMaxBoneAnimCount(rVisAnim->GetAnimCount());
    animArg.SetMaxMatAnimCount(rVisAnim->GetAnimCount());
    animArg.SetMaxCurveCount(0);

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::VisibilityAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, allocator->Alloc(size, nw::g3d::VisibilityAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rVisAnim);

    return anim;
}

//------------------------------------------------------------------------------
nw::g3d::ShapeAnimObj* BuildShapeAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResShapeAnim* rShapeAnim, int maxShapeCount )
{
    NW_ASSERT_NOT_NULL(rShapeAnim);

    nw::g3d::ShapeAnimObj* anim =
        new(allocator->Alloc(sizeof(nw::g3d::ShapeAnimObj))) nw::g3d::ShapeAnimObj();
    NW_ASSERT_NOT_NULL(anim);

    nw::g3d::ShapeAnimObj::InitArg animArg(maxShapeCount, rShapeAnim->GetVertexShapeAnimCount(), rShapeAnim->GetKeyShapeAnimCount());

    // アニメのリソースをインスタンスに固定する。
    // インスタンスを共有する場合は動的に SetResource を呼び出す。
    size_t size = nw::g3d::ShapeAnimObj::CalcBufferSize(animArg);
    anim->Init(animArg, allocator->Alloc(size, nw::g3d::ShapeAnimObj::BUFFER_ALIGNMENT), size);
    anim->SetResource(rShapeAnim);

    return anim;
}

//------------------------------------------------------------------------------
void DestroyModelAnim( nw::ut::IAllocator* allocator, nw::g3d::ModelAnimObj* anim )
{
    if (anim == NULL) { return; }

    void* buffer = anim->GetBufferPtr();
    if (buffer != NULL)
    {
        allocator->Free(buffer);
    }

    allocator->Free(anim);
}

//------------------------------------------------------------------------------
int BindShader( nw::ut::IAllocator* allocator, const SimpleShader& shader, nw::g3d::ModelObj* model, const char** blockId )
{
    for (int idxMat = 0, numMat = model->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = model->GetResource()->GetMaterial(idxMat);

        // Uniform Block のサイズを指定します。
        u32 blockSize = shader.GetFragmentBlockSize(blockId[SimpleShader::MATERIAL_BLOCK]);
        pResMaterial->SetRawParamSize(blockSize);
    }

    // Uniform Block のバッファは所定のアライメントが必要です。
    size_t blockBufferSize = model->CalcBlockBufferSize();
    void* pBlockBuffer = allocator->Alloc(blockBufferSize, nw::g3d::ModelObj::BLOCK_BUFFER_ALIGNMENT);
    NW_ASSERT_NOT_NULL(pBlockBuffer);
    bool success = model->SetupBlockBuffer(pBlockBuffer, blockBufferSize);
    NW_ASSERT(success);

    return 0;
}

//------------------------------------------------------------------------------
void DestroyShader( nw::ut::IAllocator* allocator, SimpleShader& shader )
{
    shader.FinalizeViewBlock( allocator );
    shader.Cleanup();
}

//------------------------------------------------------------------------------
int BuildFetchShader( nw::ut::IAllocator* allocator, const nw::g3d::ModelObj* mdl, nw::g3d::GfxFetchShader** pFetchShaderArray )
{
    const nw::g3d::ResModel* pResModel =  mdl->GetResource();
    // フェッチシェーダは ResVertex と1対1で対応します。フェッチシェーダの数を減らすため、
    // 複数の ResShape が ResVertex を共有している場合があります。
    int numVertex = pResModel->GetVertexCount();
    size_t fetchShaderArraySize = sizeof(nw::g3d::GfxFetchShader) * numVertex;
    *pFetchShaderArray = static_cast<nw::g3d::GfxFetchShader*>(allocator->Alloc(fetchShaderArraySize));
    NW_ASSERT_NOT_NULL(pFetchShaderArray);

    // 単純化のため、s_NumAttrib 分のバッファをアライメントで切り上げて確保します。
    // 個々の ResVertex が使用する頂点属性数から正確にバッファサイズを計算することで
    // 必要なメモリを節約することもできます。
    nw::g3d::GfxFetchShader fetchShader;
    fetchShader.SetAttribCount(s_NumAttrib);
    fetchShader.CalcSize();
    // フェッチシェーダのバッファは GPU から参照されるため、所定のアライメントが必要です。
    size_t shaderSize = nw::g3d::Align(fetchShader.GetShaderSize(), GX2_SHADER_ALIGNMENT);
    void* shaderPtr = allocator->Alloc(shaderSize * numVertex, GX2_SHADER_ALIGNMENT);
    NW_ASSERT_NOT_NULL(shaderPtr);

    for (int idxVertex = 0; idxVertex < numVertex; ++idxVertex)
    {
        // 存在する頂点属性の数を調ます。正しい位置に終端インストラクションを書き込むため、
        // ここでは正確な頂点属性数を指定する必要があります。
        const nw::g3d::ResVertex* pResVertex = pResModel->GetVertex(idxVertex);
        const nw::g3d::ResVtxAttrib* pResVtxAttrib[SimpleShader::NUM_SHADER_ATTRIB];
        memset(pResVtxAttrib, 0, sizeof(pResVtxAttrib));
        int numAttribValid = 0;
        for (int idxAttrib = 0; idxAttrib < s_NumAttrib; ++idxAttrib)
        {
            const char* name = s_AttribName[idxAttrib];
            pResVtxAttrib[idxAttrib] = pResVertex->GetVtxAttrib(name);
            if (pResVtxAttrib[idxAttrib])
            {
                ++numAttribValid;
            }
        }

        // シェーダバッファの設定。
        nw::g3d::GfxFetchShader* pFetchShader = &(*pFetchShaderArray)[idxVertex];
        new(pFetchShader) nw::g3d::GfxFetchShader();
        pFetchShader->Setup();
        pFetchShader->SetAttribCount(numAttribValid);
        pFetchShader->CalcSize();
        pFetchShader->SetShaderPtr(shaderPtr);
        pFetchShader->SetDefault();
        shaderPtr = nw::g3d::AddOffset(shaderPtr, shaderSize);

        // 個別の頂点属性の設定。
        numAttribValid = 0;
        for (int idxAttrib = 0; idxAttrib < s_NumAttrib; ++idxAttrib)
        {
            const nw::g3d::ResVtxAttrib* pAttrib = pResVtxAttrib[idxAttrib];
            if (pAttrib)
            {
                int index = numAttribValid++;
                int bufferIndex = pAttrib->GetBufferIndex();
                // ここでは単純化のため、 bufferIndex が GX2_MAX_ATTRIB_BUFFERS より
                // 小さいという制約を課します。GX2_MAX_ATTRIB_BUFFERS を超える場合は
                // 空いているインデックスに詰める処理を実装する必要があります。
                NW_ASSERT(bufferIndex < GX2_MAX_ATTRIB_BUFFERS);
                const nw::g3d::ResBuffer* pResBuffer = pResVertex->GetVtxBuffer(bufferIndex);

                pFetchShader->SetLocation(index, idxAttrib);
                pFetchShader->SetBufferSlot(index, bufferIndex);
                pFetchShader->SetFormat(index, pAttrib->GetFormat());
                pFetchShader->SetOffset(index, pAttrib->GetOffset());
                pFetchShader->SetVertexBuffer(index, pResBuffer->GetGfxBuffer());
            }
        }
        pFetchShader->DCFlush(); // GPU が参照するメモリの CPU キャッシュを吐き出します。
    }

    return numVertex;
}

//------------------------------------------------------------------------------
void DestroyFetchShader( nw::ut::IAllocator* allocator, nw::g3d::GfxFetchShader** pFetchShaderArray, int numShader )
{
    for (int i = 0; i < numShader; ++i)
    {
        nw::g3d::GfxFetchShader& fetchShader = (*pFetchShaderArray)[i];
        fetchShader.Cleanup();
    }

    if (numShader > 0)
    {
        nw::g3d::GfxFetchShader& fetchShader = (*pFetchShaderArray)[0];
        void* pShaderPtr = fetchShader.GetShaderPtr();
        allocator->Free(pShaderPtr);
        allocator->Free(*pFetchShaderArray);
        *pFetchShaderArray = NULL;
    }
}

//---------------------------------------------------------------------------
// シェーダ

//------------------------------------------------------------------------------
void SimpleShader::InitSamplerLocations( int numSampler, const char* name[] )
{
    this->mNumSampler = static_cast<u8>(numSampler);
    memset(mSamplerLocation, -1, sizeof(mSamplerLocation));
    for (int idxSampler = 0; idxSampler < numSampler; ++idxSampler)
    {
        const char* samplerName = name[idxSampler];
        Location& location = mSamplerLocation[idxSampler];
        location.vertex = GetVertexSamplerLocation(samplerName);
        location.geometry = GetGeometrySamplerLocation(samplerName);
        location.fragment = GetFragmentSamplerLocation(samplerName);
    }
}

//------------------------------------------------------------------------------
void SimpleShader::InitBlockLocations( int numBlock, const char* name[] )
{
    this->mNumBlock = static_cast<u8>(numBlock);
    memset(mBlockLocation, -1, sizeof(mBlockLocation));
    for (int idxBlock = 0; idxBlock < numBlock; ++idxBlock)
    {
        const char* blockName = name[idxBlock];
        Location& location = mBlockLocation[idxBlock];
        location.vertex = GetVertexBlockLocation(blockName);
        location.geometry = GetGeometryBlockLocation(blockName);
        location.fragment = GetFragmentBlockLocation(blockName);
    }
}


//------------------------------------------------------------------------------
void SimpleShader::InitViewBlock( nw::ut::IAllocator* allocator )
{
    void* pViewBlockBuffer = allocator->Alloc(
        sizeof(ViewBlock), GX2_UNIFORM_BLOCK_ALIGNMENT);
    mViewBlock.SetData(pViewBlockBuffer, sizeof(ViewBlock));
    mViewBlock.Setup();
}


//------------------------------------------------------------------------------
void SimpleShader::FinalizeViewBlock( nw::ut::IAllocator* allocator )
{
    if (void* pViewBlockBuffer = mViewBlock.GetData())
    {
        mViewBlock.Cleanup();
        allocator->Free(pViewBlockBuffer);
    }
}

//---------------------------------------------------------------------------
// アニメーションセット

//---------------------------------------------------------------------------
SimpleAnimSet::SimpleAnimSet()
{
    mIsInitialized      = false;
    mResFile            = NULL;
}

//---------------------------------------------------------------------------
SimpleAnimSet::~SimpleAnimSet()
{
}

//---------------------------------------------------------------------------
bool SimpleAnimSet::Initialize( void* binaryResource, const nw::g3d::ModelObj& modelObj, nw::ut::IAllocator* allocator )
{
    NW_ASSERT(nw::g3d::ResFile::IsValid( binaryResource ));

    if (mIsInitialized)
    {
        return true;
    }

    mResFile = nw::g3d::ResFile::ResCast( binaryResource );
    mResFile->Setup();

    s32 sklAnimCount        = mResFile->GetSkeletalAnimCount();
    s32 paramAnimCount      = mResFile->GetShaderParamAnimCount();
    s32 colorAnimCount      = mResFile->GetColorAnimCount();
    s32 srtAnimCount        = mResFile->GetTexSrtAnimCount();
    s32 texAnimCount        = mResFile->GetTexPatternAnimCount();
    s32 boneVisAnimCount    = mResFile->GetBoneVisAnimCount();
    s32 matVisAnimCount     = mResFile->GetMatVisAnimCount();
    s32 shapeAnimCount      = mResFile->GetShapeAnimCount();

    mModelAnimObjArray = ModelAnimObjArray( allocator );

    s32 boneCount   = modelObj.GetSkeleton()->GetBoneCount();
    s32 matCount    = modelObj.GetMaterialCount();
    s32 shapeCount  = modelObj.GetShapeCount();

    for (s32 i = 0; i < sklAnimCount; ++i)
    {
        nw::g3d::SkeletalAnimObj* anim = BuildSklAnim( allocator, mResFile->GetSkeletalAnim(i), boneCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < paramAnimCount; ++i)
    {
        nw::g3d::ShaderParamAnimObj* anim = BuildParamAnim( allocator, mResFile->GetShaderParamAnim(i), matCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < colorAnimCount; ++i)
    {
        nw::g3d::ShaderParamAnimObj* anim = BuildParamAnim( allocator, mResFile->GetColorAnim(i), matCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < srtAnimCount; ++i)
    {
        nw::g3d::ShaderParamAnimObj* anim = BuildParamAnim( allocator, mResFile->GetTexSrtAnim(i), matCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < texAnimCount; ++i)
    {
        nw::g3d::TexPatternAnimObj* anim = BuildTexAnim( allocator, mResFile->GetTexPatternAnim(i), matCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < boneVisAnimCount; ++i)
    {
        nw::g3d::VisibilityAnimObj* anim = BuildVisAnim( allocator, mResFile->GetBoneVisAnim(i), boneCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < matVisAnimCount; ++i)
    {
        nw::g3d::VisibilityAnimObj* anim = BuildVisAnim( allocator, mResFile->GetMatVisAnim(i), matCount );
        mModelAnimObjArray.PushBack( anim );
    }
    for (s32 i = 0; i < shapeAnimCount; ++i)
    {
        nw::g3d::ShapeAnimObj* anim = BuildShapeAnim( allocator, mResFile->GetShapeAnim(i), shapeCount );
        mModelAnimObjArray.PushBack( anim );
    }

    mIsInitialized = true;

    return true;
}

//---------------------------------------------------------------------------
void SimpleAnimSet::Finalize( nw::ut::IAllocator* allocator )
{
    s32 animCount = mModelAnimObjArray.Size();

    for (s32 i = 0; i < animCount; ++i)
    {
        DestroyModelAnim(allocator, mModelAnimObjArray.Back());
        mModelAnimObjArray.PopBack();
    }

    mIsInitialized = false;
}

//---------------------------------------------------------------------------
void SimpleAnimSet::Calc( nw::g3d::ModelObj* modelObj )
{
    NW_ASSERT_NOT_NULL(modelObj);

    for (int i = 0; i < mModelAnimObjArray.Size(); ++i)
    {
        GetModelAnimObj(i)->Calc();
        GetModelAnimObj(i)->ApplyTo(modelObj);
    }
}

//---------------------------------------------------------------------------
nw::g3d::ModelAnimObj* SimpleAnimSet::GetModelAnimObj(s32 index)
{
    NW_ASSERT(index >= 0 && index < AnimCount());
    return mModelAnimObjArray.at(index);
}

//---------------------------------------------------------------------------
const nw::g3d::ModelAnimObj* SimpleAnimSet::GetModelAnimObj(s32 index) const
{
    NW_ASSERT(index >= 0 && index < AnimCount());
    return mModelAnimObjArray.at(index);
}

//---------------------------------------------------------------------------
s32 SimpleAnimSet::AnimCount() const
{
    return mModelAnimObjArray.Size();
}

//---------------------------------------------------------------------------
void SimpleAnimSet::SetAnimationFrame(f32 frame)
{
    for (int i = 0; i < mModelAnimObjArray.Size(); ++i)
    {
        GetModelAnimObj(i)->GetFrameCtrl().SetFrame(frame);
    }
}

//---------------------------------------------------------------------------
f32 SimpleAnimSet::GetAnimationFrame() const
{
    f32 maxAnimationFrame = 0.0f;
    for (int i = 0; i < mModelAnimObjArray.Size(); ++i)
    {
        f32 animationFrame = GetModelAnimObj(i)->GetFrameCtrl().GetFrame();

        if (maxAnimationFrame < animationFrame)
        {
            maxAnimationFrame = animationFrame;
        }
    }

    return maxAnimationFrame;
}

//---------------------------------------------------------------------------
f32 SimpleAnimSet::GetAnimationFrameCount() const
{

    f32 maxAnimationFrameCount = 0.0f;
    for (int i = 0; i < mModelAnimObjArray.Size(); ++i)
    {
        f32 startFrame = GetModelAnimObj(i)->GetFrameCtrl().GetStartFrame();
        f32 endFrame   = GetModelAnimObj(i)->GetFrameCtrl().GetEndFrame();
        f32 animationFrameCount = endFrame - startFrame;

        if (maxAnimationFrameCount < animationFrameCount)
        {
            maxAnimationFrameCount = animationFrameCount;
        }
    }

    return maxAnimationFrameCount;
}

//---------------------------------------------------------------------------
// モデル

//---------------------------------------------------------------------------
SimpleModel::SimpleModel()
{
    mModelObj           = NULL;
    mResFile            = NULL;
    mNumFetchShader     = 0;
    mFetchShaderArray   = NULL;
    mNumShapeBlock      = 0;
    mShapeBlockArray    = NULL;
    mAnimSet            = NULL;
    mShader             = NULL;

    mDrawMatrix.Identity();
}

//---------------------------------------------------------------------------
SimpleModel::~SimpleModel()
{
}

//---------------------------------------------------------------------------
bool SimpleModel::Initialize( void* modelResource, nw::ut::IAllocator* allocator )
{
    NW_ASSERT(nw::g3d::ResFile::IsValid( modelResource ));

    mResFile = nw::g3d::ResFile::ResCast( modelResource );
    mResFile->Setup();

    mModelObj = BuildModel( allocator, mResFile->GetModel(0));

    // FetchShader 生成
    mNumFetchShader = BuildFetchShader( allocator, mModelObj, &mFetchShaderArray);

    return true;
}

//---------------------------------------------------------------------------
void SimpleModel::Finalize( nw::ut::IAllocator* allocator )
{
    DestroyFetchShader( allocator, &mFetchShaderArray, mNumFetchShader);

    if ( mModelObj )
    {
        DestroyModel( allocator, mModelObj);
    }
}

//---------------------------------------------------------------------------
void SimpleModel::SetShader( SimpleShader& shader, nw::ut::IAllocator* allocator )
{
    // シェーダをモデルにバインドします。
    BindShader( allocator, shader, mModelObj, s_BlockId );
    mShader = &shader;
}

//---------------------------------------------------------------------------
void SimpleModel::SetAnimationSet( SimpleAnimSet& animSet )
{
    mAnimSet = &animSet;

    for (s32 i = 0; i < animSet.AnimCount(); ++i)
    {
        animSet.GetModelAnimObj(i)->Bind( mModelObj );
    }
}

//---------------------------------------------------------------------------
void SimpleModel::Calc()
{
    if (mModelObj == NULL)
    {
        return;
    }

    if (mAnimSet != NULL)
    {
        mAnimSet->Calc(mModelObj);
    }
    mModelObj->CalcWorld( mDrawMatrix );
}

//---------------------------------------------------------------------------
void SimpleModel::SetDrawMatrix( nw::g3d::Mtx34& drawMatrix )
{
    mDrawMatrix = drawMatrix;
}

//---------------------------------------------------------------------------
void SimpleModel::SetAnimationFrame(f32 frame)
{
    if (mAnimSet == NULL)
    {
        return;
    }

    mAnimSet->SetAnimationFrame(frame);
}

//---------------------------------------------------------------------------
void SimpleModel::CalcBlock( const nw::g3d::math::Mtx34* viewMtx, const nw::g3d::math::Mtx44* projMtx )
{
    if (mModelObj == NULL)
    {
        return;
    }

    // GPU 待ちが必要な計算
    mModelObj->CalcMtxBlock();
    mModelObj->CalcMaterial();
    mModelObj->CalcShape();
    mModelObj->CalcView(0, *viewMtx);

    // モデルのマテリアルに混載した領域にプログラムから値を差し込む場合、
    // 所定のエンディアンで書き込んだ上で CPU キャッシュを吐き出します。
    for (int idxMat = 0, numMat = mModelObj->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::MaterialObj* pMaterialObj = mModelObj->GetMaterial(idxMat);
        nw::g3d::GfxBuffer& matBlock = pMaterialObj->GetMatBlock();
        float colorOffset[] = { 0.0f, 0.0f, 0.0f };
        nw::g3d::Copy32<true>(matBlock.GetData(), colorOffset, sizeof(colorOffset) >> 2);
        matBlock.DCFlush();
    }

    // ビュー用の Uniform Block の更新。
    NW_ASSERT_NOT_NULL(mShader);
    nw::g3d::GfxBuffer& viewBlock = mShader->GetViewBlock();
    SimpleShader::ViewBlock* pViewBlockBuffer = static_cast<SimpleShader::ViewBlock*>(viewBlock.GetData());
    nw::g3d::Copy32<true>(&pViewBlockBuffer->projMtx, projMtx, sizeof(*projMtx) >> 2);
    nw::g3d::Copy32<true>(&pViewBlockBuffer->cameraMtx, viewMtx, sizeof(*viewMtx) >> 2);
    viewBlock.DCFlush();
}


//---------------------------------------------------------------------------
void SimpleModel::Draw()
{
    if (mModelObj == NULL)
    {
        return;
    }

    NW_ASSERT_NOT_NULL(mShader);

    // ここではソートやカリング、同一の描画ステートの省略などは行っていません。
    // アプリケーションの仕様にとって最適な方法で描画処理を行うことを推奨します。
    for (int idxShape = 0, numShape = mModelObj->GetShapeCount();
        idxShape < numShape; ++idxShape)
    {
        if (!mModelObj->IsShapeVisible(idxShape)) // ビジビリティの制御を行います。
        {
            continue;
        }

        const nw::g3d::ShapeObj* pShapeObj = mModelObj->GetShape(idxShape);
        int idxMaterial = pShapeObj->GetMaterialIndex();
        const nw::g3d::MaterialObj* pMaterialObj = mModelObj->GetMaterial(idxMaterial);
        int idxVertex = pShapeObj->GetVertexIndex();
        const nw::g3d::GfxFetchShader* pFetchShader = &mFetchShaderArray[idxVertex];

        // 描画ステートのロード。
        const nw::g3d::ResRenderState* pRenderState = pMaterialObj->GetResRenderState();
        pRenderState->Load();

        // フェッチシェーダと頂点バッファのロード。
        pFetchShader->Load();

        for (int idxVtxBuf = 0, numVtxBuffer = pShapeObj->GetVertexBufferCount();
            idxVtxBuf < numVtxBuffer; ++idxVtxBuf)
        {
            const nw::g3d::GfxBuffer& vtxBuf = pShapeObj->GetVertexBuffer(idxVtxBuf);
            vtxBuf.LoadVertices(idxVtxBuf);
        }

        mShader->Load(); // シェーダのロード。

        // マテリアルの Uniform Block をロード。
        // ここではジオメトリシェーダの処理を省略しています。
        const nw::g3d::GfxBuffer& matBlock = pMaterialObj->GetMatBlock();
        if (matBlock.GetData())
        {
            const SimpleShader::Location& location =
                mShader->GetBlockLocation(SimpleShader::MATERIAL_BLOCK);
            if (location.vertex != -1)
            {
                matBlock.LoadVertexUniforms(location.vertex);
            }
            if (location.fragment != -1)
            {
                matBlock.LoadFragmentUniforms(location.fragment);
            }
        }

        const int viewIndex = 0; // ビューの番号です。
        // シェイプの Uniform Block をロード。
        // ここではジオメトリシェーダとフラグメントシェーダの処理を省略しています。
        const nw::g3d::GfxBuffer& shpBlock = pShapeObj->GetShpBlock(viewIndex);
        if (shpBlock.GetData())
        {
            const SimpleShader::Location& location =
                mShader->GetBlockLocation(SimpleShader::SHAPE_BLOCK);
            if (location.vertex != -1)
            {
                shpBlock.LoadVertexUniforms(location.vertex);
            }
        }

        const nw::g3d::SkeletonObj* pSkeletonObj = mModelObj->GetSkeleton();
        // スケルトンの Uniform Block をロード。
        // ここではジオメトリシェーダとフラグメントシェーダの処理を省略しています。
        const nw::g3d::GfxBuffer& mtxBlock = pSkeletonObj->GetMtxBlock();
        if (mtxBlock.GetData())
        {
            const SimpleShader::Location& location =
                mShader->GetBlockLocation(SimpleShader::SKELETON_BLOCK);
            if (location.vertex != -1)
            {
                mtxBlock.LoadVertexUniforms(location.vertex);
            }
        }
        // ビューの Uniform Block をロード。
        // ここではジオメトリシェーダとフラグメントシェーダの処理を省略しています。
        {
            const SimpleShader::Location& location =
                mShader->GetBlockLocation(SimpleShader::VIEW_BLOCK);
            if (location.vertex != -1)
            {
                mShader->GetViewBlock().LoadVertexUniforms(location.vertex);
            }
        }

        // サンプラーのロード。
        // ここでは頂点シェーダとジオメトリシェーダの処理を省略しています。
        for (int idxTex = 0; idxTex < s_NumSampler; ++idxTex)
        {
            const SimpleShader::Location& location = mShader->GetSamplerLocation(idxTex);
            const char* name = s_SamplerName[idxTex];
            int idxSampler = pMaterialObj->GetSamplerIndex(name);

            if (location.fragment != -1 && idxSampler != -1)
            {
                const nw::g3d::ResSampler* pSampler = pMaterialObj->GetResSampler(idxSampler);
                const nw::g3d::ResTexture* pTexture = pMaterialObj->GetResTexture(idxSampler);
                pSampler->GetGfxSampler()->LoadFragmentSampler(location.fragment);

                if (pTexture)
                {
                    pTexture->GetGfxTexture()->LoadFragmentTexture(location.fragment);
                }
            }
        }

        // 描画コール。
        pShapeObj->GetResMesh()->Draw();
    }
}

} // snd
} // nw
