﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#include <anim/AnimSoundTargetCafe.h>

#include <nw/types.h>
#include <nw/gfnd.h>
#include <nw/gfnd/gfnd_Graphics.h>


namespace {

//----------------------------------------------------------

u32 s_Counter = 0;// インスタンスの個数を管理する

// モデルのvisibilityをまとめて変更する
void SetModelVisible(nw::g3d::ModelObj* obj, bool isVisible)
{
    if (obj == NULL)
    {
        return;
    }

    const int idx_term = obj->GetSkeleton()->GetBoneCount();
    for (int i=0; i<idx_term; ++i)
    {
        obj->SetBoneVisibility(i, isVisible);
    }
}

} // anonymous namespace

namespace nw {
namespace snd {

//----------------------------------------------------------
AnimSoundTargetCafe* AnimSoundTargetCafe::Create(const CreateArg& arg)
{
    NW_ASSERT(s_Counter == 0); // 一つだけ生成を許す（コード的には複数行けるかもしれませんが）

    ++s_Counter;

    void* memory = arg.allocator->Alloc(sizeof(AnimSoundTargetCafe));
    AnimSoundTargetCafe* result =
        new(memory) AnimSoundTargetCafe(arg);

    return result;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::Destroy()
{
    this->~AnimSoundTargetCafe();

    // 終了処理
    --s_Counter;
}

//----------------------------------------------------------
u32 AnimSoundTargetCafe::OnLoadModel(const char* filePath, bool isReload)
{
    NW_ASSERT_NOT_NULL(filePath);

    // まずfilepathをmodelのキャッシュリストから探す
    ModelCacheInfo* cache = m_ModelCache.Find(filePath);

    // キャッシュが見つからないときか、リロードフラグが立っているときはロードする
    if (cache == NULL || isReload)
    {
        void* buffer = LoadBinaryFile(filePath);
        if (buffer != NULL)
        {
            if (cache == NULL)
            {
                ModelCacheInfo newCache;
                //std::memset(&newCache, 0, sizeof(ModelCacheInfo));
                std::memset(newCache.path, 0, sizeof(char) * PATH_MAX);
                std::strncpy(newCache.path, filePath, std::strlen(filePath));
                cache = m_ModelCache.Add(newCache);
            }

            nw::gfnd::Graphics::GetInstance()->LockDrawContext();
            {
                cache->model.Finalize(m_Allocator);
                cache->model.Initialize(buffer, m_Allocator);
                cache->model.SetShader(m_Shader, m_Allocator);
            }
            nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
        }
    }
#if defined(NW_DEBUG)
    else
    {
        NW_LOG("use cache\n");
    }
#endif

    if (cache == NULL)
    {
        NW_ASSERT_NOT_NULL(cache);
        return 0;
    }

    gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // 前回のモデルが存在するとき
        if (m_LatestModel != NULL)
        {
            g3d::ModelObj* target = m_LatestModel->GetModelObj();
            NW_ASSERT_NOT_NULL(target);

            // TODO: アニメージョンのアンバインドを実装
            //m_LatestModel->DeactivateAnimSet();     // これをやると、Visibleがtrueになるらしい要注意

            // latestModelのvisibleをfalseに
            SetModelVisible(target, false);
        }

        m_LatestModel = &cache->model;

        nw::g3d::ModelObj* target = m_LatestModel->GetModelObj();
        NW_ASSERT_NOT_NULL(target);

        //あたらしいlatestModelをvisible:true;にする
        SetModelVisible(target, true);
    }
    gfnd::Graphics::GetInstance()->UnlockDrawContext();

    return 0;
}

//----------------------------------------------------------
u32 AnimSoundTargetCafe::OnLoadAnimation(const char* filePath, bool isReload)
{
    NW_ASSERT_NOT_NULL(filePath);

    // filepathをアニメのキャッシュから探す
    AnimCacheInfo* cache = m_AnimCache.Find(filePath);

    // キャッシュが見つからないときか、リロードフラグが立っているときはロードする
    if (cache == NULL || isReload)
    {
        void* buffer = LoadBinaryFile(filePath);
        if (buffer != NULL)
        {
            if (cache == NULL)
            {
                AnimCacheInfo newCache;
                //std::memset(&newCache, 0, sizeof(AnimCacheInfo));
                std::memset(newCache.path, 0, sizeof(char) * PATH_MAX);
                std::strncpy(newCache.path, filePath, std::strlen(filePath));
                cache = m_AnimCache.Add(newCache);
            }

            cache->animSet.Finalize(m_Allocator);
            cache->animSet.Initialize(buffer, *m_LatestModel->GetModelObj(), m_Allocator);
        }
    }
#if defined(NW_DEBUG)
    else
    {
        NW_LOG("use cache\n");
    }
#endif

    if (cache == NULL)
    {
        NW_ASSERT_NOT_NULL(cache);
        return 0;
    }

    m_LatestModel->SetAnimationSet(cache->animSet);
    m_FrameCtrl.SetFrameRange(0.0F, cache->animSet.GetAnimationFrameCount());

    return 0;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnUnload()
{
    // - キャッシュ情報のリングバッファをリセットする
    m_AnimCache.Finalize();
    m_ModelCache.Finalize();

    // 最後に選択したモデルをクリアする
    m_LatestModel = NULL;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnUpdateInputs(const dev::Pad* pad, const nw::dev::Mouse* mouse)
{
    this->UpdateInputsImpl(pad, mouse);
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnCalc()
{
    this->CalcImpl();
}

//----------------------------------------------------------
void AnimSoundTargetCafe::UpdateInputsImpl(const dev::Pad* pad, const nw::dev::Mouse* mouse)
{
    if (pad == NULL && mouse == NULL)
    {
        return;
    }

    InputStatus inputStatus;
    inputStatus.pad = const_cast<dev::Pad*>(pad);
    inputStatus.mouse = const_cast<dev::Mouse*>(mouse);
    inputStatus.isMouseAvailable = true;
    inputStatus.isAltModified = true;
    inputStatus.isCtrlModified = false;
    inputStatus.isShiftModified = false;

    if (m_IsCameraEnabled)
    {
        m_Camera.UpdateCamera(inputStatus);
    }
}

//----------------------------------------------------------
void AnimSoundTargetCafe::CalcImpl()
{
    m_GridDrawer.SetMatrix(
        m_Projection.GetMatrix(),
        m_Camera.GetMatrix()
    );

    nw::math::MTX34 translateMtx;
    translateMtx.SetTranslate( *m_Camera.GetLookAtPos() );
    m_AxisDrawer->SetMatrix(
        m_Projection.GetMatrix(),
        m_Camera.GetMatrix(),
        translateMtx
    );

    if (m_LatestModel == NULL)
    {
        return;
    }

    // 自前のフレーム管理を使用する
    if (m_IsAnimEnable)
    {
        f32 startFrame = m_RangeStart;
        f32 endFrame = m_RangeEnd;
        f32 rate = m_FrameCtrl.GetStep();
        f32 frame = m_FrameCtrl.GetFrame() + rate;
        bool loop = m_FrameCtrl.GetPlayPolicy() == g3d::PlayPolicy_Loop ? true : false;

        if (rate > 0.0F)
        {
            if (frame > endFrame)
            {
                if (loop != false)
                {
                    if (startFrame != endFrame)
                    {
                        NW_ASSERT(startFrame < endFrame);

                        while (frame > endFrame)
                        {
                            f32 diff = frame - endFrame;
                            frame = startFrame + diff;
                        }
                    }
                    else
                    {
                        frame = endFrame;
                    }
                }
                else
                {
                    frame = endFrame;
                }
            }
        }
        else if (rate < 0.0F)
        {
            if (frame < startFrame)
            {
                if (loop != false)
                {
                    if (startFrame != endFrame)
                    {
                        NW_ASSERT(startFrame < endFrame);

                        while (frame < startFrame)
                        {
                            f32 diff = startFrame - frame;
                            frame = endFrame - diff;
                        }
                    }
                    else
                    {
                        frame = startFrame;
                    }
                }
                else
                {
                    frame = startFrame;
                }
            }
        }
        else
        {
            m_IsAnimEnable = false;
        }

        m_FrameCtrl.SetFrame(frame);

        m_LatestModel->SetAnimationFrame(m_FrameCtrl.GetFrame());
    }
    else
    {
        m_LatestModel->SetAnimationFrame(m_FrameCtrl.GetFrame());
    }

    m_LatestModel->Calc();
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnDraw()
{
    this->DrawImpl();
}

//----------------------------------------------------------
void AnimSoundTargetCafe::DrawImpl()
{
    m_GridDrawer.Draw();
    m_AxisDrawer->Draw();

    if (m_LatestModel == NULL)
    {
        return;
    }

    // モデルの描画処理
    m_LatestModel->CalcBlock(
        reinterpret_cast<const nw::g3d::math::Mtx34*>(&m_Camera.GetMatrix()),
        reinterpret_cast<const nw::g3d::math::Mtx44*>(&m_Projection.GetMatrix())
    );

    nw::g3d::CPUCache::Sync();
    nw::g3d::GPUCache::InvalidateAll();

    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
    m_LatestModel->Draw();
    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_REGISTER);

    nw::gfnd::Graphics::GetInstance()->SetDepthFunc(nw::gfnd::Graphics::DEPTH_FUNC_ALWAYS);

}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnPlayAnim()
{
    m_IsAnimEnable = true;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnStopAnim()
{
    m_IsAnimEnable = false;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnSetLoop(bool flg)
{
    if (flg)
    {
        m_FrameCtrl.SetPlayPolicy(g3d::PlayPolicy_Loop);
    }
    else
    {
        m_FrameCtrl.SetPlayPolicy(g3d::PlayPolicy_Onetime);
    }
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnSetRange(f32 start, f32 end)
{
    m_RangeStart = start;
    m_RangeEnd = end;
}

//----------------------------------------------------------
f32 AnimSoundTargetCafe::OnGetRangeStart()
{
    return m_RangeStart;
}

//----------------------------------------------------------
f32 AnimSoundTargetCafe::OnGetRangeEnd()
{
    return m_RangeEnd;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnSetRate(f32 rate)
{
    // rate => stepFrameに変換が必要かも
    m_FrameCtrl.SetStep(rate);
}

//----------------------------------------------------------
f32 AnimSoundTargetCafe::OnGetRate()
{
    // rate => stepFrameに変換が必要かも
    return m_FrameCtrl.GetStep();
}

//----------------------------------------------------------
void AnimSoundTargetCafe::OnSetFrame(f32 frame)
{
    m_FrameCtrl.SetFrame(frame);
}

//----------------------------------------------------------
f32 AnimSoundTargetCafe::OnGetFrame()
{
    return m_FrameCtrl.GetFrame();
}

//----------------------------------------------------------
bool AnimSoundTargetCafe::IsPlayingAnim()
{
    return m_IsAnimEnable;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::SetCameraEnabled(bool isCameraEnabled)
{
    m_IsCameraEnabled = isCameraEnabled;
}

//----------------------------------------------------------
AnimSoundTargetCafe::AnimSoundTargetCafe(const CreateArg& arg)
  : m_Allocator(arg.allocator),
    m_IsCameraEnabled(false),
    m_RangeStart( 0.0F),
    m_RangeEnd( 0.0F),
    m_LatestModel(NULL),
    m_GridDrawer(),
    m_AxisDrawer(NULL),
    m_FrameCtrl(),
    m_IsAnimEnable(false)
{
    NW_NULL_ASSERT(arg.allocator);

    // カメラの初期設定
    m_Camera.SetPos( 5.0f, 3.0f, 15.0f );
    m_Camera.SetLookAtPos( 0.0f, 3.0f, 0.0f );
    m_Camera.Preset();

    InitializeShader();

    // グリッド描画のためのバッファ
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        m_GridDrawer.CreateCache(m_Allocator);
        m_AxisDrawer = AxisDrawer::Create(m_Allocator);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    // キャッシュ用
    m_ModelCache.Initialize(m_Allocator);
    m_AnimCache.Initialize(m_Allocator);
}

//----------------------------------------------------------
AnimSoundTargetCafe::~AnimSoundTargetCafe()
{
    // TODO: ここでやると落ちるのを調査
    m_AnimCache.Finalize();
    m_ModelCache.Finalize();

    m_AxisDrawer->Destroy();
    m_GridDrawer.DestroyCache();

    FinalizeShader();

}

//----------------------------------------------------------
void AnimSoundTargetCafe::InitializeShader()
{
    NW_ASSERT_NOT_NULL(m_Allocator);

    nw::snd::InitializeG3dShaderResource(m_Allocator);

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::snd::SetupSimpleShader(m_Shader, m_Allocator);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

void AnimSoundTargetCafe::FinalizeShader()
{
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        nw::snd::DestroyShader( m_Allocator, m_Shader );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    nw::snd::FinalizeG3dShaderResource(m_Allocator);
}

//----------------------------------------------------------
void* AnimSoundTargetCafe::LoadBinaryFile(const char* filePath)
{
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    NW_ASSERT_NOT_NULL(fileSystem);

    void* buffer = NULL;
    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path        = filePath;
        loadArg.allocator   = m_Allocator;
        loadArg.alignment   = 8 * 1024;
        buffer = fileSystem->Load( loadArg );
        NW_ASSERT( loadArg.readSize > 0 );
    }

    return buffer;
}

//----------------------------------------------------------
AnimSoundTargetCafe::ModelCache::ModelCache()
{
}

//----------------------------------------------------------
AnimSoundTargetCafe::ModelCache::~ModelCache()
{
}

//----------------------------------------------------------
void AnimSoundTargetCafe::ModelCache::OnCacheDestory(ModelCacheInfo& oldestCache, ModelCacheInfo* newCache)
{
    gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        oldestCache.model.Finalize(GetAllocator());
    }
    gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//----------------------------------------------------------
bool AnimSoundTargetCafe::ModelCache::IsCacheHit(ModelCacheInfo& cache, const char* key)
{
    return std::strcmp(cache.path, key) == 0;
}


//----------------------------------------------------------
AnimSoundTargetCafe::AnimCache::AnimCache()
: m_Current(NULL)
{
}

//----------------------------------------------------------
AnimSoundTargetCafe::AnimCache::~AnimCache()
{
    m_Current = NULL;
}

//----------------------------------------------------------
void AnimSoundTargetCafe::AnimCache::OnCacheDestory(AnimCacheInfo& oldestCache, AnimCacheInfo* newCache)
{
    oldestCache.animSet.Finalize(GetAllocator());
}

//----------------------------------------------------------
bool AnimSoundTargetCafe::AnimCache::IsCacheHit(AnimCacheInfo& cache, const char* key)
{
    return std::strcmp(cache.path, key) == 0;
}

//----------------------------------------------------------


} // namespace snd
} // namespace nw

// eof
