﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "precompiled.h"

#include "PanelManager.h"
#include <nw/dw/control/dw_UIElementUtility.h>
#if defined(NW_SND_EDIT_USE_MCS)
#include <nw/mcs.h>
#endif
#include <resources/GlobalResources.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {

const f32 PanelManager::INVALID_POINTER_POSITION = -1.f;

PanelManager::PanelManager() :
m_ActivePanelFrameIndex(INVALID_INDEX),
m_IsPanelsVisible(true)
{
    m_RootElement.SetContents(m_RootContents);

    m_RootContents.AddItem(&m_SoundArchivePlayerPanelFrame);
    m_RootContents.AddItem(&m_SoundParameterPanelFrame);
    m_RootContents.AddItem(&m_SoundSystemPanelFrame);
    m_RootContents.AddItem(&m_PerformancePanelFrame);
#if defined(NW_ENABLE_MIDI)
    m_RootContents.AddItem(&m_PreviewBankPanelFrame);
#endif
    m_RootContents.AddItem(&m_EffectPanelFrame);
    m_RootContents.AddItem(&m_WavePreviewPanelFrame);
    m_RootContents.AddItem(&m_MasterOutputPanelFrame);
    m_RootContents.AddItem(&m_VoicePanelFrame);
}

void PanelManager::Initialize(IResourceProvider& resourceProvider, nw::ut::IAllocator* allocator)
{
    PreviewSoundArchive* pPreviewSoundArchive = reinterpret_cast<PreviewSoundArchive*>(resourceProvider.GetResource(GlobalResources::PreviewSoundArchivePath));
#if defined(NW_ENABLE_SNDEDIT)
    m_pSoundEditSession = reinterpret_cast<edit::SoundEditSession*>(resourceProvider.GetResource(GlobalResources::SoundEditSessionPath));
#endif

    NW_ASSERT_NOT_NULL( allocator );
    NW_ASSERT_NOT_NULL(pPreviewSoundArchive);

    // SoundArchivePlayerPanel
    SoundArchivePlayerPanel& soundArchivePlayerPanelContent =m_SoundArchivePlayerPanelFrame.GetContent();
    soundArchivePlayerPanelContent.Initialize(*pPreviewSoundArchive);
    m_SoundArchivePlayerPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_SoundArchivePlayerPanelFrame.SetLabel("SoundArchivePlayer");
    m_SoundArchivePlayerPanelFrame.SetTopLeft(20.f, 10.f);
    m_SoundArchivePlayerPanelFrame.SetWidth(360.f);
    m_SoundArchivePlayerPanelFrame.SetHeight(95.f);

    // SoundParameterPanel
    m_SoundParameterPanelFrame.GetContent().Initialize(&soundArchivePlayerPanelContent);
    m_SoundParameterPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_SoundParameterPanelFrame.SetLabel("SoundParameters");
    m_SoundParameterPanelFrame.SetTopLeft(20.f, 110.f);
    m_SoundParameterPanelFrame.SetWidth(240.f);
    m_SoundParameterPanelFrame.SetHeight(290.f);

    // SoundSystemPanel
    m_SoundSystemPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_SoundSystemPanelFrame.SetLabel("SoundSystem");
    m_SoundSystemPanelFrame.SetTopLeft(20.f, 405.f);
    m_SoundSystemPanelFrame.SetWidth(240.f);
    m_SoundSystemPanelFrame.SetHeight(110.f);

    // PerformancePanel
    m_PerformancePanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_PerformancePanelFrame.SetLabel("Performance (Video)");
    m_PerformancePanelFrame.SetTopLeft(20.f, 520.f);
    m_PerformancePanelFrame.SetWidth(360.f);
    m_PerformancePanelFrame.SetHeight(190.f);
    m_PerformancePanelFrame.GetContent().Initialize(resourceProvider);

#ifdef NW_SOUNDPLAYER_PRIVATE
    m_PerformancePanelFrame.GetContent().SetLogLamp(&m_PerformancePanelFrame.GetLamp());
#endif

#if defined(NW_ENABLE_MIDI)
    // PreviewBankPanel
    m_PreviewBankPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_PreviewBankPanelFrame.SetLabel("PreviewBank");
    m_PreviewBankPanelFrame.SetTopLeft(400.f, 10.f);
    m_PreviewBankPanelFrame.SetWidth(320.f);
    m_PreviewBankPanelFrame.SetHeight(220.f);
    m_PreviewBankPanelFrame.GetContent().Initialize(*pPreviewSoundArchive);
#endif

    // EffectPanel
    m_EffectPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_EffectPanelFrame.SetLabel("Effect");
    m_EffectPanelFrame.SetTopLeft(400.f, 370.f);
    m_EffectPanelFrame.SetWidth(320.f);
    m_EffectPanelFrame.SetHeight(340.f);
    m_EffectPanelFrame.GetContent().Initialize(resourceProvider);

    // WavePreviewPanel
    m_WavePreviewPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_WavePreviewPanelFrame.SetLabel("WavePreview");
    m_WavePreviewPanelFrame.SetTopLeft(740.f, 10.f);
    m_WavePreviewPanelFrame.SetWidth(488.f);
    m_WavePreviewPanelFrame.SetHeight(210.f);
    m_WavePreviewPanelFrame.GetContent().Initialize(resourceProvider);

    // MasterOutputPanel
    m_MasterOutputPanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_MasterOutputPanelFrame.SetLabel("MasterOutput (x1)");
    m_MasterOutputPanelFrame.SetTopLeft(740.f, 230.f);
    m_MasterOutputPanelFrame.SetWidth(488.f);
    m_MasterOutputPanelFrame.SetHeight(360.f);

    m_MasterOutputPanelFrame.GetLamp().SetLampedBackgroundColor(nw::internal::dw::NwTypeUtility::SRGBToLinear(nw::ut::Color4f(0.4f, 0.2f, 0.2f, 1.f)));
    m_MasterOutputPanelFrame.GetContent().Initialize(resourceProvider);
    m_MasterOutputPanelFrame.GetContent().SetRecordingLamp(&m_MasterOutputPanelFrame.GetLamp());

    // VoicePanel
    m_VoicePanelFrame.SetMeasurement(nw::internal::dw::MEASUREMENT_MANUAL);
    m_VoicePanelFrame.SetLabel("Voices");
    m_VoicePanelFrame.SetTopLeft(740.f, 610.f);
    m_VoicePanelFrame.SetWidth(488.f);
    m_VoicePanelFrame.SetHeight(100.f);

    // Lamps
#if defined(NW_SND_EDIT_USE_MCS)
    m_McsConnectionLamp.SetTopLeft(1200.f, 5.f);
    m_McsConnectionLamp.SetLabelText("MCS");
#endif

    m_RenderingModeLamp.SetTopLeft(1160.f, 5.f);

#if defined(NW_ENABLE_SNDEDIT)
    m_EditEnabledLamp.SetTopLeft(1112.f, 5.f);
    m_EditEnabledLamp.SetLabelText("EDIT");
#endif

    m_SoundArchivePlayerPanelFrame.SetFocus();
    m_ActivePanelFrameIndex = 0;

    m_MousePointer.Initialize(allocator);
}

void PanelManager::Finalize()
{
    m_MousePointer.Finalize();

#if defined(NW_ENABLE_MIDI)
    m_PreviewBankPanelFrame.GetContent().Finalize();
#endif
}

void PanelManager::UpdateInputs(const nw::dev::Pad* pPad, const nw::dev::Mouse* pMouse)
{
    if(pMouse != NULL)
    {
        m_MousePointer.SetPosition(
            nw::math::Vector2(
                pMouse->GetPointer().x,
                pMouse->GetPointer().y));
    }
    else
    {
        m_MousePointer.SetPosition(
            nw::math::Vector2(INVALID_POINTER_POSITION, INVALID_POINTER_POSITION));
    }

    if(pPad != NULL && PreUpdateInputs(*pPad))
    {
        return;
    }

#ifdef NW_SOUNDPLAYER_PRIVATE
    nw::internal::dw::Inputs inputs(pPad, pMouse);
#else
    nw::internal::dw::Inputs inputs(pPad, NULL);
#endif

    m_RootElement.UpdateInputs(inputs);
}

void PanelManager::Update(nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer)
{
    m_RootElement.Update(context, renderer);

    // レンダリングモードランプの更新
    {
        if ( AXGetDefaultMixerSelect() == AX_PB_MIXER_SELECT_PPC )
        {
            m_RenderingModeLamp.SetLabelText("CPU");
            m_RenderingModeLamp.SetIsLamped(false);  // 点灯状態
        }
        else
        {
            m_RenderingModeLamp.SetLabelText("DSP");
            m_RenderingModeLamp.SetIsLamped(true);  // 点灯状態
        }

        m_RenderingModeLamp.Update(context, renderer);
    }

    // MCSランプの更新
#if defined(NW_SND_EDIT_USE_MCS)
    {
        bool isMcsServerConnected = nw::mcs::Mcs_IsServerConnect();

        if ( isMcsServerConnected )
        {
            m_McsConnectionLamp.SetIsLamped(true);
        }
        else
        {
            m_McsConnectionLamp.SetIsLamped(false);
        }

        m_McsConnectionLamp.Update(context, renderer);
    }
#endif

#if defined(NW_ENABLE_SNDEDIT)
    // リアルタイム編集 有効状態の更新
    {
        if(m_pSoundEditSession != NULL && m_pSoundEditSession->IsConnected())
        {
            m_EditEnabledLamp.SetIsLamped(true);
        }
        else
        {
            m_EditEnabledLamp.SetIsLamped(false);
        }

        m_EditEnabledLamp.Update(context, renderer);
    }
#endif

    // パフォーマンスパネルのラベル更新
    if (m_PerformancePanelFrame.GetContent().IsVideoFrameProfile())
    {
        m_PerformancePanelFrame.SetLabel("Performance (Video)");
    }
    else
    {
        m_PerformancePanelFrame.SetLabel("Performance (Audio)");
    }

    // マスター出力パネルのラベル更新
    {
        static char label[32];
        nw::ut::snprintf(
            label,
            32, 32 - 1,
            "MasterOutput (%4.2fsec)",
            m_MasterOutputPanelFrame.GetContent().GetScaleX());

        m_MasterOutputPanelFrame.SetLabel(label);
    }

    NW_ASSERT(context.GetLevel() == 0);
    NW_ASSERT(context.GetRenderOffset() == nw::math::Vector2::Zero());
}

void PanelManager::Render(nw::internal::dw::UIElementTreeContext& context, nw::internal::dw::UIRenderer& renderer)
{
#ifndef USE_ANIM_SOUND
    m_RootElement.Render(context, renderer);
#else
    if(m_IsPanelsVisible)
    {
        m_RootElement.Render(context, renderer);
    }
#endif

    m_RenderingModeLamp.Render(context, renderer);
#if defined(NW_SND_EDIT_USE_MCS)
    m_McsConnectionLamp.Render(context, renderer);
#endif

#if defined(NW_ENABLE_SNDEDIT)
    m_EditEnabledLamp.Render(context, renderer);
#endif

    DrawMousePointer(renderer);

    NW_ASSERT(context.GetLevel() == 0);
    NW_ASSERT(context.GetRenderOffset() == nw::math::Vector2::Zero());
}

bool PanelManager::PreUpdateInputs(const nw::dev::Pad& pad)
{
    NW_ASSERT(m_RootContents.GetCount() > 0);
    NW_ASSERT(
        m_ActivePanelFrameIndex == INVALID_INDEX ||
        m_ActivePanelFrameIndex < m_RootContents.GetCount());

    // グローバルキーアサインとして処理した場合は、UI要素には処理させないように true を返します。
    // Z キーを押下中はグローバルキーアサインとします。
    if(!pad.IsHold(nw::dev::Pad::MASK_Z))
    {
        // グローバルキーアサイン : 選択サウンドの再生
        if(pad.IsHold(nw::dev::Pad::MASK_R) &&
            pad.IsTrig(nw::dev::Pad::MASK_A))
        {
            PreviewSound* pPreviewSound =
                m_SoundArchivePlayerPanelFrame.GetContent().GetSelectedPreviewSoundFirst();

            if(pPreviewSound != NULL)
            {
                pPreviewSound->Play();
            }

            return true;
        }

        // グローバルキーアサイン : 選択サウンドの再生
        if(pad.IsHold(nw::dev::Pad::MASK_R) &&
            pad.IsTrig(nw::dev::Pad::MASK_B))
        {
            PreviewSound* pPreviewSound =
                m_SoundArchivePlayerPanelFrame.GetContent().GetSelectedPreviewSoundFirst();

            if(pPreviewSound != NULL)
            {
                pPreviewSound->Stop();
            }

            return true;
        }

        return false;
    }

    // グローバルキーアサイン : パネルの表示切替
    if(pad.IsTrig(nw::dev::Pad::MASK_R))
    {
        m_IsPanelsVisible = !m_IsPanelsVisible;
        return true;
    }

    // グローバルキーアサイン : フォーカス移動
    s32 newActivePanelFrameIndex =
        m_ActivePanelFrameIndex == INVALID_INDEX ? 0 : m_ActivePanelFrameIndex;

    if(pad.IsTrig(nw::dev::Pad::MASK_DOWN))
    {
        while(true)
        {
            if(newActivePanelFrameIndex >= m_RootContents.GetCount() - 1)
            {
                newActivePanelFrameIndex = 0;
            }
            else
            {
                newActivePanelFrameIndex++;
            }

            if(m_ActivePanelFrameIndex == newActivePanelFrameIndex)
            {
                break;
            }

            if(m_RootContents[newActivePanelFrameIndex]->SetFocus())
            {
                break;
            }
        }
    }
    else if(pad.IsTrig(nw::dev::Pad::MASK_UP))
    {
        while(true)
        {
            if(newActivePanelFrameIndex == 0)
            {
                newActivePanelFrameIndex = m_RootContents.GetCount() - 1;
            }
            else
            {
                newActivePanelFrameIndex--;
            }

            if(m_ActivePanelFrameIndex == newActivePanelFrameIndex)
            {
                break;
            }

            if(m_RootContents[newActivePanelFrameIndex]->SetFocus())
            {
                break;
            }
        }
    }

    m_ActivePanelFrameIndex = newActivePanelFrameIndex;

    if(m_ActivePanelFrameIndex != newActivePanelFrameIndex)
    {
        return true;
    }

    // グローバルキーアサイン : レンダリングモード (DSP / CPU) の切り替え)
    if(pad.IsTrig(nw::dev::Pad::MASK_L))
    {
        if ( AXGetDefaultMixerSelect() == AX_PB_MIXER_SELECT_PPC )
        {
            AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
        }
        else
        {
            AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_PPC );
        }
    }

#if defined(PRINT_FOCUS_LOG) && defined(NW_DEBUG)
    if(pad.IsTrig(nw::dev::Pad::MASK_START))
    {
        nw::internal::dw::UIElementUtility::PrintUIElementTree(m_WavePreviewPanelFrame);
        return true;
    }
#endif

    return true;
}

bool PanelManager::IsPanelsVisible() const
{
    return m_IsPanelsVisible;
}

PreviewEffectManager* PanelManager::CurrentPreviewEffectManager()
{
    return m_EffectPanelFrame.GetContent().CurrentPreviewEffectManager();
}

void PanelManager::SetEffectDevice(u32 deviceType)
{
    m_EffectPanelFrame.GetContent().SetDevice(deviceType);
}

void PanelManager::DrawMousePointer(nw::internal::dw::UIRenderer& renderer)
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    // Windows 版では描画しません。
    return;
#endif

    if(m_MousePointer.GetPosition().x < 0 || m_MousePointer.GetPosition().y < 0)
    {
        return;
    }

#ifdef NW_SOUNDPLAYER_PRIVATE
    m_MousePointer.Draw(renderer);
#endif
}

} // nw::snd
} // nw
