﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"
#include "../Application.h"

#include <cafe/pad.h>
#include <nw/types.h>
#include <nw/ut.h>
#include <nw/gfnd.h>
#include <nw/snd/util/sndutil_FileReader.h>
#include <anim/GfxCoordinator.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
bool
Application::Initialize(
    FSClient* fsClient,
    FSClient* fsClientSpy,
    nw::ut::IAllocator* pAllocator,
    const nw::ut::Rect& viewport)
{
    NW_ASSERT_NOT_NULL(fsClient);
    NW_ASSERT_NOT_NULL(pAllocator);
    NW_ASSERTMSG(m_pAllocator == NULL, "Application is already initialized.");

    m_pAllocator = pAllocator;
    m_Viewport = viewport;

    m_ResourceProvider.SetAllocator(pAllocator);
    m_ResourceProvider.SetPreviewSoundArchive(&m_PreviewSoundArchive);
    m_ResourceProvider.SetPreviewEffectManager(&m_PreviewEffectManager);
    m_ResourceProvider.SetPreviewEffectManagerDrc(&m_PreviewEffectManagerDrc);
#if defined(NW_ENABLE_SNDEDIT)
    m_ResourceProvider.SetSoundEditSession(&m_SoundEditSession);
#endif

    // FS 初期化
    {
        m_FsClient = fsClient;

        m_FsCmdBlock = reinterpret_cast<FSCmdBlock*>(m_pAllocator->Alloc(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(m_FsCmdBlock);
        FSInitCmdBlock(m_FsCmdBlock);
    }

    // HostFileIO デバイスのマウント（sndedit で利用）
    {
        FSMountSource mountSrc;
        FSGetMountSource(m_FsClient, m_FsCmdBlock, FS_SOURCETYPE_HFIO, &mountSrc, FS_RET_NO_ERROR);
        FSMount(m_FsClient, m_FsCmdBlock, &mountSrc, m_FsHfioMountPath, sizeof(m_FsHfioMountPath), FS_RET_NO_ERROR);
    }

    m_ResourceProvider.SetFsClient(m_FsClient);
    m_ResourceProvider.SetFsCmdBlock(m_FsCmdBlock);
    m_ResourceProvider.SetHfioVolume(m_FsHfioMountPath);

#if defined(NW_ENABLE_SNDCTRL)
    m_PreviewSoundArchive.Initialize(
        m_FsClient,
        SOUND_ARCHIVE_FILEPATH,
        SOUND_HEAP_SIZE,
        &m_SoundObjectController);
#else
    m_PreviewSoundArchive.Initialize(fsClient, SOUND_ARCHIVE_FILEPATH, SOUND_HEAP_SIZE);
#endif
    m_PreviewSoundArchive.Open(SOUND_ARCHIVE_FILEPATH);

    m_PreviewEffectManager.Initialize(m_EffectBuffer, EFFECT_BUFFER_SIZE, OUTPUT_DEVICE_MAIN);
    m_PreviewEffectManagerDrc.Initialize(m_EffectBufferDrc, EFFECT_BUFFER_SIZE, OUTPUT_DEVICE_DRC);

    if(!InitializeSndEdit(pAllocator))
    {
        return false;
    }

    if(!m_HioManager.Initialize())
    {
        return false;
    }

    if(!InitializePassiveChannel(*pAllocator))
    {
        return false;
    }

    if(!InitializeActiveChannel(*pAllocator))
    {
        return false;
    }

    if(!InitializePingChannel(*pAllocator))
    {
        return false;
    }

    InitializeMessageHandlers();

    if(!InitializeFont())
    {
        return false;
    }

    if(!InitializeTextWriter())
    {
        return false;
    }

    if(!InitializePrimitiveRenderer())
    {
        return false;
    }

    // G3D連携の初期化
#ifdef USE_ANIM_SOUND
    {
        GfxCoordinator::InitializeArg arg;
        arg.allocator = pAllocator;
        arg.fontBinary = reinterpret_cast<u8*>(m_pFontBinary);
        arg.fontBinarySize = m_FontBinarySize;
        arg.screenWidth = static_cast<s32>(viewport.GetWidth());
        arg.screenHeight = static_cast<s32>(viewport.GetHeight());
        arg.shaderBinary = reinterpret_cast<u8*>(m_pFontShaderBinary);
        arg.shaderBinarySize = m_FontShaderBinarySize;

        GfxCoordinator::Initialize(arg);

        // note:
        // インスタンスのコンストラクタでも初期化処理が行われるため、インスタンスの取得は必要
        GfxCoordinator& coord = GfxCoordinator::GetInstance();
    }

    if (!m_AnimSoundController.Initialize(m_pAllocator, m_PreviewSoundArchive.GetSoundArchivePlayer()))
    {
        return false;
    }
#endif

    if(!InitializeUI())
    {
        return false;
    }

#if defined(NW_SND_CONFIG_ENABLE_DEV)
    {
        nw::snd::spy::SpyController::OpenArg arg;

        char hfioMountPath[FS_MAX_MOUNTPATH_SIZE];
        m_FsClientSpy = fsClientSpy;
        if (m_FsClientSpy)
        {
            // HostFileIO デバイスのマウント

            FSStatus result = FS_STATUS_OK;
            FSCmdBlock cmdBlock;
            FSInitCmdBlock(&cmdBlock);

            FSMountSource mountSrc;
            result = FSGetMountSource(
                m_FsClientSpy,
                &cmdBlock,
                FS_SOURCETYPE_HFIO,
                &mountSrc,
                FS_RET_ALL_ERROR);

            if (result == FS_STATUS_OK)
            {
                result = FSMount(
                    m_FsClientSpy,
                    &cmdBlock,
                    &mountSrc,
                    hfioMountPath,
                    sizeof(hfioMountPath),
                    FS_RET_ALL_ERROR);
            }

            if (result == FS_STATUS_OK)
            {
                arg.dataFileChannelParam.fsClient = fsClientSpy;
                arg.dataFileChannelParam.hfioMountPath = hfioMountPath;
            }
        }

        if(!InitializeSpy(arg))
        {
            return false;
        }
    }
#endif

    UpdateViewport();

    return true;
}

//---------------------------------------------------------------------------
void
Application::Finalize()
{
#if defined(NW_SND_CONFIG_ENABLE_DEV)
    FinalizeSpy();
#endif

#ifdef USE_ANIM_SOUND
    GfxCoordinator::Finalize();
    m_AnimSoundController.Finalize();
#endif

    m_PreviewEffectManagerDrc.Finalize();
    m_PreviewEffectManager.Finalize();
    m_PreviewSoundArchive.Finalize();

    FinalizeMessageHandlers();
    FinalizePassiveChannel();
    FinalizeActiveChannel();
    FinalizePingChannel();

    FinalizeSndEdit();

    m_HioManager.Finalize();

    FinalizeUI();
    FinalizePrimitiveRenderer();
    FinalizeTextWriter();
    FinalizeFont();

    if(m_FsClient != NULL && m_FsCmdBlock != NULL)
    {
        NW_ASSERT_NOT_NULL(m_pAllocator);

        FSUnmount(m_FsClient, m_FsCmdBlock, m_FsHfioMountPath, FS_RET_NO_ERROR);

        m_pAllocator->Free(m_FsCmdBlock);

        m_FsClient = NULL;
        m_FsCmdBlock = NULL;
    }
}

//---------------------------------------------------------------------------
bool
Application::InitializeTextWriter()
{
    m_pFontShaderBinary = NULL;

    m_pFontShaderBinary = LoadFile(
        FONT_SHADER_FILENAME,
        FONT_SHADER_BINARY_ALIGNMENT,
        &m_FontShaderBinarySize);

    if(m_pFontShaderBinary == NULL)
    {
        return false;
    }

    if(!m_TextRenderer.Initialize(
        *m_pAllocator,
        m_pFontBinary,
        m_FontBinarySize,
        m_pFontShaderBinary,
        m_FontShaderBinarySize))
    {
        return false;
    }

    m_TextRenderer.SetGlobalScale(0.70f);

    {
        nw::font::TextWriter& textWriter = m_TextRenderer.m_TextWriter.GetTextWriter();
        textWriter.EnableFixedWidth(true);
        textWriter.SetFixedWidth(textWriter.GetFont()->GetWidth() * 0.55f);
    }

    return true;
}

//---------------------------------------------------------------------------
bool
Application::InitializePrimitiveRenderer()
{
    nw::dev::PrimitiveRenderer* pPrimitiveRenderer = nw::dev::PrimitiveRenderer::GetInstance();

    u32 shaderBinarySize = 0;

    m_pPrimitiveRendererShaderBinary = LoadFile(
        PRIMITIVE_RENDERER_SHADER_FILENAME,
        internal::fnd::FileTraits::IO_BUFFER_ALIGNMENT,
        &shaderBinarySize);

    if(m_pPrimitiveRendererShaderBinary == NULL)
    {
        return false;
    }

    // PrimitiveRenderer を初期化します。
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        pPrimitiveRenderer->InitializeFromBinary(
            m_pAllocator,
            m_pPrimitiveRendererShaderBinary,
            shaderBinarySize);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    return true;
}

//---------------------------------------------------------------------------
bool
Application::InitializeSoundArchiveEditor(nw::ut::IAllocator* pAllocator)
{
#if defined(NW_ENABLE_SNDEDIT)
    // サウンドアーカイブエディタの初期化
    nw::snd::edit::SoundArchiveEditor::Configs configs;

    NW_ASSERT_NOT_NULL(m_SoundArchiveEditorBuffer);

    if(m_SoundArchiveEditor.Initialize(
        m_SoundArchiveEditorBuffer,
        SNDEDIT_HEAP_SIZE,
        &m_PreviewSoundArchive.GetSoundArchive(),
        &m_PreviewSoundArchive.GetSoundArchivePlayer(),
        configs).IsFailed())
    {
        return false;
    }
#endif
    return true;
}

//---------------------------------------------------------------------------
bool
Application::InitializeSndEdit(nw::ut::IAllocator* pAllocator)
{
#if defined(NW_ENABLE_SNDEDIT)
    // sndedit の初期化
    {
        m_SoundArchiveEditorBuffer = pAllocator->Alloc( SNDEDIT_HEAP_SIZE );
        if (!InitializeSoundArchiveEditor(pAllocator))
        {
            return false;
        }

        m_PreviewSoundArchive.RegisterSoundArchiveEditor(&m_SoundArchiveEditor);

        // サウンド編集セッションの初期化
        {
            nw::snd::edit::SoundEditSession::Configs configs;
            u32 memorySize = m_SoundEditSession.GetRequiredMemorySize(configs);

            m_SoundEditSessionBuffer = pAllocator->Alloc( memorySize );

// HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
            m_SoundEditSession.Interim_SetIsSoundPlayer(true);
#endif

            if(m_SoundEditSession.Initialize(
                m_SoundEditSessionBuffer,
                memorySize,
                m_FsClient,
                m_FsHfioMountPath,
                configs).IsFailed())
            {
                return false;
            }

            if(m_SoundEditSession.RegisterSoundArchiveEditor(&m_SoundArchiveEditor).IsFailed())
            {
                return false;
            }
        }

        m_SoundEditUpdateCachesThreadStack = static_cast<u8*>(pAllocator->Alloc(
            SOUND_EDIT_UPDATE_CACHES_THREAD_STACK_SIZE,
            internal::fnd::Thread::STACK_ALIGNMENT));

        if(m_SoundEditUpdateCachesHelper.Start(
            &m_SoundArchiveEditor,
            m_SoundEditUpdateCachesThreadStack,
            SOUND_EDIT_UPDATE_CACHES_THREAD_STACK_SIZE).IsFailed())
        {
            return false;
        }
    }
#endif

#if defined(NW_ENABLE_SNDCTRL)
    // sndctrl の初期化
    {
        // SoundObjectController の初期化は都合上、PreviewSoundArchive で行われます。
        // 後で整理する必要あり。

        // サウンド制御セッションの初期化
        {
            nw::snd::ctrl::SoundControlSession::Configs configs;
            u32 memorySize = m_SoundControlSession.GetRequiredMemorySize(configs);

            m_SoundControlSessionBuffer = pAllocator->Alloc( memorySize );

            if(m_SoundControlSession.Initialize(
                m_SoundControlSessionBuffer,
                memorySize,
                configs).IsFailed())
            {
                return false;
            }

            if(m_SoundControlSession.RegisterSoundObjectController(&m_SoundObjectController).IsFailed())
            {
                return false;
            }
        }

        m_SoundEditSession.Open();
        m_SoundControlSession.Open();
    }
#endif

    return true;
}

//---------------------------------------------------------------------------
void
Application::FinalizeSndEdit()
{
#if defined(NW_ENABLE_SNDEDIT)
    m_SoundEditUpdateCachesHelper.Stop();
    m_SoundEditSession.UnregisterSoundArchiveEditor(&m_SoundArchiveEditor);
    m_SoundEditSession.Close();
    m_SoundEditSession.Finalize();

    m_PreviewSoundArchive.RegisterSoundArchiveEditor(NULL);

    FinalizeSoundArchiveEditor();

    if(m_pAllocator != NULL)
    {
        ut::SafeFree(m_SoundEditSessionBuffer, m_pAllocator);
        ut::SafeFree(m_SoundArchiveEditorBuffer, m_pAllocator);
        ut::SafeFree(m_SoundEditUpdateCachesThreadStack, m_pAllocator);
    }
#endif

#if defined(NW_ENABLE_SNDCTRL)
    m_SoundControlSession.UnregisterSoundObjectController(&m_SoundObjectController);
    m_SoundControlSession.Close();
    m_SoundControlSession.Finalize();
#endif
}

//---------------------------------------------------------------------------
void
Application::FinalizeSoundArchiveEditor()
{
#if defined(NW_ENABLE_SNDEDIT)
    m_SoundArchiveEditor.Finalize();
#endif
}

//---------------------------------------------------------------------------
void*
Application::LoadFile(
    const char* path,
    u32 alignment /*= internal::fnd::FileTraits::IO_BUFFER_ALIGNMENT*/,
    u32* fileSize /*= NULL*/)
{
    util::FileReader fileReader;

    if(fileReader.Initialize(m_FsClient) != util::FileReader::RESULT_SUCCESS)
    {
        NW_FATAL_ERROR("[SoundPlayer Error] out of memory.");
        return NULL;
    }

    if(fileReader.Open(path) != util::FileReader::RESULT_SUCCESS)
    {
        NW_FATAL_ERROR("[SoundPlayer Error] open file '%s'.", path);
        return NULL;
    }

    u32 size = fileReader.GetSize();

    NW_ASSERT_NOT_NULL(m_pAllocator);
    void* result = m_pAllocator->Alloc(size, alignment);

    if(result == NULL)
    {
        NW_FATAL_ERROR("[SoundPlayer Error] out of memory.");
        fileReader.Close();
        return NULL;
    }

    if(fileReader.Read(result, size, NULL) != util::FileReader::RESULT_SUCCESS)
    {
        fileReader.Close();
        m_pAllocator->Free(result);
        return NULL;
    }

    fileReader.Close();

    if(fileSize != NULL)
    {
        *fileSize = size;
    }

    return result;
}

//----------------------------------------------------------
HioChannel::ChannelType
Application::GetChannelInfo(HioChannelType channel) const
{
#if defined(NW_SND_EDIT_USE_MCS)
    return static_cast<HioChannel::ChannelType>(channel);
#else
    switch(channel)
    {
        case HIO_SOUNDPLAYER_PASSIVE_CHANNEL:
            return "NW_SND_TOOL";
        case HIO_SOUNDPLAYER_ACTIVE_CHANNEL:
            return "NW_SND_VIEWER";
        case HIO_PING_CHANNEL:
            return "NW_SND_PING";
        default:
            NW_ASSERT(false);
            return "INVALID_CH";
    }
#endif
}

} // nw::snd
} // nw
