﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/snd.h>

namespace nw {
namespace snd {

enum PreviewEffectType
{
    EFFECT_NONE,
    EFFECT_REVERB_HI,
    EFFECT_REVERB_STD,
    EFFECT_CHORUS,
    EFFECT_DELAY,
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    EFFECT_MULTICH_REVERB,
    EFFECT_MULTICH_CHORUS,
    EFFECT_MULTICH_DELAY
#endif
};

//---------------------------------------------------------------------------
//! @brief エフェクトのインターフェイスです。
//---------------------------------------------------------------------------
class IPreviewEffect
{
public:
    virtual PreviewEffectType GetEffectType() const = 0;

    virtual bool GetIsDirty() const = 0;
    virtual void ResetIsDirty() = 0;
    virtual FxBase* GetEffect() = 0;
    virtual u32 GetRequiredMemorySize() = 0;
    virtual bool AssignBuffer(void* pBuffer, unsigned long bufferSize) = 0;
    virtual void ReleaseBuffer() = 0;
};

//---------------------------------------------------------------------------
//! @brief エフェクトのテンプレート実装クラスです。
//---------------------------------------------------------------------------
template<PreviewEffectType TEffectType, class TEffect, class TParameter>
class PreviewEffect : public IPreviewEffect
{
public:
    /* ctor */ PreviewEffect() :
        m_IsDirty(false)
    { }

    /* dtor */ virtual ~PreviewEffect() { }

public:
    const TParameter& GetParameters() const
    {
        return m_Parameters;
    }

    void SetParameters(const TParameter& parameters)
    {
        m_Parameters = parameters;
        m_Effect.SetParam(m_Parameters);
    }

    /* override */ bool GetIsDirty() const
    {
        return m_IsDirty;
    }

    /* override */ void ResetIsDirty()
    {
        SetIsDirty(false);
    }

    /* override */ PreviewEffectType GetEffectType() const
    {
        return TEffectType;
    }

    /* override */ FxBase* GetEffect()
    {
        if(m_IsDirty)
        {
            m_Effect.SetParam(m_Parameters);
        }

        return &m_Effect;
    }

    /* override */ u32 GetRequiredMemorySize()
    {
        if(m_IsDirty)
        {
            m_Effect.SetParam(m_Parameters);
        }

        return m_Effect.GetRequiredMemSize();
    }

    /* override */ bool AssignBuffer(void* pBuffer, unsigned long bufferSize)
    {
        NW_NULL_ASSERT(pBuffer);
        return m_Effect.AssignWorkBuffer(pBuffer, bufferSize);
    }

    /* override */ void ReleaseBuffer()
    {
        m_Effect.ReleaseWorkBuffer();
    }

protected:
    TEffect* GetDerivedEffect()
    {
        if(m_IsDirty)
        {
            m_Effect.SetParam(m_Parameters);
        }

        return &m_Effect;
    }

    TParameter& GetParameters()
    {
        return m_Parameters;
    }

    void SetIsDirty(bool value)
    {
        m_IsDirty = value;
    }

private:
    TEffect m_Effect;
    TParameter m_Parameters;
    bool m_IsDirty;
};

//---------------------------------------------------------------------------
//! @brief エフェクトの NULL オブジェクトです。
//---------------------------------------------------------------------------
class PreviewEffectNone : public IPreviewEffect
{
public:
    /* override */ bool GetIsDirty() const { return false; }
    /* override */ void ResetIsDirty() { }
    /* override */ PreviewEffectType GetEffectType() const { return EFFECT_NONE; }
    /* override */ FxBase* GetEffect() { return NULL; }
    /* override */ u32 GetRequiredMemorySize() { return 0; }
    /* override */ bool AssignBuffer(void* pBuffer, unsigned long bufferSize) { return true; }
    /* override */ void ReleaseBuffer() { }
};

//---------------------------------------------------------------------------
//! @brief 高品質リバーブ エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectReverbHi : public PreviewEffect<EFFECT_REVERB_HI, FxReverbHi, FxReverbHi::ReverbHiParam>
{
public:
    snd::FxReverbEarlyReflectionMode GetEarlyReflectionMode();
    void SetEarlyReflectionMode(snd::FxReverbEarlyReflectionMode value);
    f32 GetPreDelayTime();
    void SetPreDelayTime(f32 value);
    f32 GetPreDelayTimeMax();
    void SetPreDelayTimeMax(f32 value);
    snd::FxReverbFusedMode GetFusedMode();
    void SetFusedMode(snd::FxReverbFusedMode value);
    f32 GetFusedTime();
    void SetFusedTime(f32 value);
    f32 GetColoration();
    void SetColoration(f32 value);
    f32 GetDamping();
    void SetDamping(f32 value);
    f32 GetCrosstalk();
    void SetCrosstalk(f32 value);
    f32 GetEarlyGain();
    void SetEarlyGain(f32 value);
    f32 GetFusedGain();
    void SetFusedGain(f32 value);
    f32 GetOutGain();
    void SetOutGain(f32 value);
};

//---------------------------------------------------------------------------
//! @brief 標準品質リバーブ エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectReverbStd : public PreviewEffect<EFFECT_REVERB_STD, FxReverbStd, FxReverbStd::ReverbStdParam>
{
public:
    snd::FxReverbEarlyReflectionMode GetEarlyReflectionMode();
    void SetEarlyReflectionMode(snd::FxReverbEarlyReflectionMode value);
    f32 GetPreDelayTime();
    void SetPreDelayTime(f32 value);
    f32 GetPreDelayTimeMax();
    void SetPreDelayTimeMax(f32 value);
    snd::FxReverbFusedMode GetFusedMode();
    void SetFusedMode(snd::FxReverbFusedMode value);
    f32 GetFusedTime();
    void SetFusedTime(f32 value);
    f32 GetColoration();
    void SetColoration(f32 value);
    f32 GetDamping();
    void SetDamping(f32 value);
    f32 GetEarlyGain();
    void SetEarlyGain(f32 value);
    f32 GetFusedGain();
    void SetFusedGain(f32 value);
    f32 GetOutGain();
    void SetOutGain(f32 value);
};

//---------------------------------------------------------------------------
//! @brief コーラス エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectChorus : public PreviewEffect<EFFECT_CHORUS, FxChorus, FxChorus::ChorusParam>
{
public:
    f32 GetDelayTime();
    void SetDelayTime(f32 value);
    f32 GetDepth();
    void SetDepth(f32 value);
    f32 GetRate();
    void SetRate(f32 value);
    f32 GetFeedback();
    void SetFeedback(f32 value);
    f32 GetOutGain();
    void SetOutGain(f32 value);
};

//---------------------------------------------------------------------------
//! @brief ディレイ エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectDelay : public PreviewEffect<EFFECT_DELAY, FxDelay, FxDelay::DelayParam>
{
public:
    f32 GetDelayTime();
    void SetDelayTime(f32 value);
    f32 GetMaxDelayTime();
    void SetMaxDelayTime(f32 value);
    f32 GetFeedback();
    void SetFeedback(f32 value);
    f32 GetOutGain();
    void SetOutGain(f32 value);
    f32 GetLpf();
    void SetLpf(f32 value);
};

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
//---------------------------------------------------------------------------
//! @brief マルチチャンネルリバーブ エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectMultiChReverb : public PreviewEffect<EFFECT_MULTICH_REVERB, FxMultiChReverb, FxMultiChReverb::MultiChReverbParam>
{
public:
    u32 GetEarlyMode();
    void SetEarlyMode(u32 value);
    u32 GetLateMode();
    void SetLateMode(u32 value);
    f32 GetPreDelayTime();
    void SetPreDelayTime(f32 value);
    f32 GetDecayTime();
    void SetDecayTime(f32 value);
    f32 GetHfDecayRatio();
    void SetHfDecayRatio(f32 value);
    f32 GetColoration();
    void SetColoration(f32 value);
    f32 GetEarlyGain();
    void SetEarlyGain(f32 value);
    f32 GetLateGain();
    void SetLateGain(f32 value);
    f32 GetReverbGain();
    void SetReverbGain(f32 value);

    void SetChannelMode(FxBase::ChannelMode mode);
};

//---------------------------------------------------------------------------
//! @brief マルチチャンネルコーラス エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectMultiChChorus : public PreviewEffect<EFFECT_MULTICH_CHORUS, FxMultiChChorus, FxMultiChChorus::MultiChChorusParam>
{
public:
    f32 GetDelayTime();
    void SetDelayTime(f32 value);
    f32 GetDepth();
    void SetDepth(f32 value);
    f32 GetRate();
    void SetRate(f32 value);
    f32 GetFeedback();
    void SetFeedback(f32 value);
    f32 GetOutGain();
    void SetOutGain(f32 value);

    void SetChannelMode(FxBase::ChannelMode mode);
};

//---------------------------------------------------------------------------
//! @brief マルチチャンネルディレイ エフェクトです。
//---------------------------------------------------------------------------
class PreviewEffectMultiChDelay : public PreviewEffect<EFFECT_MULTICH_DELAY, FxMultiChDelay, FxMultiChDelay::MultiChDelayParam>
{
public:
    f32 GetDelay();
    void SetDelay(f32 value);
    f32 GetFeedback();
    void SetFeedback(f32 value);
    f32 GetOutGain();
    void SetOutGain(f32 value);
    f32 GetChannelSpread();
    void SetChannelSpread(f32 value);
    f32 GetLpf();
    void SetLpf(f32 value);

    void SetChannelMode(FxBase::ChannelMode mode);
};
#endif

} // nw::snd
} // nw
