﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW4R_VIEWER_SOUND_EFFECT_H_
#define NW4R_VIEWER_SOUND_EFFECT_H_

#pragma once

#include <nw/snd.h>
#include <nw/snd/fnd/binary/sndfnd_PrimitiveTypes.h>
//using namespace NW4R::ProtocolSound;
//using namespace nw4r::snd;

namespace nw {
namespace snd {

//-----------------------------------------------------------------------
/* バスの種類 */
enum ToolSoundAuxBusType {
        /* Aux A  */
        AuxBusType_AuxA =	0,
        /* Aux B  */
        AuxBusType_AuxB =	1,
        /* Aux C  */
        AuxBusType_AuxC =	2,
        /* enumを4Byte型にするための定義 */
        ToolSoundAuxBusType_4BYTE_ENUM =	0xFFFFFFFF
};

//-----------------------------------------------------------------------
/* エフェクトの種類 */
enum ToolSoundEffectType {
        /* NO EFFECT   */
        EffectType_NoEffect =	0,
        /* REVERB HI   */
        EffectType_ReverbHi =	1,
        /* REVERB STD  */
        EffectType_ReverbStd =	2,
        /* CHORUS      */
        EffectType_Chorus =	3,
        /* DELAY       */
        EffectType_Delay =	4,
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
        /* MultiChannelReverb       */
        EffectType_MultiChannelReverb =	5,
        /* MultiChannelChorus       */
        EffectType_MultiChannelChorus =	6,
        /* MultiChannelDelay        */
        EffectType_MultiChannelDelay =	7,
#endif
        /* enumを4Byte型にするための定義 */
        ToolSoundEffectType_4BYTE_ENUM =	0xFFFFFFFF
};

//-----------------------------------------------------------------------
/* 後部残響音の種類 */
enum ToolSoundFusedMode {
        /* old axfx     */
        FusedMode_OldAxfx =	0,
        /* metal tank   */
        FusedMode_MetalTank =	1,
        /* small room   */
        FusedMode_SmallRoom =	2,
        /* large room   */
        FusedMode_LargeRoom =	3,
        /* hall         */
        FusedMode_Hall =	4,
        /* cavernous    */
        FusedMode_Cavernous =	5,
        /* enumを4Byte型にするための定義 */
        ToolSoundFusedMode_4BYTE_ENUM =	0xFFFFFFFF
};

//-----------------------------------------------------------------------
/* 各種エフェクトパラメータ共通部 */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B:エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /* 44B: reserved */
    unsigned char	reserved[44];
} ToolSoundEffectCommonParameter;

//-----------------------------------------------------------------------
/* NO EFFECT パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B:エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /* 44B: reserved */
    unsigned char	reserved[44];
} ToolSoundNoEffectParameter;

//-----------------------------------------------------------------------
/* REVERB HI パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B: エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: early mode */
    internal::fnd::BinS32	earlyMode;
    /*  4B: pre delay time max */
    internal::fnd::BinF32	preDelayTimeMax;
    /*  4B: pre delay time */
    internal::fnd::BinF32	preDelayTime;
    /*  4B: fused mode */
    internal::fnd::BinU32   fusedMode;
    /*  4B: fused time */
    internal::fnd::BinF32	fusedTime;
    /*  4B: coloration */
    internal::fnd::BinF32	coloration;
    /*  4B: damping */
    internal::fnd::BinF32	damping;
    /*  4B: crosstalk */
    internal::fnd::BinF32	crosstalk;
    /*  4B: early gain */
    internal::fnd::BinF32	earlyGain;
    /*  4B: fused gain */
    internal::fnd::BinF32	fusedGain;
    /*  4B: out gain */
    internal::fnd::BinF32	outGain;
} ToolSoundReverbHiParameter;

//-----------------------------------------------------------------------
/* REVERB STD パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B:エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: early mode */
    internal::fnd::BinS32	earlyMode;
    /*  4B: pre delay time max */
    internal::fnd::BinF32	preDelayTimeMax;
    /*  4B: pre delay time */
    internal::fnd::BinF32	preDelayTime;
    /*  4B: fused mode */
    internal::fnd::BinU32	fusedMode;
    /*  4B: fused time */
    internal::fnd::BinF32	fusedTime;
    /*  4B: coloration */
    internal::fnd::BinF32	coloration;
    /*  4B: damping */
    internal::fnd::BinF32	damping;
    /*  4B: early gain */
    internal::fnd::BinF32	earlyGain;
    /*  4B: fused gain */
    internal::fnd::BinF32	fusedGain;
    /*  4B: out gain */
    internal::fnd::BinF32	outGain;
    /*  4B: reserved */
    unsigned char	reserved[4];
} ToolSoundReverbStdParameter;

//-----------------------------------------------------------------------
/* CHORUS パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B: エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: delay time */
    internal::fnd::BinF32	delayTime;
    /*  4B: depth */
    internal::fnd::BinF32	depth;
    /*  4B: rate */
    internal::fnd::BinF32	rate;
    /*  4B: feedback */
    internal::fnd::BinF32	feedback;
    /*  4B: out gain */
    internal::fnd::BinF32	outGain;
    /* 24B: reserved */
    unsigned char	reserved[24];
} ToolSoundChorusParameter;

//-----------------------------------------------------------------------
/* DELAY パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B: エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: max delay time */
    internal::fnd::BinF32	maxDelayTime;
    /*  4B: delay time */
    internal::fnd::BinF32	delayTime;
    /*  4B: feedback */
    internal::fnd::BinF32	feedback;
    /*  4B: out gain */
    internal::fnd::BinF32	outGain;
    /*  4B: LPF */
    internal::fnd::BinF32	lpf;
    /* 24B: reserved */
    unsigned char	reserved[24];
} ToolSoundDelayParameter;

#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
//-----------------------------------------------------------------------
/* MULTI CHANNEL REVERB パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B: エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: early mode */
    internal::fnd::BinU32	earlyMode;
    /*  4B: late mode */
    internal::fnd::BinU32	lateMode;
    /*  4B: pre delay time */
    internal::fnd::BinF32	preDelayTime;
    /*  4B: decay time */
    internal::fnd::BinF32	decayTime;
    /*  4B: hf decay ratio */
    internal::fnd::BinF32	hfDecayRatio;
    /*  4B: coloration */
    internal::fnd::BinF32	coloration;
    /*  4B: early gain */
    internal::fnd::BinF32	earlyGain;
    /*  4B: late gain */
    internal::fnd::BinF32	lateGain;
    /*  4B: reverb gain */
    internal::fnd::BinF32	reverbGain;
    /*  8B: reserved */
    unsigned char	reserved[8];
} ToolSoundMultiChannelReverbParameter;

//-----------------------------------------------------------------------
/* MULTI CHANNEL CHORUS パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B: エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: delay time */
    internal::fnd::BinF32	delayTime;
    /*  4B: depth */
    internal::fnd::BinF32	depth;
    /*  4B: rate */
    internal::fnd::BinF32	rate;
    /*  4B: feedback */
    internal::fnd::BinF32	feedback;
    /*  4B: out gain */
    internal::fnd::BinF32	outGain;
    /* 24B: reserved */
    unsigned char	reserved[24];
} ToolSoundMultiChannelChorusParameter;

//-----------------------------------------------------------------------
/* MULTI CHANNEL DELAY パラメータ */
typedef struct {
    /*  4B: バスの種類 */
    internal::fnd::BinU32	busType;
    /*  4B: インデックス */
    internal::fnd::BinU32	effectIndex;
    /*  4B: エフェクトの種類 */
    internal::fnd::BinU32	effectType;
    /*  4B: delay */
    internal::fnd::BinF32	delay;
    /*  4B: feedback */
    internal::fnd::BinF32	feedback;
    /*  4B: out gain */
    internal::fnd::BinF32	outGain;
    /*  4B: channel spread */
    internal::fnd::BinF32	channelSpread;
    /*  4B: lpf */
    internal::fnd::BinF32	lpf;
    /* 24B: reserved */
    unsigned char	reserved[24];
} ToolSoundMultiChannelDelayParameter;
#endif

/**/
class Effect //: public Newable
{
public:
    Effect(void);
    ~Effect(void);

    // エフェクトの設定
    //Result SetParameter(const ToolSoundEffectCommonParameter* pParam);
    void SetParameter(const ToolSoundEffectCommonParameter* pParam);

private:
    // エフェクトパラメータの適用
    void OnApplyNoEffect(ToolSoundNoEffectParameter* pParam);
    void OnApplyReverbHi(ToolSoundReverbHiParameter* pParam);
    void OnApplyReverbStd(ToolSoundReverbStdParameter* pParam);
    void OnApplyChorus(ToolSoundChorusParameter* pParam);
    void OnApplyDelay(ToolSoundDelayParameter* pParam);
#if defined(NW_SND_CONFIG_ENABLE_MULTICHEFT)
    void OnApplyMultiChannelReverb(ToolSoundMultiChannelReverbParameter* pParam);
    void OnApplyMultiChannelChorus(ToolSoundMultiChannelChorusParameter* pParam);
    void OnApplyMultiChannelDelay(ToolSoundMultiChannelDelayParameter* pParam);
#endif
    nw::snd::FxReverbFusedMode GetFxReverbFusedMode(ToolSoundFusedMode mode);
    nw::snd::FxReverbEarlyReflectionMode GetFxReverbEarlyReflectionMode(long mode);

private:
    // エフェクトの属性
    //AuxBus       m_nAuxBus; // 補助バス
    unsigned int m_nIndex;  // インデックス
};


} // namespace snd
} // namespace nw

#endif // NW4R_VIEWER_SOUND_EFFECT_H_
