﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/ut.h>
#include <nw/snd.h>
#include <nw/snd/fnd/model/sndfnd_WaveFormat.h>
#include <io/IStreamWriter.h>
#include <core/AlignedArray.h>

namespace nw {
namespace snd {

namespace internal {
class Voice;
}

class WaveRenderer : public IStreamWriter
{
private:
    static const u32 MAX_VOICES = 6;

    static const u32 WAVE_BUFFER_COUNT      = 8;					//!< 再生バッファ数です。
    static const u32 WAVE_BUFFER_DATA_SIZE  = sizeof(u32) * 1024;	//!< 再生バッファのデータサイズです。(4096 byte)
    static const u32 WAVE_BUFFER_SPARE_SIZE = 256;					//!< 再生バッファのデータが小さすぎる場合に前バッファに連結するための予備領域です。
    static const u32 WAVE_BUFFER_SIZE       =
        WAVE_BUFFER_DATA_SIZE + WAVE_BUFFER_SPARE_SIZE;				//!< 再生バッファ１つ分のサイズです。(4096 byte)

    static const u32 PREPARE_BUFFER_COUNT = 2;						//!< 再生開始に必要なバッファ数です。

    static const u32 INVALID_INDEX = static_cast<u32>(-1);

    typedef void (*COPY_SAMPLE_HANDLER)(void*, const void*);

public:
    /* ctor */ WaveRenderer();
    /* dtor */ virtual ~WaveRenderer();

public:
    bool Open(const internal::fnd::WaveFormat& waveFormat, u32 dataSize);
    /* override */ void Close();

    /* override */ u32 Write(const void* pBuffer, u32 size);
    /* override */ void EndOfStream();

    bool IsOpened() const;
    bool IsFinished() const;

protected:
    virtual void SetOutMix(
        internal::Voice& voice, const internal::fnd::WaveFormat& waveFormat, u32 channelIndex) { }

private:
    void WriteFrame(const void* pBuffer);
    void AppendWaveBuffer();
    u32 GetFreeWaveBufferIndex();
    u32 GetFreeWaveBufferCount();
    void StartPreparedVoices(bool isImmediate);
    COPY_SAMPLE_HANDLER GetCopySampleHandler() const;

private:
    bool m_IsOpened;

    internal::Voice m_Voices[MAX_VOICES];
    u32 m_VoiceCount;
    u32 m_BytesPerSample;
    u32 m_DataSize;

    WaveBuffer m_WaveBufferContainers[MAX_VOICES][WAVE_BUFFER_COUNT];
    AlignedArray<u8, WAVE_BUFFER_SIZE, 128> m_WaveBuffer[MAX_VOICES][WAVE_BUFFER_COUNT];

    u32 m_CurrentWaveBufferIndex;
    u32 m_CurrentWaveBufferValidSize;
    u32 m_TotalWrittenSize;

    nw::ut::CriticalSection m_CloseLock;
};

} // snd
} // nw
