﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <controls/F32ParameterListItem.h>

namespace nw {
namespace snd {

template<class TModel>
F32ParameterListItem<TModel>::F32ParameterListItem() :
m_DefaultValue(0.f),
m_ValueChange(0.01f),
m_ValueLargeChange(0.1f),
m_MinimumValue(0.f),
m_MaximumValue(1.f),
m_pGetValueHandler(NULL),
m_pSetValueHandler(NULL),
m_pPrintValueHandler(NULL)
{
}

template<class TModel>
void F32ParameterListItem<TModel>::Initialize(
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler)
{
    Initialize(NULL, NULL, pGetValueHandler, pSetValueHandler, pPrintValueHandler);
}

template<class TModel>
void F32ParameterListItem<TModel>::Initialize(
    GetIsCheckedHandler pGetIsCheckedHandler,
    SetIsCheckedHandler pSetIsCheckedHandler,
    GetValueHandler pGetValueHandler,
    SetValueHandler pSetValueHandler,
    PrintValueHandler pPrintValueHandler)
{
    NW_ASSERTMSG(m_pGetValueHandler == NULL, "F32ParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pSetValueHandler == NULL, "F32ParameterListItem is already initialized.");
    NW_ASSERTMSG(m_pPrintValueHandler == NULL, "F32ParameterListItem is already initialized.");
    NW_ASSERT(pGetValueHandler != NULL);
    NW_ASSERT(pSetValueHandler != NULL);
    NW_ASSERT(pPrintValueHandler != NULL);

    m_pGetValueHandler = pGetValueHandler;
    m_pSetValueHandler = pSetValueHandler;
    m_pPrintValueHandler = pPrintValueHandler;

    Base::Initialize(pGetIsCheckedHandler, pSetIsCheckedHandler);
}

template<class TModel>
f32 F32ParameterListItem<TModel>::GetValue() const
{
    NW_ASSERTMSG(m_pGetValueHandler != NULL, "F32ParameterListItem is not initialized.");
    return (*m_pGetValueHandler)(GetModel());
}

template<class TModel>
void F32ParameterListItem<TModel>::SetValue(f32 value)
{
    NW_ASSERTMSG(m_pSetValueHandler != NULL, "F32ParameterListItem is not initialized.");
    (*m_pSetValueHandler)(GetModel(), value);
}

template<class TModel>
void F32ParameterListItem<TModel>::ResetValue()
{
    SetValue(m_DefaultValue);
}

template<class TModel>
f32 F32ParameterListItem<TModel>::GetDefaultValue() const
{
    return m_DefaultValue;
}

template<class TModel>
void F32ParameterListItem<TModel>::SetDefaultValue(f32 value)
{
    m_DefaultValue = value;
}

template<class TModel>
f32 F32ParameterListItem<TModel>::GetValueChange() const
{
    return m_ValueChange;
}

template<class TModel>
void F32ParameterListItem<TModel>::SetValueChange(f32 value)
{
    m_ValueChange = value;
}

template<class TModel>
f32 F32ParameterListItem<TModel>::GetValueLargeChange() const
{
    return m_ValueLargeChange;
}

template<class TModel>
void F32ParameterListItem<TModel>::SetValueLargeChange(f32 value)
{
    m_ValueLargeChange = value;
}

template<class TModel>
f32 F32ParameterListItem<TModel>::GetMinimumValue() const
{
    return m_MinimumValue;
}

template<class TModel>
void F32ParameterListItem<TModel>::SetMinimumValue(f32 value)
{
    m_MinimumValue = value;
}

template<class TModel>
f32 F32ParameterListItem<TModel>::GetMaximumValue() const
{
    return m_MaximumValue;
}

template<class TModel>
void F32ParameterListItem<TModel>::SetMaximumValue(f32 value)
{
    m_MaximumValue = value;
}

template<class TModel>
void F32ParameterListItem<TModel>::NextValue(bool isLargeChange)
{
    f32 newValue = GetValue() + (isLargeChange ? m_ValueLargeChange : m_ValueChange);
    if ( newValue > m_MaximumValue - ( 0.5f * m_ValueChange ) )
    {
        newValue = m_MaximumValue;
    }

    SetValue(newValue);
}

template<class TModel>
void F32ParameterListItem<TModel>::PreviousValue(bool isLargeChange)
{
    f32 newValue = GetValue() - (isLargeChange ? m_ValueLargeChange : m_ValueChange);
    if ( newValue < m_MinimumValue + ( 0.5f * m_ValueChange ) )
    {
        newValue = m_MinimumValue;
    }

    SetValue(newValue);
}

template<class TModel>
void F32ParameterListItem<TModel>::PrintValue(char* pText, u32 bufferLength)
{
    NW_ASSERTMSG(m_pPrintValueHandler != NULL, "F32ParameterListItem is not initialized.");
    (*m_pPrintValueHandler)(pText, bufferLength, GetValue());
}

} // snd
} // nw
