﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef VWR_CAMERA_H_
#define VWR_CAMERA_H_

#include <nw/types.h>
#include <nw/math.h>
#include <nw/dev/dev_Pad.h>
#include <nw/dev/dev_Mouse.h>
#include <nw/ut.h>

#include <nw/config/undef_windows.h> // near, far のマクロ展開抑制



//! @brief 入力状態を表す構造体です。
struct InputStatus
{
    nw::dev::Pad*   pad;                //!< パッド管理クラスのインスタンスへのポインタです。
    nw::dev::Mouse* mouse;              //!< マウス管理クラスのインスタンスへのポインタです。 NULL を指定すると利用しません。
    bool            isMouseAvailable;   //!< マウスが有効かを指定します。
    bool            isAltModified;      //!< Altキーによる修飾がされているかを指定します。
    bool            isCtrlModified;     //!< Ctrlキーによる修飾がされているかを指定します。
    bool            isShiftModified;    //!< Shiftキーによる修飾がされているかを指定します。

    //! @brief コンストラクタです。
    InputStatus()
     : pad ( NULL ),
        mouse( NULL ),
        isMouseAvailable( false ),
        isAltModified( false ),
        isCtrlModified( false ),
        isShiftModified( false )
    {}
};

//---------------------------------------------------------------------------
//! @brief        ビューアカメラ
//---------------------------------------------------------------------------
class ViewerCamera
{
//---------------------------------------------------------
public:

    //---------------------------------------------------------------------------
    //! @brief        移動用のイベントです。
    //---------------------------------------------------------------------------
    class Event
    {
    public:
        typedef enum
        {
            ROTATE_BASED_ON_TARGET   = 0x00000001, //!< ターゲットを基準に回転をおこなう
            ROTATE_VERTICAL_360      = 0x00000002, //!< 縦にも360度回転する
            EXPAND_BASED_ON_TARGET   = 0x00000004, //!< ターゲットを基準に伸縮をおこなう
            POSITION_RESET           = 0x00000008, //!< 位置のリセット
            MOVE_ALONG_AXIS          = 0x00000010, //!< 座標軸に沿って並行移動する             (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
            MOVE_ALONG_VIEW          = 0x00000020, //!< XYZ共に視線ベクトルに沿って並行移動する(デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
            REVERSE_ON_VIEW_CAMERA   = 0x00000040, //!< ビューカメラの操作の場合に反転する
        } FlagType;


        //---------------------------------------------------------------------------
        //! @brief        コンストラクタです。
        //---------------------------------------------------------------------------
        /* ctor */ Event()
          : m_Flags( 0 )
          , m_Move( 0.0f, 0.0f, 0.0f )
          , m_RotateH( 0.0f )
          , m_RotateV( 0.0f )
          , m_Zoom( 0.0f )
          , m_Twist( 0.0f )
        {
        }

        //---------------------------------------------------------------------------
        //! @brief        設定をクリアします。
        //---------------------------------------------------------------------------
        void  Clear() { *this = Event(); }

        //---------------------------------------------------------------------------
        //! @brief        ズーム値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  Zoom() { return m_Zoom; }
        f32   Zoom() const { return m_Zoom; }

        //---------------------------------------------------------------------------
        //! @brief        水平方向の回転値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  RotateH() { return m_RotateH; }
        f32   RotateH() const { return m_RotateH; }

        //---------------------------------------------------------------------------
        //! @brief        垂直方向の回転値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  RotateV()       { return m_RotateV; }
        f32   RotateV() const { return m_RotateV; }

        //---------------------------------------------------------------------------
        //! @brief        ビュー方向を軸にした回転値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  Twist() { return m_Twist; }
        f32   Twist() const { return m_Twist; }

        //---------------------------------------------------------------------------
        //! @brief        平行移動値を設定/取得します。
        //---------------------------------------------------------------------------
        const nw::math::Vector3& Move() const { return m_Move; }
        nw::math::Vector3& Move() { return m_Move; }

        //---------------------------------------------------------------------------
        //! @brief        移動イベントのオプションフラグを設定/取得します。
        //---------------------------------------------------------------------------
        u32&   Flags() { return m_Flags; }
        u32    Flags() const { return m_Flags; }

        //---------------------------------------------------------------------------
        //! @brief        前方への移動値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  MoveForward() { return m_Move.z; }
        f32   MoveForward() const { return m_Move.z; }

        //---------------------------------------------------------------------------
        //! @brief        アップベクトル方向への移動値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  MoveUp() { return m_Move.y; }
        f32   MoveUp() const { return m_Move.y; }

        //---------------------------------------------------------------------------
        //! @brief        ライトベクトル方向への移動値を設定/取得します。
        //---------------------------------------------------------------------------
        f32&  MoveRight() { return m_Move.x; }
        f32   MoveRight() const { return m_Move.x; }


    private:
        u32  m_Flags;

        //-----------------
        // 回転オプション
        //   ROTATE_BASED_ON_TARGET     ：ターゲットを基準に回転 (デフォルトは首振り回転)
        //   ROTATE_VERTICAL_360        ：縦方向にも360度回転    (デフォルトは真上/真下まで)
        //   REVERSE_ON_VIEW_CAMERA     ：ビューカメラ操作の場合に反転
        f32  m_RotateH;    // 垂直回転係数 標準値[-1.0f, 1.0f]
        f32  m_RotateV;    // 水平回転係数 標準値[-1.0f, 1.0f]

        //-----------------
        // 移動オプション
        //   MOVE_ALONG_AXIS            ：座標軸に沿って並行移動する              (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
        //   MOVE_ALONG_VIEW            ：XYZ共に視線ベクトルに沿って並行移動する (デフォルトはXZ平面上を視線ベクトルに沿って、Y方向は軸に沿って移動)
        //   REVERSE_ON_VIEW_CAMERA     ：ビューカメラ操作の場合に反転
        //     (※MOVE_ALONG_AXISとMOVE_ALONG_VIEWフラグは共存不可)

        nw::math::Vector3 m_Move;

        //-----------------
        // ズームオプション
        //   EXPAND_BASED_ON_TARGET     ：ターゲットを基準に伸縮 (デフォルトはターゲットが動く)
        f32  m_Zoom;  // 伸縮係数 標準値[-1.0f, 1.0f]

        //-----------------
        // ツイストオプション
        f32  m_Twist;           // ツイスト係数  標準値[-1.0f, 1.0f]
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */          ViewerCamera();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual /* dtor */  ~ViewerCamera() {}

    void               UpdateCamera( const InputStatus& input );

    //---------------------------------------------------------------------------
    //! @brief       他のLookAtObjectの設定をコピー
    //!
    //! @param[out]  dirObj コピー元の LookAtObjec です。
    //---------------------------------------------------------------------------
    void                SetDirection( ViewerCamera* dirObj );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトの座標を設定します。
    //---------------------------------------------------------------------------
    void                SetPos( f32 x, f32 y, f32 z );
    void                SetPos( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトの座標を取得します。
    //---------------------------------------------------------------------------
    const nw::math::VEC3*   GetPos()     const;
    void                GetPos( nw::math::VEC3* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       ターゲットの座標を設定します。
    //!
    //! @param[in]    x       設定する注視点のX座標
    //! @param[in]    y       設定する注視点のY座標
    //! @param[in]    z       設定する注視点のZ座標
    //---------------------------------------------------------------------------
    void                SetLookAtPos( f32 x, f32 y, f32 z );
    //---------------------------------------------------------------------------
    //! @param[in]    vec     設定する注視点の座標
    //---------------------------------------------------------------------------
    void                SetLookAtPos( const nw::math::VEC3& vec );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットの座標を取得します。
    //!
    //! @return       注視点座標の const ポインタ
    //---------------------------------------------------------------------------
    const nw::math::VEC3*   GetLookAtPos()     const;
    //---------------------------------------------------------------------------
    //! @param[in]    pos     注視点の座標を取得するためのポインタ
    //---------------------------------------------------------------------------
    void                GetLookAtPos( nw::math::VEC3* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       Up ベクトルの取得
    //!
    //! @return       Up ベクトルを示す const ポインタ
    //---------------------------------------------------------------------------
    const nw::math::VEC3*   GetUpVec() const;
    //---------------------------------------------------------------------------
    //! @param[in]    pos     Up ベクトルを取得するためのポインタ
    //---------------------------------------------------------------------------
    void                GetUpVec( nw::math::VEC3* pos ) const;

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトとターゲットの距離を取得します。
    //!
    //! @return       オブジェクト位置から注視点の距離を返します。
    //---------------------------------------------------------------------------
    f32                 GetDistance () const;
    //---------------------------------------------------------------------------
    //! @brief       オブジェクトとターゲットの距離を設定します。
    //!
    //! @details     注視点を固定し、オブジェクトの位置を移動します。
    //!
    //! @param[in]    dist    オブジェクト位置から注視点の距離を設定します。
    //---------------------------------------------------------------------------
    void                SetDistance ( f32 dist );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトをターゲットへ近づけます。
    //---------------------------------------------------------------------------
    void                AddDistanceByMovingObj( f32 length );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットをオブジェクトへ近づけます。
    //---------------------------------------------------------------------------
    void                AddDistanceByMovingTarget( f32 length );

    //---------------------------------------------------------------------------
    //! @brief       距離の範囲を指定します。
    //---------------------------------------------------------------------------
    void                SetDistanceRange( f32 min, f32 max ) { m_DistanceMin = min; m_DistanceMax = max; }
    //---------------------------------------------------------------------------
    //! @brief       距離の範囲の最大値を取得します。
    //---------------------------------------------------------------------------
    f32                 GetDistanceMax() const { return m_DistanceMax; }
    //---------------------------------------------------------------------------
    //! @brief       距離の範囲の最小値を取得します。
    //---------------------------------------------------------------------------
    f32                 GetDistanceMin() const { return m_DistanceMin; }

    //---------------------------------------------------------------------------
    //! @brief       垂直方向の角度を取得します。
    //---------------------------------------------------------------------------
    u32                 GetAngleV() const { RecalcIf_( RECALC_ANGLE ); return m_AngleV; }

    //---------------------------------------------------------------------------
    //! @brief       水平方向の角度を取得します。
    //---------------------------------------------------------------------------
    u32                 GetAngleH() const { RecalcIf_( RECALC_ANGLE ); return m_AngleH; }

    //---------------------------------------------------------------------------
    //! @brief       軸の回転角度を取得します。
    //---------------------------------------------------------------------------
    f32                 GetTwist() const { return m_Twist; }

    //---------------------------------------------------------------------------
    //! @brief       軸の回転角度を設定します。
    //---------------------------------------------------------------------------
    void                SetTwist( f32 twist ) { m_Twist = twist; }

    //---------------------------------------------------------------------------
    //! @brief       LookAt オブジェクトを平行移動します。
    //!
    //! @param[in]    x       X 方向の移動距離です。
    //! @param[in]    y       Y 方向の移動距離です。
    //! @param[in]    z       Z 方向の移動距離です。
    //---------------------------------------------------------------------------
    void                Move( f32 x, f32 y, f32 z );

    //---------------------------------------------------------------------------
    //! @brief       ビュー方向に移動します。
    //!
    //! @param[in]   right   ビューのライトベクトル方向への移動距離です。
    //! @param[in]   forward ビューの正面方向への移動距離です。
    //! @param[in]   up      ビューのアップベクトル方向への移動距離です。
    //! @param[in]   view    ビューカメラ位置を指定します。
    //---------------------------------------------------------------------------
    void                MoveView( f32 right, f32 forward, f32 up, const ViewerCamera& view );

    //---------------------------------------------------------------------------
    //! @brief       ビューベクトルを XZ 平面に投影したベクトルを基準に平行移動します。
    //!
    //! @param[in]   right   ビューを XZ 平面に投影したライトベクトル方向への移動距離です。
    //! @param[in]   forward ビューを XZ 平面に投影した正面方向への移動距離です。
    //! @param[in]   view    ビューカメラ位置を指定します。
    //---------------------------------------------------------------------------
    void                MoveViewHorizontal( f32 right, f32 forward, const ViewerCamera& view );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトを中心に回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Rotate( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットを中心に回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Revolve( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       オブジェクトを中心に360度回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Rotate360( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       ターゲットを中心に360度回転します。
    //!
    //! @param[in]    angleV  垂直回転角度
    //! @param[in]    angleH  水平回転角度
    //---------------------------------------------------------------------------
    void                Revolve360( s32 angleV, s32 angleH );

    //---------------------------------------------------------------------------
    //! @brief       位置を最後にプリセットした位置へ戻します。
    //---------------------------------------------------------------------------
    void                Reset();

    //---------------------------------------------------------------------------
    //! @brief       Reset 時に状態を戻せるように現在の設定を記憶します。
    //---------------------------------------------------------------------------
    void                Preset();

    //---------------------------------------------------------------------------
    //! @brief       Twist の状態のみを Preset します。
    //---------------------------------------------------------------------------
    void                PresetTwist();

    //---------------------------------------------------------------------------
    //! @brief       Bind されているオブジェクトに追従します。
    //---------------------------------------------------------------------------
    void                Follow( const ViewerCamera* boundObj );

    //---------------------------------------------------------------------------
    //! @brief       Bindされているオブジェクトとの相対位置を更新します。
    //---------------------------------------------------------------------------
    void                UpdateFollowingBase( const ViewerCamera* boundObj );

    //---------------------------------------------------------------------------
    //! @brief       現在の座標・アングルをプリント出力します。
    //---------------------------------------------------------------------------
    void                Dump() const;

    void MakeCameraEventClassic( Event* events, const InputStatus& input, f32 frameSpeed = 1.0f );

#if 0
    enum { TIME_TO_GOAL = 10 };

    //---------------------------------------------------------------------------
    //! @brief       ゴール位置を設定します。
    //---------------------------------------------------------------------------
    void                SetGoal( const nw::math::VEC3& pos, const nw::math::VEC3& target ) { m_PosGoal = pos; m_LookAtPosGoal = target; }

    //---------------------------------------------------------------------------
    //! @brief       ゴール位置への移動を開始します。
    //!
    //! @details     MoveToDestination を実行することで、TIME_TO_GOAL の回数で指定位置に移動します。
    //---------------------------------------------------------------------------
    void                StartToGoal()  { m_GoalFrame = TIME_TO_GOAL; }

    //---------------------------------------------------------------------------
    //! @brief       ゴール位置への移動を終了します。
    //---------------------------------------------------------------------------
    void                StopToGoal()   { m_GoalFrame = 0;            }
#endif
    //---------------------------------------------------------------------------
    //! @brief       設定された目的位置まで、TIME_TO_GOALのフレーム数だけ掛けてシームレスに移動させます。
    //---------------------------------------------------------------------------
    void                MoveToDestination();

    //---------------------------------------------------------------------------
    //! @brief       イベントクラスの状態を元に移動します。
    //---------------------------------------------------------------------------
    void                MoveByEvent( const Event& refEvent, const ViewerCamera* view = NULL );

#if 0
    //---------------------------------------------------------------------------
    //! @brief       バインド種別を示す列挙体です。
    //---------------------------------------------------------------------------
    typedef enum
    {
        BINDTYPE_INDEPENDENT,   //!< バインドに制限されず、独立して動作します。
        BINDTYPE_SAME,          //!< バインド先のオブジェクトと同じ位置に制約されます。
        BINDTYPE_OPPOSITE,      //!< 注視点を基準にして、バインド先のオブジェクトと逆の位置に制約されます。
        BINDTYPE_DEPEND,        //!< バインド先のオブジェクトを基準とした座標系で、常に同じ位置に制約されます。
        BINDTYPE_NUM            //!< @details :private
    } BindType;

    //---------------------------------------------------------------------------
    //! @brief       他の LookAtObject にバインドします。
    //!
    //! @param[in]   boundObj バインドする LookAtObject です。
    //! @param[in]   bindType バインド種別です。
    //---------------------------------------------------------------------------
    void                BindTo( const ViewerCamera* boundObj, BindType bindType );

    //---------------------------------------------------------------------------
    //! @brief       バインドを解除します。
    //---------------------------------------------------------------------------
    void                UnBind() { this->BindTo( NULL, BINDTYPE_INDEPENDENT ); }

    //---------------------------------------------------------------------------
    //! @brief       バインド種別を取得します。
    //---------------------------------------------------------------------------
    BindType            GetBindType() const { return m_BindType; }
#endif

    //---------------------------------------------------------------------------
    //! @brief       姿勢行列を取得します。
    //---------------------------------------------------------------------------
    const nw::math::Matrix34&  GetMatrix() const { this->GetTargetMtx_( &m_Matrix ); return m_Matrix; }

//---------------------------------------------------------
protected:
    mutable nw::math::VEC3  m_Pos;               //!< @details :private カメラ位置
    mutable nw::math::VEC3  m_LookAtPos;         //!< @details :private ターゲット位置

    mutable f32             m_Distance;          //!< @details :private 注視点からの距離
    mutable u32             m_AngleV;            //!< @details :private カメラの水平回転角度
    mutable u32             m_AngleH;            //!< @details :private カメラの垂直回転角度
    f32                     m_Twist;             //!< @details :private カメラのツイスト角度

    mutable nw::math::VEC3  m_LookVec;           //!< @details :private 視線方向ベクトル
    mutable nw::math::VEC3  m_UpVec;             //!< @details :private 視線上方向ベクトル

    f32                     m_DistanceMax;       //!< @details :private 距離最大値
    f32                     m_DistanceMin;       //!< @details :private 距離最小値

    nw::math::VEC3          m_PosReset;          //!< @details :private リセット時の位置
    nw::math::VEC3          m_LookAtPosReset;    //!< @details :private リセット時のターゲット位置
    f32                     m_TwistReset;        //!< @details :private リセット時のツイスト角度

 //   nw::math::VEC3          m_PosGoal;           //!< @details :private 移動目標パラメータ
 //   nw::math::VEC3          m_LookAtPosGoal;     //!< @details :private 移動目標パラメータ
 //   s32                     m_GoalFrame;         //!< @details :private 移動用フレーム管理

 //   BindType                m_BindType;          //!< @details :private バインドタイプを設定します
 //   nw::math::VEC3          m_RelativePos;       //!< @details :private バインド時の相対座標を設定します
 //   nw::math::VEC3          m_RelativeLookAtPos; //!< @details :private バインド時の相対ターゲット座標を設定します

    mutable nw::math::Matrix34  m_Matrix;        //!< @details :private

    enum
    {
        RECALC_NONE = 0,    //!< @details :private 再計算の必要なし
        RECALC_OBJECT,      //!< @details :private カメラ位置を再計算
        RECALC_TARGET,      //!< @details :private ターゲット位置を再計算
        RECALC_ANGLE        //!< @details :private 回転角度を再計算
    };

    mutable u32         m_RecalcFlg;         //!< @details :private 再計算フラグ

    void                RecalcIf_( u32 recalcFlg ) const; //!< @details :private 特定の条件の場合に再計算します。
    void                Recalc_()          const;         //!< @details :private 再計算します。
    void                RecalcAngle_()     const;         //!< @details :private オブジェクトと注視点の座標から角度と距離を計算します。
    void                RecalcPos_()       const;         //!< @details :private 注視点座標と角度からカメラの座標を計算します。
    void                RecalcLookAtPos_() const;         //!< @details :private オブジェクト座標と角度から注視点座標を計算します。
    void                ClampDistance_();                 //!< @details :private 位置と注視点の距離が設定されたMin/Maxの範囲から外れていないかをチェックし、範囲内にクランプします。

    //! @details :private
    void                GetTargetMtx_( nw::math::MTX34* mtx ) const;   // 注視点を中心とした座標系への変換行列を取得

    //! 垂直角度を設定
    //! @details :private
    void         SetAngleV_( u32 angleV ) const
    {
        m_AngleV = angleV;
    }

    //! 水平角度を設定
    //! @details :private
    void         SetAngleH_( u32 angleH )
    {
        m_AngleH = angleH;
    }
};



//---------------------------------------------------------------------------
//! @brief        射影クラスです。
//---------------------------------------------------------------------------
class ViewerProjection
{
public:
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ ViewerProjection()
     : m_Mode( MODE_ORTHO )
     , m_Near( 0.0f )
     , m_Far( 1.0f )
     // , m_Config( NULL )
    // , m_UpdateConfigEventHandler( this, &Projection::OnUpdateConfig )
    {
        m_Rect.SetEdge(0.0f, 1.0f, 1.0f, 0.0f);
        const f32 ASPECT = 16.f / 9.f;
        const f32 FOVY_DEG = 30.f;
        const f32 NEAR_CLIP = 0.1f;
        const f32 FAR_CLIP  = 10000.0f;

        SetPerspectiveDeg( ASPECT, FOVY_DEG, NEAR_CLIP, FAR_CLIP );
    }

    //---------------------------------------------------------------------------
    //! @brief        Near を設定します。
    //!
    //! @param[in]    near      設定する Near です。
    //---------------------------------------------------------------------------
    void       SetNear( f32 near ) { m_Near = near; }

    //---------------------------------------------------------------------------
    //! @brief        Near を取得します。
    //!
    //! @return       Near を返します。
    //---------------------------------------------------------------------------
    f32        GetNear() const { return m_Near; }

    //---------------------------------------------------------------------------
    //! @brief        Far を設定します。
    //!
    //! @param[in]    far       設定する Far です。
    //---------------------------------------------------------------------------
    void       SetFar( f32 far ) { m_Far = far; }

    //---------------------------------------------------------------------------
    //! @brief        Far を取得します。
    //!
    //! @return       Far を返します。
    //---------------------------------------------------------------------------
    f32        GetFar() const { return m_Far; }

    //---------------------------------------------------------------------------
    //! @brief        Near と Far を設定します。
    //!
    //! @param[in]    near      設定する Near です。
    //! @param[in]    far       設定する Far です。
    //---------------------------------------------------------------------------
    void       SetNearFar( f32 near, f32 far )
    {
        m_Near = near;
        m_Far  = far;
    }

    //---------------------------------------------------------------------------
    //! @brief        アスペクトを取得します。
    //!
    //! @return       アスペクトを返します。
    //---------------------------------------------------------------------------
    f32        GetAspect() const { return m_Aspect; }

    //---------------------------------------------------------------------------
    //! @brief        FOVY を Rad 角で取得します。
    //!
    //! @return       FOVY を返します。
    //---------------------------------------------------------------------------
    f32        GetFovyRad() const { return m_FovyRad; }

    //---------------------------------------------------------------------------
    //! @brief        射影を透視投影として設定します。
    //!
    //! @param[in]    aspect    アスペクトです。
    //! @param[in]    fovyRad   Rad 角の FOVY です。
    //---------------------------------------------------------------------------
    void       SetPerspective( f32 aspect, f32 fovyRad )
    {
        m_Mode    = MODE_PERSPECTIVE;
        m_Aspect  = aspect;
        m_FovyRad = fovyRad;
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を透視投影として設定します。
    //!
    //! @param[in]    aspect    アスペクトです。
    //! @param[in]    fovyRad   Rad 角の FOVY です。
    //! @param[in]    near      Near です。
    //! @param[in]    far       Far です。
    //---------------------------------------------------------------------------
    void       SetPerspective( f32 aspect, f32 fovyRad, f32 near, f32 far )
    {
        this->SetPerspective( aspect, fovyRad );
        this->SetNearFar( near, far );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を透視投影として設定します。
    //!
    //! @param[in]    aspect    アスペクトです。
    //! @param[in]    fovyDeg   Deg の FOVY です。
    //---------------------------------------------------------------------------
    void       SetPerspectiveDeg( f32 aspect, f32 fovyDeg )
    {
        m_Mode    = MODE_PERSPECTIVE;
        m_Aspect  = aspect;
        m_FovyRad = nw::math::DegToRad( fovyDeg );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を透視投影として設定します。
    //!
    //! @param[in]    aspect    アスペクトです。
    //! @param[in]    fovyDeg   Deg の FOVY です。
    //! @param[in]    near      Near です。
    //! @param[in]    far       Far です。
    //---------------------------------------------------------------------------
    void       SetPerspectiveDeg( f32 aspect, f32 fovyDeg, f32 near, f32 far )
    {
        this->SetPerspectiveDeg( aspect, fovyDeg );
        this->SetNearFar( near, far );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を正投影として設定します。
    //!
    //! @param[in]    left      左端です。
    //! @param[in]    right     右端です。
    //! @param[in]    bottom    下端です。
    //! @param[in]    top       上端です。
    //---------------------------------------------------------------------------
    void       SetOrtho( f32 left, f32 right, f32 bottom, f32 top )
    {
        m_Mode = MODE_ORTHO;
        m_Rect.SetEdge( left, right, bottom, top );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を正投影として設定します。
    //!
    //! @param[in]    left      左端です。
    //! @param[in]    right     右端です。
    //! @param[in]    bottom    下端です。
    //! @param[in]    top       上端です。
    //! @param[in]    near      Near です。
    //! @param[in]    far       Far です。
    //---------------------------------------------------------------------------
    void       SetOrtho( f32 left, f32 right, f32 bottom, f32 top, f32 near, f32 far )
    {
        this->SetOrtho( left, right, bottom, top );
        this->SetNearFar( near, far );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を Frustum で設定します。
    //!
    //! @param[in]    left      左端です。
    //! @param[in]    right     右端です。
    //! @param[in]    bottom    下端です。
    //! @param[in]    top       上端です。
    //---------------------------------------------------------------------------
    void       SetFrustum( f32 left, f32 right, f32 bottom, f32 top )
    {
        m_Mode = MODE_FRUSTUM;
        m_Rect.SetEdge( left, right, bottom, top );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影を Frustum で設定します。
    //!
    //! @param[in]    left      左端です。
    //! @param[in]    right     右端です。
    //! @param[in]    bottom    下端です。
    //! @param[in]    top       上端です。
    //! @param[in]    near      Near です。
    //! @param[in]    far       Far です。
    //---------------------------------------------------------------------------
    void       SetFrustum( f32 left, f32 right, f32 bottom, f32 top, f32 near, f32 far )
    {
        this->SetFrustum( left, right, bottom, top );
        this->SetNearFar( near, far );
    }

    //---------------------------------------------------------------------------
    //! @brief        射影行列を取得します。
    //!
    //! @return       射影行列を返します。
    //---------------------------------------------------------------------------
    const nw::math::Matrix44& GetMatrix()
    {
        switch ( m_Mode )
        {
        case MODE_ORTHO:
            m_Matrix.SetOrtho( m_Rect.left, m_Rect.right, m_Rect.bottom, m_Rect.top, m_Near, m_Far );
            break;
        case MODE_FRUSTUM:
            m_Matrix.SetFrustum( m_Rect.left, m_Rect.right, m_Rect.bottom, m_Rect.top, m_Near, m_Far );
            break;
        case MODE_PERSPECTIVE:
            m_Matrix.SetPerspective( m_FovyRad, m_Aspect, m_Near, m_Far );
            break;
        default:
            NW_FATAL_ERROR("Invalid mode");
        }

        return m_Matrix;
    }

    //---------------------------------------------------------------------------
    //! @brief        コンフィグファイルからデータを設定します。
    //!
    //! @param[in]    config    コンフィグ管理クラスのインスタンスへのポインタです。
    //---------------------------------------------------------------------------
#if 0
    void UpdateConfig( nw::vwr3d::util::ViewerConfig* config )
    {
        NW_ASSERT_NOT_NULL( config );
        m_Config = config;

        {
            // Near
            nw::vwr3d::util::ViewerConfig::MenuArgNumeric<f32> arg;
            arg.valueMin = 0.f;
            arg.valueMax = 1000.f;
            arg.valueChange = 0.1f;
            arg.valueLargeChange = 1.f;

            config->UpdateValueFloat( "CameraNear", &m_Near, m_Near );
            config->AddEditMenuFloat( "CameraNear", &m_Near, arg, &m_UpdateConfigEventHandler );
        }

        {
            // Far
            nw::vwr3d::util::ViewerConfig::MenuArgNumeric<f32> arg;
            arg.valueMin = 0.f;
            arg.valueMax = 100000.f;
            arg.valueChange = 100.f;
            arg.valueLargeChange = 1000.f;

            config->UpdateValueFloat( "CameraFar", &m_Far, m_Far );
            config->AddEditMenuFloat( "CameraFar", &m_Far, arg, &m_UpdateConfigEventHandler );
        }

        {
            // Fovy
            nw::vwr3d::util::ViewerConfig::MenuArgNumeric<f32> arg;
            arg.valueMin = 0.f;
            arg.valueMax = nw::math::DegToRad( 180.f );
            arg.valueChange = 0.01f;
            arg.valueLargeChange = 1.f;

            config->UpdateValueFloat( "CameraFovyRadian", &m_FovyRad, m_FovyRad );
            config->AddEditMenuFloat( "CameraFovyRadian", &m_FovyRad, arg, &m_UpdateConfigEventHandler );
        }

        {
            // AspectRatio
            nw::vwr3d::util::ViewerConfig::MenuArgNumeric<f32> arg;
            arg.valueMin = 0.f;
            arg.valueMax = 10000.f;
            arg.valueChange = 0.01f;
            arg.valueLargeChange = 1.f;

            config->UpdateValueFloat( "CameraAspectRatio", &m_Aspect, m_Aspect );
            config->AddEditMenuFloat( "CameraAspectRatio", &m_Aspect, arg, &m_UpdateConfigEventHandler );
        }
    }
#endif

private:
    //! @brief コンフィグが更新されたときに呼ばれます。
    void OnUpdateConfig() {}

    enum { MODE_ORTHO, MODE_PERSPECTIVE, MODE_FRUSTUM };

    u32 m_Mode;         //!< @details :private

    f32 m_Near;         //!< @details :private
    f32 m_Far;          //!< @details :private

    f32 m_Aspect;       //!< @details :private
    f32 m_FovyRad;      //!< @details :private
    nw::ut::Rect m_Rect;    //!< @details :private

    nw::math::Matrix44  m_Matrix;   //!< @details :private

//    nw::vwr3d::util::ViewerConfig* m_Config; //!< コンフィグ管理です。
//    nw::ut::Delegate0<Projection, void> m_UpdateConfigEventHandler;
};



#endif //  VWR_CAMERA_H_
